/******************************************************************************
Description:

  test_udp.c
  
  To run this program:
  
      Unicast: test_udp -nx -n 1

      Multicast: test_udp -nx -nodes <nodelist>
  
CVS Information:

  $Source: /home/globdev/CVS/globus-current/Globus/Communication/nexus/tests/miscellaneous/test_udp.c,v $
  $Date: 1998/11/03 21:45:15 $
  $Revision: 1.13 $
  $State: Exp $
  $Author: toonen $
******************************************************************************/

/******************************************************************************
			     Include header files
******************************************************************************/
#include <stdio.h>
#include "nexus.h"


/******************************************************************************
		       Define module specific constants
******************************************************************************/

#undef DEBUG_LEVEL 2

/*
 * MULTICAST - if defined, then a multicast test will be performed
 *
 * MULTICAST_ADDR - any address in the range of 224.0.0.0 through
 *     239.255.255.255 may be used providing that address is not already
 *     reserved (see the "INTERNET MULTICAST ADDRESSES" section in
 *     http://info.internet.isi.edu:80/in-notes/std/files/std2.txt for a list
 *     of reserved addresses.
 *
 * MULTICAST_PORT - if defined, the specified port will be used.  If not
 *     defined, a port based on the time of day will be selected.
 *
 * MULTICAST_TTL - number of hops before the packet will be dropped by
 *     router.  This is useful for limiting who sees the packets sent.
 *     If this is less than one, pr_udp will fill in the system default
 *     defined by IP_DEFAULT_MULTICAST_TTL in netinet/in.h.
 */
#undef MULTICAST
#define MULTICAST_ADDR "224.0.1.20"
#define MULTICAST_PORT 4000
#define MULTICAST_TTL  5

/*
 * FT_ENABLE - enable testing of fault tolerance
 */
#undef FT_ENABLE

#define NUM_MESSAGES	50
#define USLEEP_INTERVAL	10000
#define MESSAGE_SIZE	(128)


/******************************************************************************
			 Define module specific macros
******************************************************************************/
#if defined(DEBUG_LEVEL)
#   define debug_printf(level, message)		\
    {						\
        if ((level) <= (DEBUG_LEVEL))		\
        {					\
            nexus_printf message;		\
        }					\
    }
#else
#   define debug_printf(level, message)
#endif


/******************************************************************************
			       Type definitions
******************************************************************************/
typedef struct
{
    globus_mutex_t				mutex;
    globus_cond_t				cond;
    volatile int				flag;
    globus_nexus_startpoint_t *			sp;
}
Test_UDP_L_Startpoint_Monitor_t;    
    

typedef struct
{
    volatile int				count;
    globus_mutex_t				mutex;
    globus_cond_t				cond;
}
Test_UDP_L_Shutdown_Barrier_t;


/******************************************************************************
			   Module specific variables
******************************************************************************/
static nexus_endpoint_t			GlobalEndpoint;


/******************************************************************************
			  Module specific prototypes
******************************************************************************/
static
void
Test_UDP_l_run_Test(
    nexus_node_t *				nodes,
    int						n_nodes);

static
void
Test_UDP_l_get_Startpoint(
    int						port,
    nexus_startpoint_t *			dest_sp,
    nexus_startpoint_t *			udp_sp);

static
void
Test_UDP_l_handle_Get_Startpoint(
    nexus_endpoint_t *				endpoint, 
    nexus_buffer_t *				recv_buf,
    nexus_bool_t				 is_non_threaded_handler);

static
void
Test_UDP_l_handle_Get_Startpoint_Reply(
    nexus_endpoint_t *				endpoint, 
    nexus_buffer_t *				recv_buf,
    nexus_bool_t				is_non_threaded_handler);

static
void
Test_UDP_l_handle_Message(
    nexus_endpoint_t *				endpoint, 
    nexus_buffer_t *				buffer,
    nexus_bool_t				is_non_threaded_handler);

static
void
Test_UDP_l_shutdown_Nodes(
    nexus_node_t *				nodes,
    int						n_nodes);

static
void
Test_UDP_l_handle_Unknown(
    nexus_endpoint_t *				endpoint,
    nexus_buffer_t *				buffer,
    int						handler_id);

static
void
Test_UDP_l_handle_Shutdown_Nodes(
    nexus_endpoint_t *				endpoint,
    nexus_buffer_t *				buffer,
    nexus_bool_t				is_non_threaded_handler);

static
void
Test_UDP_l_handle_Shutdown_Nodes_Reply(
    nexus_endpoint_t *				endpoint,
    nexus_buffer_t *				buffer,
    nexus_bool_t				is_non_threaded_handler);

static
int
Test_UDP_l_handle_Fault(
    void *					user_arg,
    int						fault_code);


/******************************************************************************
			  Initialized handler tables
******************************************************************************/
static nexus_handler_t tcp_handlers[] =
{
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (nexus_handler_func_t) Test_UDP_l_handle_Get_Starpoint},
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (nexus_handler_func_t) Test_UDP_l_handle_Get_Starpoint_Reply},
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (nexus_handler_func_t) Test_UDP_l_handle_Shutdown_Nodes},
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (nexus_handler_func_t) Test_UDP_l_handle_Shutdown_Nodes_Reply}
};

#define UDP_STARTPOINT_GET 0
#define UDP_STARTPOINT_GET_REPLY 1
#define SHUTDOWN_NODE_HANDLER 2
#define SHUTDOWN_NODE_REPLY_HANDLER 3

static nexus_handler_t udp_handlers[] =
{
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (nexus_handler_func_t) Test_UDP_l_handle_Message}
};

#define UDP_RCV_HANDLER_ID 0


int
NexusBoot(nexus_startpoint_t * startpoint)
{
    nexus_endpointattr_t		tcp_ep_attr;

    /* initializing an endpointattr */
    if (nexus_endpointattr_init(&tcp_ep_attr))
    {
	goto abort; 
    }
    
    if (nexus_endpointattr_set_unknown_handler(
	&tcp_ep_attr,
	NexusUnknownHandler,
	NEXUS_HANDLER_TYPE_NON_THREADED) != 0)
    {
	goto abort;
    }

    if (nexus_endpointattr_set_handler_table(
	&tcp_ep_attr, 
	tcp_handlers, 
	(sizeof(tcp_handlers) / sizeof(nexus_handler_t))) != 0)
    {
	goto abort;
    }

    /* initializing endpoint */
    if (nexus_endpoint_init(&GlobalEndpoint, &tcp_ep_attr) != 0)
    {
	goto abort;
    }

    /* binding startpoint to endpoint */
    if (nexus_startpoint_bind(startpoint, &GlobalEndpoint) != 0)
    {
	goto abort;
    }

    if (nexus_endpointattr_destroy(&tcp_ep_attr) != 0)
    {
	goto abort;
    }

    return NEXUS_SUCCESS;

  abort:
    return NEXUS_FAILURE;
}
/* NexusBoot() */


/******************************************************************************
Function:	main()

Description:	main test program

Parameters:	argc		number of command line arguments
		argv		pointers to the actual arguments

Returns:	exit value (always zero)
******************************************************************************/
int
main(int argc,
     char **argv)
{
    int					i;

    /*
     * Nexus must be activated before it can be used
     */
    globus_module_activate(GLOBUS_NEXUS_MODULE);


#   if defined(FT_ENABLE)
    {
	nexus_enable_fault_tolerance(Test_UDP_l_handle_Fault, NULL);
    }
#   endif

    nexus_startpoint_bind(startpoint, &GlobalEndpoint);

    /*
     * See if we have the correct number of nodes
     */
    if (n_nodes < 2)
    {
	nexus_printf("ERROR: Must run with at least 2 nodes.\n",n_nodes);
	shutdown_nodes(nodes, n_nodes);
	nexus_abort();
    } /* endif */

    /* Perform tests */
    nexus_printf("Starting UDP test\n");
    udp_test(nodes, n_nodes);

    /* Shutdown remote nodes */
    nexus_printf("Shutting down Remote Nodes\n");
    shutdown_nodes(nodes, n_nodes);

    /* Cleanup */
    nexus_printf("Cleaning Up Endpoints\n");
    nexus_endpoint_destroy(&GlobalEndpoint);

    /* Free the node list */
    nexus_printf("Freeing the Node List\n");
    for(i = 0; i < n_nodes; i++)
    {
	nexus_free(nodes[i].name);
	/* OK to call startpoint destroy on nodes[0].startpoint */
	/* even though nodes[0].startpoint known to be NULL     */
	nexus_startpoint_destroy(&(nodes[i].startpoint));
    } /* endfor */
    nexus_free(nodes);
    
    /* Terminate master process */
    nexus_printf("Terminating master process\n");
    nexus_context_destroy(NEXUS_FALSE);

    nexus_printf("main(): ERROR: We should never get here.\n");

    return 0;

} /* end main() */


/******************************************************************************
Function:	udp_test()

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
udp_test(nexus_node_t * nodes,
	 int n_nodes)
{
    int	*				message;
    nexus_buffer_t			buffer;
    int					node;
    int					msg;
    nexus_startpoint_t *		udp_sp;

    NexusMalloc(udp_test(),
		udp_sp,
		nexus_startpoint_t *,
		sizeof(nexus_startpoint_t) * n_nodes);

    NexusMalloc(udp_test(),
		message,
		int *,
		MESSAGE_SIZE * sizeof(int));

    for(node = 1; node < n_nodes; node++)
    {
	udp_startpoint_get(MULTICAST_PORT,
			   &nodes[node].startpoint, 
			   &udp_sp[node]);
    }

    /* Initialize the message buffer */
    /* Send messages */
    for (msg = 0; msg < NUM_MESSAGES; msg++)
    {
	for(node = 1; node < n_nodes; node++)
	{
	    int				msgn;

#           if defined(MULTICAST)
	    {		    
		msgn = msg * (n_nodes - 1) + node - 1;
	    }
#           else
	    {
		msgn = msg;
	    }
#           endif

	    if (!nexus_startpoint_test(&udp_sp[node]))
	    {
		int				cnt;

		for (cnt = 0; cnt < MESSAGE_SIZE; cnt++)
		{
			message[cnt] = msgn + cnt;
		}

		nexus_buffer_init(&buffer, nexus_sizeof_int(MESSAGE_SIZE), 0);
		nexus_put_int(&buffer, message, MESSAGE_SIZE);

		debug_printf(2, ("BEFORE SENT: %d\n", msgn));

		if (nexus_send_rsr(&buffer, 
				   &udp_sp[node], 
				   UDP_RCV_HANDLER_ID, 
				   NEXUS_TRUE, 
				   NEXUS_FALSE) != 0)
		{
		    nexus_printf("udp_test(): send() to node %d failed\n",
				node);
		}

		debug_printf(2, ("AFTER SENT: %d\n", msgn));
		nexus_printf("udp_test(): message %d sent to node %d\n",
			     msgn, node);

		nexus_usleep(USLEEP_INTERVAL);
	    }
	    else
	    {
		nexus_printf("udp_test(): node %d reports an error. "
			     "skipping message %d\n", node, msgn);
	    }
	}
    }

    for(node = 1; node < n_nodes; node++)
    {
	if (nexus_startpoint_destroy(&udp_sp[node]))
	{
	    nexus_fatal("udp_tes(): startpoint[%d] destroy failed\n", node);
	}
    }

    NexusFree(message);
    NexusFree(udp_sp);
}


/******************************************************************************
Function:	udp_rcv_handler()

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
udp_rcv_handler(nexus_endpoint_t *endpoint, 
		nexus_buffer_t *buffer,
		nexus_bool_t is_non_threaded_handler)
{
    int					cnt;
    int *				message;
    nexus_bool_t			correct_message = NEXUS_TRUE;
    static int				msg_cnt = 0;

    debug_printf(1, ("udp_rcv_handler(): enter\n"));

    NexusMalloc(udp_test(),
		message,
		int *,
		MESSAGE_SIZE * sizeof(int));

    nexus_get_int(buffer, message, MESSAGE_SIZE);
 
    if (message[0] != msg_cnt)
    {
	correct_message = NEXUS_FALSE;
	nexus_printf("udp_rcv_handler(): INCORRECT message number, got=%d, "
		     "expected=%d\n", message[0], msg_cnt);
	msg_cnt = message[0];
    }

    for (cnt = 0; cnt < MESSAGE_SIZE && correct_message; cnt++)
    {
	if (message[cnt] != msg_cnt + cnt)
	{
	    correct_message = NEXUS_FALSE;
	    nexus_printf("udp_rcv_handler(): INCORRECT message body, "
			 "loc=%d, value=%d\n", cnt, message[cnt]);
	}
    }

    if (correct_message)
    {
	nexus_printf("udp_recv_handler(): message %d received\n", msg_cnt);
    }

    msg_cnt++;

    nexus_buffer_destroy(buffer);

    NexusFree(message);

    debug_printf(1, ("udp_rcv_handler(): exit\n"));
}
/* udp_rcv_handler() */


/******************************************************************************
Function:	udp_startpoint_get()

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
udp_startpoint_get(int port,
		   nexus_startpoint_t * dest_sp,
		   nexus_startpoint_t * udp_sp)
{
    nexus_startpoint_t			my_sp;
    udp_startpoint_get_monitor_t	monitor;
    nexus_buffer_t			send_buf;
    int					send_buf_sz;

    debug_printf(1, ("udp_startpoint_get(): enter\n"));

    /*
     * Initialize monitor structure
     */
    nexus_mutex_init(&monitor.mutex, (nexus_mutexattr_t *) NULL);
    nexus_cond_init(&monitor.cond, (nexus_condattr_t *) NULL);
    monitor.sp = udp_sp;
    monitor.flag = 0;
    nexus_endpoint_set_user_pointer(&GlobalEndpoint, (void *) &monitor);

    /*
     * Construct message consisting of the startpoint to which the UDP
     * startpoint should be sent, and then send the message
     */
    nexus_startpoint_bind(&my_sp, &GlobalEndpoint);
    send_buf_sz = nexus_sizeof_startpoint(&my_sp, 1) + nexus_sizeof_int(1);
    nexus_buffer_init(&send_buf, send_buf_sz, 0);
    nexus_put_int(&send_buf, &port, 1);
    nexus_put_startpoint_transfer(&send_buf, (&my_sp), 1);
    if (nexus_send_rsr(&send_buf,
		       dest_sp,
		       UDP_STARTPOINT_GET,
		       NEXUS_TRUE,
		       NEXUS_FALSE) != 0)
    {
	nexus_fatal("udp_startpoint_get(): send failed\n");
    }

    /*
     * Wait for the UDP startpoint to be received
     */
    nexus_mutex_lock(&monitor.mutex);
    while (!monitor.flag)
    {
	nexus_cond_wait(&monitor.cond, &monitor.mutex);
    }
    nexus_mutex_unlock(&monitor.mutex);
    
    nexus_mutex_destroy(&monitor.mutex);
    nexus_cond_destroy(&monitor.cond);

    debug_printf(1, ("udp_startpoint_get(): exit\n"));
}
/* udp_startpoint_get() */


/******************************************************************************
Function:	udp_startpoint_get_handler()

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
udp_startpoint_get_handler(nexus_endpoint_t *endpoint, 
			   nexus_buffer_t *recv_buf,
			   nexus_bool_t is_non_threaded_handler)
{
    nexus_endpointattr_t		udp_ep_attr;
    static nexus_endpoint_t		udp_ep;
    nexus_startpoint_t			udp_sp;
    nexus_startpoint_t			tcp_sp;
    nexus_buffer_t			send_buf;
    int					send_buf_sz;
    int					port;

    debug_printf(1, ("udp_startpoint_get_handler(): enter\n"));

    nexus_get_int(recv_buf, &port, 1);
    nexus_get_startpoint(recv_buf, &tcp_sp, 1);
    
    /* initializing an endpointattr */
    nexus_endpointattr_init(&udp_ep_attr);
    nexus_endpointattr_set_unknown_handler(&udp_ep_attr,
					  NexusUnknownHandler,
					  NEXUS_HANDLER_TYPE_NON_THREADED);
    nexus_endpointattr_set_handler_table(&udp_ep_attr,
					 udp_handlers, 
					 (sizeof(udp_handlers)
					  / sizeof(nexus_handler_t)));

    /* setting endpointattr to use the UDP protocol */
#   if defined(MULTICAST)
    {
	nexus_proto_info_udp_t proto_info;  

	proto_info.host = MULTICAST_ADDR;
	proto_info.port = port;
	proto_info.ttl = MULTICAST_TTL;

	if (nexus_endpointattr_set_protocol(&udp_ep_attr,
					    NEXUS_PROTO_TYPE_UDP,
					    &proto_info,
					    sizeof(nexus_proto_info_udp_t)
	                                    ) != 0)
	{
	    nexus_fatal("udp_startpoint_get_handler: set protocol failed\n");
	}
    }
#   else
    {
	if (nexus_endpointattr_set_protocol(&udp_ep_attr,
					    NEXUS_PROTO_TYPE_UDP,
					    NULL,
					    0) != 0)
	{
	    nexus_fatal("udp_startpoint_get_handler: set protocol failed\n");
	}
    }
#   endif

    /* initializing endpoint ... no particular address */
    if (nexus_endpoint_init(&udp_ep, &udp_ep_attr) != 0)
    {
	nexus_fatal("udp_startpoint_get_handler(): endpoint init failed\n");
    }

    /* binding startpoint to endpoint */
    if (nexus_startpoint_bind(&udp_sp, &udp_ep) != 0)
    {
	nexus_fatal("udp_startpoint_get_handler(): startpoint bind failed\n");
    }

    nexus_endpointattr_destroy(&udp_ep_attr);

    send_buf_sz = nexus_sizeof_startpoint(&udp_sp, 1);
    nexus_buffer_init(&send_buf, send_buf_sz, 0);
    nexus_put_startpoint_transfer(&send_buf, &udp_sp, 1);

    if (nexus_send_rsr(&send_buf,
		       &tcp_sp,
		       UDP_STARTPOINT_GET_REPLY,
		       NEXUS_TRUE,
		       NEXUS_TRUE) != 0)
    {
	nexus_fatal("udp_startpoint_get_handler(): send failed\n");
    }

    if (nexus_startpoint_destroy(&tcp_sp))
    {
	nexus_fatal("udp_startpoint_get_handler(): "
		    "startpoint destroy failed\n");
    }

    debug_printf(1, ("udp_startpoint_get_handler(): exit\n"));
}
/* udp_startpoint_get_handler() */


/******************************************************************************
Function:	udp_startpoint_get_reply_handler()

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
udp_startpoint_get_reply_handler(nexus_endpoint_t *endpoint, 
				 nexus_buffer_t *recv_buf,
				 nexus_bool_t is_non_threaded_handler)
{
    udp_startpoint_get_monitor_t *	monitor;

    debug_printf(1, ("udp_startpoint_get_reply_handler(): enter\n"));

    monitor = (udp_startpoint_get_monitor_t *)
	nexus_endpoint_get_user_pointer(endpoint);

    nexus_mutex_lock(&monitor->mutex);
    if (nexus_get_startpoint(recv_buf, monitor->sp, 1))
    {
	nexus_fatal("udp_startpoint_get_reply_handler(): "
		    "get startpoint failed\n");
    }
    monitor->flag = NEXUS_TRUE;
    nexus_cond_signal(&monitor->cond);
    nexus_mutex_unlock(&monitor->mutex);

    debug_printf(1, ("udp_startpoint_get_reply_handler(): exit\n"));
}
/* udp_startpoint_get_reply_handler() */


/******************************************************************************
Function:	shutdown_node_handler

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
shutdown_node_handler(nexus_endpoint_t *endpoint,
		      nexus_buffer_t *buffer,
		      nexus_bool_t called_from_non_threaded_handler)
{
    nexus_buffer_t reply_buffer;
    nexus_startpoint_t barrier_sp;

    debug_printf(1, ("shutdown_node_handler(): enter\n"));

    nexus_get_startpoint(buffer, &barrier_sp, 1);
    
    debug_printf(2,
		 ("shutdown_node_handler(): sending shutdown node reply\n"));
    nexus_buffer_init(&reply_buffer, 0, 0);
    if (nexus_send_rsr(&reply_buffer,
		       &barrier_sp,
		       SHUTDOWN_NODE_REPLY_HANDLER,
		       NEXUS_TRUE,
		       called_from_non_threaded_handler) != 0)
    {
	nexus_printf("shutdown_node_handler(): WARNING: send failed\n");
    }

    nexus_startpoint_destroy(&barrier_sp);
    
    debug_printf(2, ("shutdown_node_handler(): destroying context\n"));
    nexus_context_destroy(called_from_non_threaded_handler);

    debug_printf(1, ("shutdown_node_handler(): exit\n"));
} /* shutdown_node_handler() */


/******************************************************************************
Function:	shutdown_node_reply_handler

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
shutdown_node_reply_handler(nexus_endpoint_t *endpoint,
			    nexus_buffer_t *buffer,
			    nexus_bool_t called_from_non_threaded_handler)
{
    shutdown_nodes_barrier_t *shutdown_nodes_barrier;

    debug_printf(1, ("shutdown_node_reply_handler(): entering\n"));

    shutdown_nodes_barrier =
        (shutdown_nodes_barrier_t *)nexus_endpoint_get_user_pointer(endpoint);
    nexus_mutex_lock(&(shutdown_nodes_barrier->mutex));
    if (--shutdown_nodes_barrier->count == 0)
    {
	nexus_cond_signal(&(shutdown_nodes_barrier->cond));
    }
    nexus_mutex_unlock(&(shutdown_nodes_barrier->mutex));

    debug_printf(1, ("shutdown_node_reply_handler(): exiting\n"));
}
/* shutdown_node_reply_handler() */


/******************************************************************************
Function:	shutdown_nodes

Description:	

Parameters:	

Returns:	
******************************************************************************/
static void
shutdown_nodes(nexus_node_t *nodes,
	       int n_nodes)
{
    int i;
    nexus_buffer_t buffer;
    nexus_startpoint_t barrier_sp;
    shutdown_nodes_barrier_t shutdown_nodes_barrier;
    int buf_size;

    debug_printf(1, ("shutdown_nodes(): enter\n"));

    shutdown_nodes_barrier.count = 0;
    nexus_mutex_init(&(shutdown_nodes_barrier.mutex),
		     (nexus_mutexattr_t *) NULL);
    nexus_cond_init(&(shutdown_nodes_barrier.cond),
		    (nexus_condattr_t *) NULL);
    nexus_endpoint_set_user_pointer(&GlobalEndpoint,
				    (void *)&shutdown_nodes_barrier);
    
    for (i = 1; i < n_nodes; i++)
    {
	if (nodes[i].return_code == NEXUS_NODE_NEW &&
	    nexus_startpoint_test(&nodes[i].startpoint) == 0)
	{
	    nexus_startpoint_bind(&barrier_sp, &GlobalEndpoint);
	    buf_size = nexus_sizeof_startpoint(&barrier_sp, 1);
	    nexus_buffer_init(&buffer, buf_size, 0);
	    nexus_put_startpoint_transfer(&buffer, (&barrier_sp), 1);
	    debug_printf(2, ("shutdown_nodes(): sending destroy context "
			     "message to node %d\n", i));
	    if (nexus_send_rsr(&buffer,
			       &(nodes[i].startpoint),
			       SHUTDOWN_NODE_HANDLER,
			       NEXUS_TRUE,
			       NEXUS_TRUE) != 0)
	    {
		debug_printf(2, ("shutdown_nodes(): WARNING: send failed\n"));
	    }
	    nexus_mutex_lock(&(shutdown_nodes_barrier.mutex));
	    shutdown_nodes_barrier.count++;
	    nexus_mutex_unlock(&(shutdown_nodes_barrier.mutex));
	}
    }

    debug_printf(2, ("shutdown_nodes(): waiting for replies\n"));
    nexus_mutex_lock(&(shutdown_nodes_barrier.mutex));
    while (shutdown_nodes_barrier.count > 0)
    {
	nexus_cond_wait(&(shutdown_nodes_barrier.cond),
			&(shutdown_nodes_barrier.mutex) );
    }
    nexus_mutex_unlock(&(shutdown_nodes_barrier.mutex));
    
    nexus_mutex_destroy(&(shutdown_nodes_barrier.mutex));
    nexus_cond_destroy(&(shutdown_nodes_barrier.cond));
    
    debug_printf(1, ("shutdown_nodes(): enter\n"));
}
/* shutdown_nodes() */


/******************************************************************************
Function:	NexusUnknownHandler

Description:	catch any bad handler invocation requests

Parameters:	endpoint	
		buffer
		hanlder_id

Returns:	none
******************************************************************************/
void NexusUnknownHandler(nexus_endpoint_t *endpoint,
			 nexus_buffer_t *buffer,
			 int handler_id)
{
    nexus_printf("NexusUnknownHandler(): handler_id=%d\n", handler_id);
    nexus_printf("NexusUnknownHandler(): exiting\n");
}
/* NexusUnknownHandler() */


/******************************************************************************
Function:	ft_callback_handler

Description:	catch any faults that happen and print them out

Parameters:	user_arg	(not used)
		fault_code	code identifying the fault which occurred

Returns:	zero - cleanup and proceed
******************************************************************************/
static int
ft_callback_handler(void * user_arg,
		    int fault_code)
{
    nexus_printf("ft_callback_handler(), code=%d, desc=%d\n",
		 fault_code, nexus_fault_strings[fault_code]);

    return(0);
}
