/******************************************************************************
Description:

  test_totem.c
  
  To run this program:
  
      test_totem <totem_host> <process group> <reps> <message>

CVS Information:

  $Source: /home/globdev/CVS/globus-current/Globus/Communication/nexus/tests/miscellaneous/test_totem.c,v $
  $Date: 1998/08/24 22:28:06 $
  $Revision: 1.4 $
  $State: Exp $
  $Author: toonen $
******************************************************************************/

/******************************************************************************
			     Include header files
******************************************************************************/
#include <stdio.h>
#include <string.h>
#include "globus_nexus.h"


/******************************************************************************
		       Define module specific constants
******************************************************************************/

#define DEBUG_LEVEL 2

/*
 * FT_ENABLE - enable testing of fault tolerance
 */
#define FT_ENABLE

#define NUM_MESSAGES	50
#define USLEEP_INTERVAL	10000
#define MESSAGE_SIZE	(128)


/******************************************************************************
			 Define module specific macros
******************************************************************************/
#if defined(DEBUG_LEVEL)
#   define debug_printf(level, message)		\
    {						\
        if ((level) <= (DEBUG_LEVEL))		\
        {					\
            nexus_printf message;		\
        }					\
    }
#else
#   define debug_printf(level, message)
#endif


/******************************************************************************
			       Type definitions
******************************************************************************/
typedef struct
{
    globus_mutex_t			mutex;
    globus_cond_t			cond;
    volatile int			cnt;
} Test_Toteml_monitor_t;    
    

/******************************************************************************
			  Module specific prototypes
******************************************************************************/
static void
Test_Toteml_handle_Msg(
    globus_nexus_endpoint_t *		endpoint, 
    globus_nexus_buffer_t *		recv_buf,
    globus_bool_t			is_non_threaded_handler);

void Test_Toteml_handle_Unknown(
    globus_nexus_endpoint_t *		endpoint,
    globus_nexus_buffer_t *		buffer,
    int					handler_id);

static int
Test_Toteml_handle_FT_Event(
    void *				user_arg,
    int					fault_code);


/******************************************************************************
			  Initialized handler tables
******************************************************************************/
static globus_nexus_handler_t Test_Toteml_handler_table[] =
{
    {NEXUS_HANDLER_TYPE_NON_THREADED,
     (globus_nexus_handler_func_t) Test_Toteml_handle_Msg}
};

#define Test_Toteml_handle_Msg_HID 0


/******************************************************************************
Function:	main()

Description:	main test program

Parameters:	argc		number of command line arguments
		argv		pointers to the actual arguments

Returns:	0 = success, !0 = failure
******************************************************************************/
int
main(int argc,
     char **argv)
{
    int					args_process_group;
    int					args_reps;
    char *				args_message;

    globus_nexus_endpointattr_t		endpoint_attr;
    globus_nexus_proto_info_totem_t	proto_info;  
    globus_nexus_endpoint_t		endpoint;
    globus_nexus_startpoint_t		startpoint;

    globus_nexus_buffer_t		buffer;
    int					message_size;

    Test_Toteml_monitor_t		monitor;
    int					cnt;

    int					rc;

    /*
     * Initialize Nexus
     */
    rc = globus_module_activate(GLOBUS_NEXUS_MODULE);
    if (rc != GLOBUS_SUCCESS)
    {
	globus_nexus_printf("main(): nexus_init() failed with rc=%d\n", rc);
	return(1);
    }

    /*
     * Enable fault tolerance if we are testing that
     *
     * NOTE: fault tolerance in Nexus means that Nexus will detect the fault,
     * put itself back in a reasonable state so it can continue, and the notify
     * the user that the fault has occurred.  It does not, however, provide the
     * user with a toolset to help their code tolerate the fault.  In other
     * words, knowing what messages the remote nodes have received and
     * processed is completely left to the user.
     */
#   if defined(FT_ENABLE)
    {
	globus_nexus_enable_fault_tolerance(
	    Test_Toteml_handle_FT_Event,
	    (void *) NULL);
    }
#   endif

    /*
     * Process commmand line arguments.  We do this after nexus_init() to allow
     * nexus_init() to remove any arguments which are targeted at Nexus rather
     * than the application.
     */
    if (argc != 4)
    {
	globus_nexus_printf(
	    "usage: test_totem <process group> <reps> <message>\n");
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	return(2);
    }

    NexusMalloc(
	main(),
	args_message,
	char *,
	strlen(argv[3]));

    args_process_group = atoi(argv[1]);
    args_reps = atoi(argv[2]);
    strcpy(args_message, argv[3]);

    /*
     * Initialize monitor structure
     */

    globus_mutex_init(&monitor.mutex, (globus_mutexattr_t *) NULL);
    globus_cond_init(&monitor.cond, (globus_condattr_t *) NULL);
    monitor.cnt = 0;

    /*
     * Create a endpoint attribute definition for Totem.  To do this, we need
     * to register the handler table, the routine to handle bad message handler
     * a IDs, as well as the information specific to the Totem protocol.
     */
    globus_nexus_endpointattr_init(&endpoint_attr);

    globus_nexus_endpointattr_set_unknown_handler(
	&endpoint_attr,
	Test_Toteml_handle_Unknown,
	NEXUS_HANDLER_TYPE_NON_THREADED);

    globus_nexus_endpointattr_set_handler_table(
	&endpoint_attr,
	Test_Toteml_handler_table, 
	(sizeof(Test_Toteml_handler_table) / sizeof(globus_nexus_handler_t)));

    proto_info.process_group = args_process_group;
    rc = globus_nexus_endpointattr_set_protocol(
	&endpoint_attr,
	NEXUS_PROTO_TYPE_TOTEM,
	&proto_info,
	sizeof(nexus_proto_info_totem_t));
    if (rc != GLOBUS_SUCCESS)
    {
	globus_nexus_printf(
	    "main(): nexus_endpointattr_set_protocol() failed, rc=%d\n",
	    rc);
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	return(3);
    }

    /*
     * Construct a new endpoint using the endpoint attributes defined
     * previously.  Once this completes, we will start receiving data from the
     * specified process group.
     */
    debug_printf(1, ("main(): calling nexus_endpoint_init()\n"));
    rc = globus_nexus_endpoint_init(&endpoint, &endpoint_attr);
    if (rc != GLOBUS_SUCCESS)
    {
	globus_nexus_printf(
	    "main(): nexus_endpoint_init() failed, rc=%d\n", rc);
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	return(4);
    }
    globus_nexus_endpoint_set_user_pointer(&endpoint, &monitor);

    /*
     * Construct a Totem startpoint from the endpoint
     */
    debug_printf(1, ("main(): calling nexus_startpoint_bind()\n"));

    rc = globus_nexus_startpoint_bind(&startpoint, &endpoint);
    if (rc != GLOBUS_SUCCESS)
    {
	nexus_printf("main(): nexus_startpoint_bind() failed, rc=%d\n", rc);
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	return(5);
    }

    /*
     * We are done with the endpoint attribute information which we previously
     * defined, so release any resource held by this definition
     */
    globus_nexus_endpointattr_destroy(&endpoint_attr);


    /*
     * Send a series of messages 
     */
    debug_printf(1, ("main(): calling nexus_send_rsr() %d times\n",
		     args_reps));

    monitor.cnt = args_reps;
    message_size = strlen(args_message);
    debug_printf( 1, ("message size is %d\n", message_size ));

    for (cnt = 0; cnt < args_reps; cnt++)
    {
      sleep(1);
	globus_nexus_buffer_init(
	    &buffer,
	    nexus_sizeof_int(1) + nexus_sizeof_char(message_size),
	    0);
	globus_nexus_put_int(&buffer, &message_size, 1);
	globus_nexus_put_char(&buffer, args_message, message_size);


	rc = globus_nexus_send_rsr(&buffer, 
				   &startpoint,
				   Test_Toteml_handle_Msg_HID,
				   GLOBUS_TRUE, 
				   GLOBUS_FALSE);
	if (rc != GLOBUS_SUCCESS)
	{
	    globus_nexus_printf(
		"main(): nexus_send_rsr() failed, rc=%d\n", rc);
	    globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	    return(6);
	}
    }

    globus_mutex_lock(&monitor.mutex);
    {
	while (monitor.cnt > 0)
	{
	    globus_cond_wait(&monitor.cond, &monitor.mutex);
	}
    }
    globus_mutex_unlock(&monitor.mutex);

    /*
     * Cleanup 
     */
    debug_printf(1, ("main(): calling nexus_startpoint_destroy()\n"));
    rc = globus_nexus_startpoint_destroy(&startpoint);
    if (rc != NEXUS_SUCCESS)
    {
	globus_nexus_printf(
	    "main(): nexus_startpoint_destroy() failed, rc=%d\n", rc);
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	return(7);
    }

    debug_printf(1, ("main(): calling nexus_endpoint_destroy()\n"));
    globus_nexus_endpoint_destroy(&endpoint);
    if (rc != NEXUS_SUCCESS)
    {
	globus_nexus_printf(
	    "main(): nexus_endpoint_destroy() failed, rc=%d\n", rc);
	globus_module_deactivate(GLOBUS_NEXUS_MODULE);
	return(8);
    }


    /*
     * Shutdown Nexus
     */
    debug_printf(1, ("main(): calling nexus_shutdown()\n"));
    globus_module_deactivate(GLOBUS_NEXUS_MODULE);


    debug_printf(1, ("main(): completed successfully()\n"));
    return(0);

} /* end main() */


/******************************************************************************
Function:	Test_Toteml_handle_Msg()

Description:	receive a message

Parameters:	endpoint	
		buffer
		hanlder_id

Returns:	none
******************************************************************************/
void Test_Toteml_handle_Msg(
    globus_nexus_endpoint_t *		endpoint,
    globus_nexus_buffer_t *		buffer,
    int					handler_id)
{
    Test_Toteml_monitor_t *		monitor;
    int					msg_sz;
    char				msg[1024];

    monitor = (Test_Toteml_monitor_t *)
	globus_nexus_endpoint_get_user_pointer(endpoint);

    globus_nexus_get_int(buffer, &msg_sz, 1);
    globus_nexus_get_char(buffer, msg, msg_sz);

    msg[msg_sz] = '\0';

    globus_mutex_lock(&monitor->mutex);
    {
	globus_nexus_printf(
	    "Test_Toteml_handle_Msg(): %d-%s\n", monitor->cnt, msg);

	monitor->cnt--;
	globus_cond_signal(&monitor->cond);
    }
    globus_mutex_unlock(&monitor->mutex);

}
/* Test_Toteml_handle_Msg() */


/******************************************************************************
Function:	Test_Toteml_handle_Unknown()

Description:	catch any bad handler invocation requests

Parameters:	endpoint	
		buffer
		hanlder_id

Returns:	none
******************************************************************************/
void Test_Toteml_handle_Unknown(
    globus_nexus_endpoint_t *		endpoint,
    globus_nexus_buffer_t *		buffer,
    int					handler_id)
{
    globus_nexus_printf(
	"Test_Toteml_handle_Unknown(): handler_id=%d\n", handler_id);
}
/* Test_Toteml_handle_Unknown() */


/******************************************************************************
Function:	Test_Toteml_handle_FT_Event

Description:	catch any faults that happen and print them out

Parameters:	user_arg	(not used)
		fault_code	code identifying the fault which occurred

Returns:	zero - cleanup and proceed
******************************************************************************/
static int
Test_Toteml_handle_FT_Event(
    void *				user_arg,
    int					fault_code)
{
    globus_nexus_printf(
	"Test_Toteml_handle_FT_Event(), code=%d\n", fault_code);

    return(0);
}
/* Test_Toteml_handle_FT_Event() */
