/*
 * test_harness.h
 *
 * harness <-> test interface
 *
 * harness_* -> provided by harness
 * test_*    -> provided by test
 *
 * Except for test_usage_string(), all test_ functions
 * will be called while Nexus is running.
 */

#ifndef TEST_HARNESS_H
#define TEST_HARNESS_H

#include "globus_nexus.h"
#include "cmd_line.h"

/*
 * test_usage_string()
 *
 * Print the usage string using printf().
 *
 * Format:
 * <4 spaces>section name
 * <1 tab>option<tabs>description (default)
 * trailing newline on all lines.
 */
void test_usage_string();

/*
 * test_startup()
 *
 * Do whatever global initialization is needed.
 * Test_debug_level is from the command line option
 * '-debug test=x', and defaults to -1 if none given.
 */
int test_startup(int test_debug_level);

/*
 * test_parse_options() 
 *
 * Called only on server, each arglist should be converted
 * to a string to be sent to all contexts.  Strings
 * should be malloc'd by test, harness will free them.
 * parsed_options[0] is endpoint options,
 * parsed_options[1..n] are runtest options.
 * return 0 on success.
 */
int test_parse_options(arg_list_t *options,
		       int n_tests, arg_list_t *runtest,
		       char **parsed_options);

/*
 * test_create_endpoint()
 *
 * *ep should be set to point to the test's endpoint.
 * cleanup_func will be called to allow the test to clean
 *	up data associated with the endpoint.
 * the harness will destroy the endpoint.
 * free_ep should be set to GLOBUS_TRUE if the endpoint was
 *	dynamically allocated, and the harness will free it.
 * return 0 on success.
 */
int test_create_endpoint(char *options,
			 globus_nexus_endpoint_t **ep,
			 void (**cleanup_func)(globus_nexus_endpoint_t *),
			 globus_bool_t *free_ep);

/*
 * test_init()
 * test_run()
 * test_done()
 *
 * For each -runtest given on the command line, these will be called.
 * test_init() allows the test to set up the endpoint for handling
 *	RSRs.  No RSRs should be sent.  The returned value will be
 *	passed to test_run and test_done.
 * test_run() all nodes are guaranteed to have finished test_init.
 * test_done() all nodes have finished test_run.
 */
void *test_init(char *test_options,
		int size, int rank,
		globus_nexus_startpoint_t **sp);
void test_run(void *test_arg);
void test_done(void *test_arg);

/*
 * test_cleanup()
 *
 * All tests have been run, do final cleanup.  After this,
 * the endpoint will be destroyed, nexus shutdown, and
 * harness_main will return.
 */
void test_cleanup();

/*
 * harness_barrier()
 * harness_barrier_passive()
 * harness_barrier_set_udelay()
 */
#define harness_barrier() \
	harness_barrier_n(1)
#define harness_barrier_passive() \
	harness_barrier_passive(1)
#define harness_barrier_set_udelay(d) \
	harness_barrier_set_udelay_n(1, d)
    
int harness_barrier_n(int n);
void harness_barrier_passive_n(int n);
void harness_barrier_set_udelay_n(int n, int udelay);

/*
 * main()
 *
 * Provided by the harness.
 */
int main(int argc, char **argv);

/*
 * harness_abort()
 *
 * Got fatal error: shutdown all nodes, attempt cleanup.
 */
void harness_abort();

/*
 * harness_set_error()
 *
 * Set the error code to be returned to the system.
 * All contexts will return the same code.  0 is ok,
 * 1..9 are reserved for the harness, 10 and above
 * are available for the test.  In case of multiple
 * errors, the lowest non-zero value will have
 * precedence.
 */
void harness_set_error(int error);

#endif
