/*
 * startup.c
 */

#include "globus_common.h"

#include <stdlib.h>
#include <stdio.h>

#include "globus_nexus.h"
#if defined(GLOBUS_GRAM_ENABLED)
#   include "globus_gram_myjob.h"
#endif
#include "startup.h"
#include "harness_options.h"
#include "cmd_line.h"
#include "harness_endpoint.h"
#include "ep_list.h"
#include "sp_group.h"
#include "barrier.h"

static struct
{
    globus_mutex_t	mutex;
    globus_cond_t	cond;
    volatile int	done;
} monitor;

static int		attaches_total	= 0;
static volatile int	attaches_sofar	= 0;
static sp_list_t	*attach_lists	= NULL;

int
startup_init(int *argc, char ***argv)
{
    int	r;

    r = globus_module_activate(GLOBUS_NEXUS_MODULE);

    if (r != GLOBUS_SUCCESS)
    {
	globus_libc_printf("startup_init(): globus_module_activate"
			   "(GLOBUS_NEXUS_MODULE) failed, "
			   "returned %i\n", r);
	return (1);
    }
    if (globus_mutex_init(&(monitor.mutex), (globus_mutexattr_t *) NULL)
	!= GLOBUS_SUCCESS)
    {
	globus_libc_printf("startup_init(): globus_mutex_init failed\n");
	return (1);
    }
    if (globus_cond_init(&(monitor.cond), (globus_condattr_t *) NULL)
	!= GLOBUS_SUCCESS)
    {
	globus_libc_printf("startup_init(): globus_cond_init failed");
	return (1);
    }
    monitor.done = 0;
    r = ep_list_startup();
    if (r != 0)
    {
	globus_libc_printf("startup_init(): ep_list_startup() failed\n");
	return (1);
    }
    r = sp_group_startup();
    if (r != 0)
    {
	globus_libc_printf("startup_init(): sp_group_startup() failed\n");
	return (1);
    }
    r = barrier_startup();
    if (r != 0)
    {
	globus_libc_printf("startup_init(): barrier_startup() failed\n");
	return (1);
    }
    r = harness_endpoint_init();
    if (r != 0)
    {
	printf("startup_init(): harness_endpoint_init() "
	       "failed, return %i\n", r);
	return (1);
    }
    return (0);
}

static void
send_debug_level(int harness_dbg_lvl,
		 int test_dbg_lvl,
		 sp_list_t *sp_list)
{
    globus_nexus_buffer_t	buf;
    int				i, x;

    x = -1;
    for (i = 0; i < sp_list_get_size(sp_list); i++)
    {
	if (globus_nexus_buffer_init(&buf, globus_nexus_sizeof_int(1), 0)
	    != GLOBUS_SUCCESS)
	{
	    globus_libc_printf("send_debug_level(): buffer_init failed\n");
	    harness_abort();
	}
	globus_nexus_put_int(&buf, &harness_dbg_lvl, 1);
	globus_nexus_put_int(&buf, &test_dbg_lvl, 1);
	globus_nexus_put_int(&buf, &x, 1);
	if (globus_nexus_send_rsr(&buf,
				  sp_list_get_sp(sp_list, i),
				  SET_DEBUG_LEVEL_HANDLER_ID,
				  GLOBUS_TRUE, GLOBUS_FALSE)
	    != GLOBUS_SUCCESS)
	{
	    globus_libc_printf("send_debug_level(): send_rsr failed\n");
	    harness_abort();
	}
    }
}

int
startup_local(int *argc, char ***argv,
	      sp_list_t *sp_list,
	      int local_dbg_lvl,
	      int test_dbg_lvl)
{
#   if defined(GLOBUS_GRAM_ENABLED)
        globus_byte_t	buf[GRAM_MYJOB_MAX_BUFFER_LENGTH];
#   endif
    int			r;
    int			rank, size;
    
#   if defined(GLOBUS_GRAM_ENABLED)
    {
        r = globus_module_activate(GLOBUS_GRAM_MYJOB_MODULE);
        if (r != GLOBUS_SUCCESS)
        {
	    globus_libc_printf("startup_local(): gram_myjob_init() "
			       "failed with error %i\n", r);
	    return (1);
        }
        r = globus_gram_myjob_rank(&rank);
        if (r != GLOBUS_SUCCESS)
        {
	    globus_libc_printf("startup_local(): globus_gram_myjob_rank() "
			       "failed with error %i\n", r);
	    return (1);
        }
        r = globus_gram_myjob_size(&size);
        if (r != GLOBUS_SUCCESS)
        {
	    globus_libc_printf("startup_local(): globus_gram_myjob_size() "
			       "failed with error %i\n", r);
	    return (1);
        }
    }
#   else
    {
	rank=0;
	size=1;
    }
#   endif
    HelloStu(globus_libc_printf("startup_local(): rank = %i of %i\n",
				rank, size));
    if (rank == 0)
    {
	globus_nexus_startpoint_t	*sp;
	globus_byte_t			*buf_ptr;
	int				buf_len;
	int				i, fmt, src;

	r = sp_list_set_size(sp_list, size);
	if (r != 0)
	{
	    globus_libc_printf("startup_local(): failed to set "
			       "size for sp_list\n");
	    return (1);
	}
	r = sp_list_alloc_sps(sp_list, 0, size);
	if (r != 0)
	{
	    globus_libc_printf("startup_local(): create_sps failed\n");
	    return (1);
	}
	sp = sp_list_get_sp(sp_list, 0);
	r = globus_nexus_startpoint_bind(sp,
					 ep_list_get_ep(HARNESS_ENDPOINT_ID));
	if (r != GLOBUS_SUCCESS)
	{
	    globus_libc_printf("startup_local(): failed to bind "
			       "server sp\n");
	    return (1);
	}
	for (i = 1; i < size; i++)
	{
#           if defined(GLOBUS_GRAM_ENABLED)
	    {
	        r = gram_myjob_receive(buf, &buf_len);
	        if (r != GLOBUS_SUCCESS)
	        {
		    globus_libc_printf("startup_local(): gram_myjob"
				       "_receive() failed with error %i\n",
				       r);
		    return (1);
	        }
	        fmt = buf[0];
	        buf_ptr = &(buf[1]);
	        globus_nexus_user_get_int(&buf_ptr, &src, 1, fmt);
	        if ((src < 1) || (src >= size)) {
		    globus_libc_printf("startup_local(): got invalid "
				       "source %i\n", src);
		    return (1);
	        }
	        sp = sp_list_get_sp(sp_list, src);
	        globus_nexus_user_get_startpoint(&buf_ptr, sp, 1, fmt);
	        TraceStu(globus_libc_printf("harness_startup_local(): "
					    "got sp from %i\n", src));
	    }
#           endif
	}
	send_debug_level(local_dbg_lvl, test_dbg_lvl, sp_list);
    }
    else
    {
#       if defined(GLOBUS_GRAM_ENABLED)
	{
	    globus_nexus_startpoint_t	sp;
	    globus_byte_t			*buf_ptr;
	    int				buf_len;
    
	    r = sp_list_set_size(sp_list, 0);
	    if (r != 0)
	    {
	        globus_libc_printf("startup_local(): failed to set "
			           "sp_list size\n");
	        return (1);
	    }
	    r = globus_nexus_startpoint_bind(&sp,
					     ep_list_get_ep(HARNESS_ENDPOINT_ID));
	    if (r != GLOBUS_SUCCESS)
	    {
	        globus_libc_printf("harness_startup_local(): "
			           "globus_nexus_startpoint_"
			           "bind() failed with error %i\n", r);
	        return (1);
	    }
	    buf[0] = (globus_byte_t) globus_nexus_user_format();
	    buf_ptr = &(buf[1]);
	    globus_nexus_user_put_int(&buf_ptr, &rank, 1);
	    globus_nexus_user_put_startpoint_transfer(&buf_ptr, &sp, 1);
	    buf_len = buf_ptr - buf;
	    r = gram_myjob_send(0, buf, buf_len);
	    if (r != GLOBUS_SUCCESS)
	    {
	        globus_libc_printf("startup_local(): send failed\n");
	        return (1);
	    }
	}
#       endif
    }
#   if defined(GLOBUS_MYJOB_ENABLED)
    {
        r = globus_module_deactivate(GLOBUS_GRAM_MYJOB_MODULE);
        if (r != GLOBUS_SUCCESS)
        {
	    globus_libc_printf("startup_local(): globus_module_deactivate"
			       "(GLOBUS_GRAM_MYJOB_MODULE) "
			       "returned error %i\n", r);
        }
    }
#   endif
    TraceStu(globus_libc_printf("startup_local(): success\n"));
    DebugStu(sp_list_print(sp_list));
    return (0);
}

static int
attach_request(void *user_arg, char *url, globus_nexus_startpoint_t *sp)
{
    globus_libc_printf("attach_request(): url=%s\n", url);
    globus_nexus_startpoint_bind(sp, ep_list_get_ep(HARNESS_GROUP_ID));
    return(0);
}

int
startup_listen(int n_attaches,
	       sp_list_t *sp_list,
	       int global_dbg_lvl,
	       int test_dbg_lvl)
{
    int			i, r;
    char		*host;
    unsigned short	port;
    
    globus_libc_printf("startup_listen(): for %i attaches\n", n_attaches);
    if (n_attaches < 1)
    {
	return (1);
    }
    globus_libc_printf("startup_listen(): starting with %i SPs\n",
		       sp_list_get_size(sp_list));
    attach_lists = (sp_list_t *)
	globus_malloc(n_attaches * sizeof(sp_list_t));
    if (attach_lists == NULL)
    {
	globus_libc_printf("startup_listen(): malloc failed\n");
	return (1);
    }
    globus_mutex_lock(&(monitor.mutex));
    attaches_total = n_attaches;
    attaches_sofar = 0;
    port = 0;
    r = globus_nexus_allow_attach(&port,
				  &host,
				  &attach_request,
				  (void *) NULL);
    if (r != GLOBUS_SUCCESS)
    {
	globus_libc_printf("startup_listen(): allow_attach failed\n");
	return (1);
    }
    globus_libc_printf("startup listen(): ATTACH TO: x-nexus://%s:%hu\n",
		       host, port);
    fflush(stdout);
    while (attaches_sofar < attaches_total)
    {
	globus_cond_wait(&(monitor.cond), &(monitor.mutex));
    }
    globus_nexus_disallow_attach(port);
    globus_libc_printf("startup_listen(): attaches disallowed\n");
    globus_mutex_unlock(&(monitor.mutex));
    for (i = 0; i < n_attaches; i++)
    {
	send_debug_level(global_dbg_lvl,
			 test_dbg_lvl,
			 &(attach_lists[i]));
	r = sp_list_append_list(sp_list, &(attach_lists[i]));
	if (r != 0)
	{
	    globus_libc_printf("startup_listen(): append failed\n");
	    return (1);
	}
    }
    for (i = 0; i < n_attaches; i++)
    {
	r = sp_list_done(&(attach_lists[i]));
    }
    globus_free(attach_lists);
    globus_libc_printf("startup_listen(): now have %i sps\n",
		       sp_list_get_size(sp_list));
    DebugStu(sp_list_print(sp_list));
    return (0);
}

int
startup_attach(char *url, sp_list_t *sp_list)
{
    globus_nexus_buffer_t	buffer;
    int			r, s;
    globus_nexus_startpoint_t	sp;

    /*
      sprintf(url, "%s/spec1//spec\\/3/spec4", attach_client_url);
      globus_libc_printf("attach_client(): attaching to url <%s>\n", url);
      */
    r = globus_nexus_attach(url, &sp);
    if (r != GLOBUS_SUCCESS)
    {
	globus_libc_printf("startup_attach(): attach failed\n");
	return (1);
    }
    globus_libc_printf("startup_attach(): attached\n");
    DebugStu(sp_list_print(sp_list));
    r = sp_list_buffer_size(sp_list, &s);
    if (r != 0)
    {
	globus_libc_printf("startup_attach(): sp_list_buffer_size failed\n");
	return (1);
    }
    r = globus_nexus_buffer_init(&buffer, s, 0);
    if (r != GLOBUS_SUCCESS)
    {
	globus_libc_printf("startup_attach(): buffer_init failed\n");
	return (1);
    }
    globus_libc_printf("startup_attach(): sending %i sps\n",
		       sp_list_get_size(sp_list));
    r = sp_list_buffer_put(sp_list, &buffer);
    if (r != 0)
    {
	globus_libc_printf("startup_attach(): buffer_put failed\n");
	return (1);
    }
    r = globus_nexus_send_rsr(&buffer,
			      &sp,
			      ATTACH_STARTPOINTS_HANDLER_ID,
			      GLOBUS_TRUE,   /* destroy buffer */
			      GLOBUS_FALSE); /* from non-threaded handler */
    globus_libc_printf("startup_attach(): sent rsr, r=%i\n", r);
    globus_nexus_startpoint_destroy(&sp);
    return (r);
}

void
startup_done(sp_list_t *sp_list)
{
    if (sp_list == NULL)
    {
	/* slave */
	TraceStu(globus_libc_printf("startup_done(): slave\n"));
    }
    else
    {
	/* master */
	globus_nexus_startpoint_t	sp;
	globus_nexus_buffer_t		buf;
	int				rank;
	int				size;

	TraceStu(globus_libc_printf("startup_done(): master\n"));
	DebugStu(sp_list_print(sp_list));
	/* save SPs to slaves */
	sp_group_bootstrap(HARNESS_GROUP_ID, HARNESS_ENDPOINT_ID,
			   sp_list_get_size(sp_list),
			   0,		/* rank */
			   sp_list);
	rank = sp_group_rank(HARNESS_GROUP_ID);
	size = sp_group_size(HARNESS_GROUP_ID);
	DebugStu(globus_libc_printf("startup_done(): rank=%i of %i\n",
				    rank, size));
	/* Now send copy of master SP to others */
	for (rank = 1; rank < size; rank++)
	{
	    if (globus_nexus_startpoint_bind(
		&sp, ep_list_get_ep(HARNESS_ENDPOINT_ID))
		!= GLOBUS_SUCCESS)
	    {
		globus_libc_printf("startup_done(): sp_bind failed\n");
		return; /* XXX should crash */
	    }
	    if (globus_nexus_buffer_init(&buf,
				  2 * globus_nexus_sizeof_int(1) +
				  globus_nexus_sizeof_startpoint(&sp, 1),
				  0) != GLOBUS_SUCCESS)
	    {
		globus_libc_printf("startup_done(): buf_init failed\n");
		return; /* XXX should crash */
	    }
	    globus_nexus_put_int(&buf, &size, 1);
	    globus_nexus_put_int(&buf, &rank, 1);
	    globus_nexus_put_startpoint_transfer(&buf, &sp, 1);
	    TraceStu(globus_libc_printf("sending to slave %i\n", rank));
	    if (globus_nexus_send_rsr(&buf,
				      sp_group_sp(HARNESS_GROUP_ID, rank),
				      HARNESS_STARTUP_HANDLER_ID,
				      GLOBUS_TRUE,
				      GLOBUS_FALSE) != GLOBUS_SUCCESS)
	    {
		globus_libc_printf("send failed\n");
	    }
	}
    }
    globus_mutex_destroy(&(monitor.mutex));
    globus_cond_destroy(&(monitor.cond));
    TraceStu(globus_libc_printf("startup_done(): goodbye\n"));
}

void
harness_abort()
{
    globus_libc_printf("harness_abort(): Aborting\n");
    sp_group_shutdown(GLOBUS_TRUE);
    ep_list_shutdown(GLOBUS_TRUE);
    globus_silent_fatal();
}

/*
 *
 * handlers
 *
 */

void
attach_startpoints_handler(globus_nexus_endpoint_t *endpoint,
			   globus_nexus_buffer_t *buffer,
			   globus_bool_t non_threaded_handler)
{
    sp_list_t	*sp_list;

    globus_mutex_lock(&(monitor.mutex));
    if (attaches_sofar >= attaches_total)
    {
	globus_libc_printf("attach_startpoints(): error: sofar=%i, "
			   "total=%i\n", attaches_sofar, attaches_total);
	globus_mutex_unlock(&(monitor.mutex));
	return;
    }
    else
    {
	TraceStu(globus_libc_printf("attach_startpoints(): saving to %i\n",
				    attaches_sofar));
    }
    sp_list = &(attach_lists[attaches_sofar]);
    if (sp_list_init(sp_list) != 0)
    {
	globus_libc_printf("attach_startpoints_handler(): "
			   "sp_list_init failed\n");
	harness_abort();
    }
    sp_list_buffer_get(sp_list, buffer);
    DebugStu(globus_libc_printf("attach_startpoints(): got %i sps\n",
				sp_list_get_size(sp_list)));
    DebugStu(sp_list_print(sp_list));
    attaches_sofar++;
    if (attaches_sofar == attaches_total)
    {
	globus_libc_printf("attach_startpoints(): last attach; signaling\n");
	globus_cond_signal(&(monitor.cond));
    }
    globus_mutex_unlock(&(monitor.mutex));
}

void
startup_handler(globus_nexus_endpoint_t *endpoint,
		globus_nexus_buffer_t *buffer,
		globus_bool_t is_non_threaded_handler)
{
    globus_nexus_startpoint_t	*sp;
    sp_list_t			sp_list;
    int				size, rank;

    globus_mutex_lock(&(monitor.mutex));
    if (monitor.done < 0)
    {
	globus_libc_printf("startup_handler(): done == %i\n",
		     monitor.done);
	globus_mutex_unlock(&(monitor.mutex));
	return;
    }
    if (sp_list_init(&sp_list) != 0)
    {
	globus_libc_printf("startup_handler(): sp_list_init failed\n");
	return;
    }
    if ((sp = (globus_nexus_startpoint_t *)
	 globus_malloc(sizeof(globus_nexus_startpoint_t))) == NULL)
    {
	globus_libc_printf("startup_handler(): malloc failed\n");
	return;
    }
    globus_nexus_get_int(buffer, &size, 1);
    globus_nexus_get_int(buffer, &rank, 1);
    globus_nexus_get_startpoint(buffer, sp, 1);
    if (sp_list_append_sp(&sp_list, sp) != 0)
    {
	globus_libc_printf("startup_handler(): append_sp failed\n");
	return;
    }
    sp_group_bootstrap(HARNESS_GROUP_ID, HARNESS_ENDPOINT_ID,
		       size, rank,
		       &sp_list);
    TraceStu(globus_libc_printf("startup_handler(): rank=%i,size=%i\n",
				sp_group_rank(HARNESS_GROUP_ID),
				sp_group_size(HARNESS_GROUP_ID)));
    monitor.done--;
    if (monitor.done == 0)
    {
	TraceStu(globus_libc_printf("startup_handler(): done = 0; "
				    "signalling\n"));
	globus_cond_signal(&(monitor.cond));
    }
    globus_mutex_unlock(&(monitor.mutex));
}

void
shutdown_handler(globus_nexus_endpoint_t *endpoint,
		 globus_nexus_buffer_t *buffer,
		 globus_bool_t is_non_threaded_handler)
{
    /*
     * ignoring buffer contents
     * - who
     * - why
     */
    globus_libc_printf("shutdown_handler(): EXITING\n");
    harness_abort();
}
