/*
 * sp_list.h
 *
 * Routines to manipulate lists of startpoint pointers.
 *
 * All functions return 0 on success.
 */

#ifndef SP_LIST_H
#define SP_LIST_H

#include "globus_nexus.h"

/*
 * #define SKIP_CHECKS when compiling sp_list.c, and the
 * checks for valid arguments will not be done.
 */

typedef struct
{
    globus_nexus_startpoint_t	**sp;
    int				max_n_sps;
    int				cur_n_sps;
} sp_list_t;

/*
 * sp_list_init()
 * sp_list_done()
 *
 * Initialize or clean up a list.  Done() will free the
 * memory of any startpoints left in the list, but
 * only warns if they were not previously destroyed.
 */
int sp_list_init(sp_list_t *sl);
int sp_list_done(sp_list_t *sl);

/*
 * sp_list_get_size()
 * sp_list_get_sp()
 *
 * Find out how many startpoints are in the list,
 * and get a pointer to one of them.
 */
int sp_list_get_size(sp_list_t *sl);
globus_nexus_startpoint_t *sp_list_get_sp(sp_list_t *sl, int sp_num);

/*
 * sp_list_destroy_sps()
 *
 * Calls globus_nexus_startpoint_destroy() for each SP in the given
 * range.  Skips any NULL pointers.
 */
int sp_list_destroy_sps(sp_list_t *sl, int first, int count);

/*
 * sp_list_append_sp()
 * sp_list_append_list()
 *
 * The SP given to append_sp() must be dynamically allocated.
 * The src list in append_list() will be emptied.
 */
int sp_list_append_sp(sp_list_t *dst, globus_nexus_startpoint_t *sp);
int sp_list_append_list(sp_list_t *dst, sp_list_t *src);

/*
 * sp_list_buffer_size()
 * sp_list_buffer_put()
 * sp_list_buffer_get()
 *
 * For sending sp_lists through nexus buffers.  Size()
 * determines how much space the list will take.  Put()
 * uses globus_nexus_put_startpoint_transfer, and empties the
 * list.  Get() appends the received SPs to the list.
 */
int sp_list_buffer_size(sp_list_t *sl, int *size);
int sp_list_buffer_put(sp_list_t *sl, globus_nexus_buffer_t *buf);
int sp_list_buffer_get(sp_list_t *sl, globus_nexus_buffer_t *buf);

/*
 * sp_list_set_size()
 * sp_list_alloc_sps()
 * sp_list_set_sp()
 *
 * Set_size() will change the size of the list.  It does not
 * free the memory of any truncated SPs.  If the list is
 * expanded, the new pointers will be set to NULL.
 * Alloc_sps() will, for each NULL pointer in the range,
 * allocate memory for a startpoint and set it null.
 * Set_sp() will clobber the old pointer with the new.
 */
int sp_list_set_size(sp_list_t *sl, int size);
int sp_list_alloc_sps(sp_list_t *sl, int first, int count);
int sp_list_set_sp(sp_list_t *sl, int sp_num, globus_nexus_startpoint_t *sp);

/*
 * sp_list_print()
 *
 * For debugging -- prints out the list size and addresses of SPs.
 * Returns non-zero if the list is totally corrupt.
 */
int sp_list_print(sp_list_t *sl);

#endif
