/*
 * sp_list.c
 */

#include "globus_nexus.h"
#include "sp_list.h"

#ifdef SKIP_CHECKS
#  define VALID_CHECK(func, p_list, ret_err_val)
#  define INDEX_CHECK(func, p_list, index, ret_err_val)
#  define RANGE_CHECK(func, p_list, first, count, ret_err_val)
#else
#  define VALID_CHECK(func, p_list, ret_err_val)		\
	if ( ((p_list) == NULL) || ((p_list)->cur_n_sps < 0)	\
	     || ((p_list)->max_n_sps < 0)) {			\
		globus_libc_printf(func "(): invalid list\n");	\
		return (ret_err_val);				\
	}
#  define INDEX_CHECK(func, p_list, index, ret_err_val)		\
	if (( (index) < 0) || ((index) > (p_list)->cur_n_sps)) {\
		globus_libc_printf(func "(): index %i out of range"	\
			     ", size=%i\n", (index),		\
			     (p_list)->cur_n_sps);		\
		return (ret_err_val);				\
	}
#  define RANGE_CHECK(func, p_list, first, count, ret_err_val)	\
	if (( (first) < 0) || ((count) < 0) ||			\
	    ( (first) + (count) > (p_list)->cur_n_sps)) {	\
		globus_libc_printf(func "(): first=%i, count=%i, "	\
			     "size=%i\n", (first), (count),	\
			     (p_list)->cur_n_sps);		\
		return (ret_err_val);				\
	}
#endif

#define LIST_MOD_SIZE	8
#define LIST_MAX_SIZE	120

/*
 * sp_list_resize()
 *
 * Used internally to change the number of SPs in
 * the list.  Never destroys SPs, but will free them
 * if they are lost due to downsizing.  If the list
 * is expanded, the new pointers will be NULL.
 */
static int
sp_list_resize(sp_list_t *sl, int new_size)
{
    globus_nexus_startpoint_t	**new_list; 
    int			i, new_mod_size;

    if ((new_size < 0) || (new_size > LIST_MAX_SIZE)) {
	globus_libc_printf("sp_list_resize(): bad size=%i\n", new_size);
	return (1);
    }
    if (new_size == sl->cur_n_sps)
	return (0);
    new_mod_size = new_size + LIST_MOD_SIZE - 1;
    new_mod_size -= (new_mod_size % LIST_MOD_SIZE);
    if (new_size < sl->cur_n_sps) {
	/* shrink list */
	for (i = new_size; i < sl->cur_n_sps; i++) {
	    if (sl->sp[i] != NULL) {
		if (!globus_nexus_startpoint_is_null(sl->sp[i]))
		    globus_libc_printf("sp_list_resize(): "
				 "!null(sp[%i])\n", i);
		globus_free(sl->sp[i]);
		sl->sp[i] = NULL;
	    }
	}
	if (new_mod_size < sl->max_n_sps) {
	    if (new_mod_size == 0)
		globus_free(sl->sp);
	    else {
		new_list = (globus_nexus_startpoint_t **)
		    globus_malloc(new_mod_size *
				 sizeof(globus_nexus_startpoint_t *));
		if (new_list == NULL) {
		    globus_libc_printf("sp_list_resize(): error: "
				 "malloc() failed\n");
		    return (1);
		}
		for (i = 0; i < new_mod_size; i++)
		    new_list[i] = sl->sp[i];
		globus_free(sl->sp);
		sl->sp = new_list;
	    }
	}
    } else {
	/* grow list */
	if (new_mod_size > sl->max_n_sps) {
		new_list = (globus_nexus_startpoint_t **)
		    globus_malloc(new_mod_size *
				 sizeof(globus_nexus_startpoint_t *));
		if (new_list == NULL) {
		    globus_libc_printf("sp_list_resize(): error: "
				 "malloc() failed\n");
		    return (1);
		}
		for (i = 0; i < sl->max_n_sps; i++)
		    new_list[i] = sl->sp[i];
		for (i = sl->max_n_sps; i < new_mod_size; i++)
		    new_list[i] = NULL;
		globus_free(sl->sp);
		sl->sp = new_list;
	}
    }
    sl->max_n_sps = new_mod_size;
    sl->cur_n_sps = new_size;
    return (0);
}

int
sp_list_init(sp_list_t *sl)
{
    if (sl == NULL) {
	globus_libc_printf("sp_list_init(): sl == NULL\n");
	return (1);
    }
    sl->sp = (globus_nexus_startpoint_t **) NULL;
    sl->max_n_sps = 0;
    sl->cur_n_sps = 0;
    return (0);
}

int
sp_list_done(sp_list_t *sl)
{
    int		r;

    VALID_CHECK("sp_list_done", sl, 1);
    r = sp_list_resize(sl, 0);
    if (r != 0)
	globus_libc_printf("sp_list_done(): resize(0) failed\n");
    sl->max_n_sps = -1;
    sl->cur_n_sps = -1;
    return (r);
}

int
sp_list_get_size(sp_list_t *sl)
{
    VALID_CHECK("sp_list_get_size", sl, -1);
    if (sl->cur_n_sps < 0)
	globus_libc_printf("sp_list_get_size(): bad size=%i\n",
		     sl->cur_n_sps);
    return (sl->cur_n_sps);
}

globus_nexus_startpoint_t *
sp_list_get_sp(sp_list_t *sl, int sp_num)
{
    VALID_CHECK("sp_list_get_sp", sl, NULL);
    INDEX_CHECK("sp_list_get_sp", sl, sp_num, NULL);
    return (sl->sp[sp_num]);
}

int
sp_list_destroy_sps(sp_list_t *sl, int first, int count)
{
    int		i, r;

    VALID_CHECK("sp_list_destroy_sps", sl, 1);
    RANGE_CHECK("sp_list_destroy_sps", sl, first, count, 1);
    for (i = first; i < (first + count); i++) {
	if (sl->sp[i] != NULL) {
	    r = globus_nexus_startpoint_destroy(sl->sp[i]);
	    if (r != GLOBUS_SUCCESS)
		globus_libc_printf("sp_list_destroy_sps(): sp_destroy[%i] "
			     "returned error %i\n", i, r);
	}
    }
    return (0);
}

int
sp_list_append_sp(sp_list_t *dst, globus_nexus_startpoint_t *sp)
{
    VALID_CHECK("sp_list_append_sp", dst, 1);
    if (sp == NULL)
	globus_libc_printf("sp_list_append_sp(): warning: sp == NULL\n");
    if (sp_list_resize(dst, dst->cur_n_sps + 1) != 0) {
	globus_libc_printf("sp_list_append_sp(): resize failed\n");
	return (1);
    }
    dst->sp[dst->cur_n_sps - 1] = sp;
    return (0);
}

int
sp_list_append_list(sp_list_t *dst, sp_list_t *src)
{
    int		old_size, new_size;
    int		i;

    VALID_CHECK("sp_list_append_list(dst)", dst, 1);
    VALID_CHECK("sp_list_append_list(src)", src, 1);
    old_size = dst->cur_n_sps;
    new_size = old_size + src->cur_n_sps;
    if (sp_list_resize(dst, new_size) != 0) {
	globus_libc_printf("sp_list_append_list(): resize failed\n");
	return (1);
    }
    for (i = 0; i < src->cur_n_sps; i++) {
	dst->sp[old_size + i] = src->sp[i];
	src->sp[i] = NULL;
    }
    if (sp_list_resize(src, 0) != 0)
	globus_libc_printf("sp_list_append_list(): src resize failed\n");
    return (0);
}

int
sp_list_buffer_size(sp_list_t *sl, int *size)
{
    int	i, r, s;

    VALID_CHECK("sp_list_buffer_size", sl, 1);
    r = 0;
    s = globus_nexus_sizeof_int(1);
    for (i = 0; i < sl->cur_n_sps; i++)
	if (sl->sp[i] == NULL)
	    r++;
	else
	    s += globus_nexus_sizeof_startpoint(sl->sp[i], 1);
    *size = s;
    if (r > 0)
	globus_libc_printf("sp_list_buffer_size(): %i NULL ptrs\n", r);
    return (r);
}

int
sp_list_buffer_put(sp_list_t *sl, globus_nexus_buffer_t *buf)
{
    int	i, r;

    VALID_CHECK("sp_list_buffer_put", sl, 1);
    r = 0;
    globus_nexus_put_int(buf, &(sl->cur_n_sps), 1);
    for (i = 0; i < sl->cur_n_sps; i++)
	if (sl->sp[i] == NULL)
	    r++;
	else
	    globus_nexus_put_startpoint_transfer(buf, sl->sp[i], 1);
    if (r > 0)
	globus_libc_printf("sp_list_buffer_put(): %i NULL ptrs\n", r);
    if (sp_list_resize(sl, 0) != 0)
	globus_libc_printf("sp_list_buffer_put(): resize failed\n");
    return (r);
}

int
sp_list_buffer_get(sp_list_t *sl, globus_nexus_buffer_t *buf)
{
    globus_nexus_startpoint_t	*sp;
    int			i, l;

    VALID_CHECK("sp_list_buffer_get", sl, 1);
    globus_nexus_get_int(buf, &l, 1);
    if (l < 0) {
	globus_libc_printf("sp_list_buffer_get(): len=%i\n", l);
	return (1);
    }
    for (i = 0; i < l; i++) {
	sp = (globus_nexus_startpoint_t *)
	    globus_malloc(sizeof(globus_nexus_startpoint_t));
	if (sp == NULL) {
	    globus_libc_printf("sp_list_buffer_get(): malloc failed\n");
	    return (1);
	}
	globus_nexus_get_startpoint(buf, sp, 1);
	if (sp_list_append_sp(sl, sp) != 0) {
	    globus_libc_printf("sp_list_buffer_get(): append_sp failed\n");
	    return (1);
	}
    }
    return (0);
}

int
sp_list_set_size(sp_list_t *sl, int size)
{
    int		i;

    VALID_CHECK("sp_list_set_size", sl, 1);
    if (size < sl->cur_n_sps)
	for (i = size - 1; i < sl->cur_n_sps; i++)
	    sl->sp[i] = NULL;
    i = sp_list_resize(sl, size);
    if (i != 0)
	globus_libc_printf("sp_list_set_size(): resize failed\n");
    return (i);
}

int
sp_list_alloc_sps(sp_list_t *sl, int first, int count)
{
    int		i;

    VALID_CHECK("sp_list_alloc_sps", sl, 1);
    RANGE_CHECK("sp_list_alloc_sps", sl, first, count, 1);
    for (i = first; i < (first + count); i++)
	if (sl->sp[i] == NULL) {
	    sl->sp[i] = (globus_nexus_startpoint_t *)
		globus_malloc(sizeof(globus_nexus_startpoint_t));
	    if (sl->sp[i] == NULL) {
		globus_libc_printf("sp_list_alloc_sps(): malloc failed\n");
		return (1);
	    }
	    globus_nexus_startpoint_set_null(sl->sp[i]);
	}
    return (0);
}

int
sp_list_set_sp(sp_list_t *sl, int sp_num, globus_nexus_startpoint_t *sp)
{
    VALID_CHECK("sp_list_set_sp", sl, 1);
    INDEX_CHECK("sp_list_set_sp", sl, sp_num, 1);
    sl->sp[sp_num] = sp;
    return (0);
}

int
sp_list_print(sp_list_t *sl)
{
    int		i, used, extra;

    VALID_CHECK("sp_list_print", sl, 1);
    used = 0;
    for (i = 0; i < sl->cur_n_sps; i++)
	if (sl->sp[i] != NULL)
	    used++;
    extra = used;
    for (i = sl->cur_n_sps; i < sl->max_n_sps; i++)
	if (sl->sp[i] != NULL)
	    extra++;
    globus_libc_printf("sp_list_print(): \n"
		 "\tcur=%i, used=%i\n"
		 "\tmax=%i, used=%i\n",
		 sl->cur_n_sps, used,
		 sl->max_n_sps, extra);
    for (i = 0; i < sl->max_n_sps; i++)
	globus_libc_printf("\tsp[%i] @ 0x%lx\n", i, (long) sl->sp[i]);
    return (0);
}
