/*
 * sp_group.c
 */

#include "globus_nexus.h"
#include "sp_group.h"
#include "harness_options.h"
#include "sp_list.h"
#include "ep_list.h"
#include "harness_endpoint.h"
#include "startup.h"

#ifdef SKIP_CHECKS
#  define VALID_GROUP(func, group, ret_err_val)
#else
#  define VALID_GROUP(func, group, ret_err_val)			\
	if ((group < 0) ||					\
	    ((group >= next_group_id) && (group_zero_done ==1))	\
	   ) {							\
	    globus_libc_printf(func "(): bad group=%i\n",	\
			 group);				\
	    return (ret_err_val);				\
	}
#endif

typedef struct
{
    int		ep_id;
    int		size;
    int		rank;
    sp_list_t	sp_list;
} sp_group_t;

static	sp_group_t	sp_group[MAX_N_GROUPS];
static	int		next_group_id;
static	int		group_zero_done = 0;

static	sp_list_t	*gather_lists;

static	struct
{
    globus_mutex_t	mutex;
    globus_cond_t	cond;
    volatile int	done;
} monitor;
	    
static int
unique_group_id()
{
    int		g;

    g = next_group_id++;
    if (g >= MAX_N_GROUPS) {
	globus_libc_printf("ran out of sp_groups!\n");
	next_group_id--;
	return (-1);
    }
    return (g);
}

int
sp_group_startup()
{
    int		i;

    HelloSpg(globus_libc_printf("sp_group_startup(): hello\n"));
    next_group_id = 0;
    for (i = 0; i < MAX_N_GROUPS; i++)
    {
	sp_group[i].ep_id = -1;
	sp_group[i].size  = -1;
	sp_group[i].rank  = -1;
	if (sp_list_init(&(sp_group[i].sp_list)) != 0) {
	    globus_libc_printf("sp_group_init(): "
			 "sp_list_init() failed\n");
	    return (1);
	}
    }
    gather_lists = (sp_list_t *) NULL;
    if (globus_cond_init(&(monitor.cond), (globus_condattr_t *) NULL)
	!= GLOBUS_SUCCESS) {
	globus_libc_printf("sp_group_startup(): cond_init failed\n");
	return (1);
    }
    if (globus_mutex_init(&(monitor.mutex), (globus_mutexattr_t *)
			 NULL) != GLOBUS_SUCCESS) {
	globus_libc_printf("sp_group_startup(): mutex_init failed\n");
	return (1);
    }
    monitor.done = 0;
    return (0);
}

void
sp_group_shutdown(int aborted)
{
    int		i;

    HelloSpg(globus_libc_printf("sp_group_shutdown(): hello\n"));
    for (i = 0; i < MAX_N_GROUPS; i++)
	if (sp_group[i].ep_id != -1) {
	    if (aborted == GLOBUS_FALSE)
		globus_libc_printf("warning: sp_group_shutdown(): "
			     "sp_group[%i].epid=%i\n",
			     i, sp_group[i].ep_id);
	    if ((sp_list_done(&(sp_group[i].sp_list)) != 0)
		&& (aborted == GLOBUS_FALSE))
		globus_libc_printf("sp_group_shutdown(): sp_list error\n");
	}
    globus_cond_destroy(&(monitor.cond));
    globus_mutex_destroy(&(monitor.mutex));
}

int
sp_group_size(int group_id)
{
    VALID_GROUP("sp_group_size", group_id, -1);
    return (sp_group[group_id].size);
}

int
sp_group_rank(int group_id)
{
    VALID_GROUP("sp_group_rank", group_id, -1);
    return (sp_group[group_id].rank);
}

globus_nexus_startpoint_t *
sp_group_sp(int group_id, int sp_num)
{
    VALID_GROUP("sp_group_sp", group_id, NULL);
    return (sp_list_get_sp(&(sp_group[group_id].sp_list), sp_num));
}

int
sp_group_bootstrap(int group_id, int ep_id,
		   int size, int rank,
		   sp_list_t *sp_list)
{
    if (group_id != HARNESS_GROUP_ID) {
	globus_libc_printf("sp_group_bootstrap(): bad group=%i\n", group_id);
	return (1);
    }
    if (ep_id != HARNESS_ENDPOINT_ID) {
	globus_libc_printf("sp_group_bootstrap(): bad ep_id=%i\n", ep_id);
	return (1);
    }
    sp_group[group_id].ep_id = ep_id;
    sp_group[group_id].size = size;
    sp_group[group_id].rank = rank;
    if (sp_list_append_list(&(sp_group[group_id].sp_list),
			    sp_list) != 0) {
	globus_libc_printf("sp_group_bootstrap(): list append failed\n");
	return (1);
    }
    TraceSpg(globus_libc_printf("sp_group_bootstrap(): \n"
			  "\tsize=%i\n"
			  "\trank=%i\n",
			  size,
			  rank));
    return (0);
}
    
int
sp_group_create(int master, int ep_id, int *who)
{
    int		group, size;

    if (who != NULL) {
	globus_libc_printf("sp_group_create(): who != NULL\n");
	harness_abort();
	return (-1); /* don't get here */
    } else
	size = sp_group[HARNESS_GROUP_ID].size;
    group = unique_group_id();
    DebugSpg(globus_libc_printf("sp_group_create(): creating group %i\n",
			  group));
    if (sp_group_rank(HARNESS_GROUP_ID) == master) {
	globus_nexus_startpoint_t	*tmp;
	globus_nexus_buffer_t		buf;
	int			rank;
	int			x, y;

	/* Make room for replies */
	DebugSpg(globus_libc_printf("sp_group_create(): preparing\n"));
	gather_lists = (sp_list_t *)
	    globus_malloc(size * sizeof(sp_list_t));
	if (gather_lists == NULL) {
	    globus_libc_printf("sp_group_create(): malloc failed\n");
	    harness_abort();
	}
	for (x = 0; x < size; x++) {
	    if (sp_list_init(&(gather_lists[x])) != 0) {
		globus_libc_printf("sp_group_create(): "
			     "sp_list_init() failed\n");
		harness_abort();
		return (-1); /* not reached */
	    }
	}
	if (group == HARNESS_GROUP_ID)
	    monitor.done = size - 1;
	else
	    monitor.done = size;
	/* Send out requests */
	DebugSpg(globus_libc_printf("sp_group_create(): sending requests\n"));
	for (rank = (group == HARNESS_GROUP_ID ? 1 : 0);
	     rank < size; rank++)
	{
	    if (globus_nexus_buffer_init(&buf,
					 5 * globus_nexus_sizeof_int(1),
					 0)
		!= GLOBUS_SUCCESS) {
		globus_libc_printf("sp_group_create(): buf_init failed\n");
		harness_abort();
		return (-1); /* not reached */
	    }
	    globus_nexus_put_int(&buf, &master, 1);
	    globus_nexus_put_int(&buf, &ep_id, 1);
	    globus_nexus_put_int(&buf, &group, 1);
	    globus_nexus_put_int(&buf, &rank, 1);
	    globus_nexus_put_int(&buf, &size, 1);
	    TraceSpg(globus_libc_printf("sp_group_create(): request to %i\n",
				  rank));
	    if (globus_nexus_send_rsr(&buf,
			       sp_group_sp(HARNESS_GROUP_ID, rank),
			       REQUEST_SP_COPIES_HANDLER_ID,
			       GLOBUS_TRUE,	/* destroy buf  */
			       GLOBUS_FALSE)	/* non-threaded */
		!= GLOBUS_SUCCESS) {
		globus_libc_printf("sp_group_create(): send_rsr failed\n");
		/* harness_abort();
		return (-1);*/ /* not reached */
	    }
	}
	globus_mutex_lock(&(monitor.mutex));
	/* Wait for replies */
	TraceSpg(globus_libc_printf("sp_group_create(): wait to collect\n"));
	while (monitor.done != 0)
	    globus_cond_wait(&(monitor.cond),
			    &(monitor.mutex));
	if (group != HARNESS_GROUP_ID)
	    monitor.done = 1; /* will send to self later */
	globus_mutex_unlock(&(monitor.mutex));
	/* Translate for resend */
	DebugSpg(globus_libc_printf("sp_group_create(): translating\n"));
	if (group != HARNESS_GROUP_ID) {
	    /*
	     * Normal case: size x size matrix
	     */
	    DebugSpg(globus_libc_printf("normal translate, size=%i\n", size));
	    for (y = 1; y < size; y++)
		for (x = 0; x < y; x++) {
		    tmp = sp_list_get_sp(&(gather_lists[y]), x);
		    sp_list_set_sp(&(gather_lists[y]), x,
				   sp_list_get_sp(&(gather_lists[x]),
						  y));
		    sp_list_set_sp(&(gather_lists[x]), y, tmp);
		}
	} else {
	    /*
	     * Special case:
	     * gather_lists[0] remains empty.
	     * lists[1..size-1] are of length size-1.
	     */
	    int	x2, y2;

	    DebugSpg(globus_libc_printf("translating harness_group\n"));
	    for (y = 2; y < size; y++)
		for (x = 0; x < (y - 1); x++) {
		    x2 = (y + 0) - 1;
		    y2 = (x + 1) - 0;
		    tmp = sp_list_get_sp(&(gather_lists[y]), x);
		    sp_list_set_sp(&(gather_lists[y]), x,
				   sp_list_get_sp(&(gather_lists[y2]),
						  x2));
		    sp_list_set_sp(&(gather_lists[y2]), x2, tmp);
		}
	}
	DebugSpg(globus_libc_printf("sp_group_create(): translated\n"));
	/* Send them out to others */
	DebugSpg(globus_libc_printf("sp_group_create(): resending\n"));
	if (size > 1)
	    if (sp_list_buffer_size(&(gather_lists[1]), &y) != 0) {
		globus_libc_printf("sp_group_create(): "
			     "sp_list_buf_siz() failed\n");
		harness_abort();
		return (-1); /* not reached */
	    }
	y += globus_nexus_sizeof_int(1);
	for (x = (group == HARNESS_GROUP_ID ? 1 : 0); x < size; x++)
	{
	    if (globus_nexus_buffer_init(&buf, y, 0) != GLOBUS_SUCCESS) {
		globus_libc_printf("sp_group_create(): "
			     "buffer_init() failed\n");
		harness_abort();
		return (-1); /* not reached */
	    }
	    globus_nexus_put_int(&buf, &group, 1);
	    if (sp_list_buffer_put(&(gather_lists[x]), &buf) != 0) {
		globus_libc_printf("sp_group_create(): sp_buf_put failed\n");
		harness_abort();
		return (-1); /* not reached */
	    }
	    TraceSpg(globus_libc_printf("sp_group_create(): scatter to %i\n",
				  x));
	    if (globus_nexus_send_rsr(&buf,
			       sp_group_sp(HARNESS_GROUP_ID, x),
			       SCATTER_SP_COPIES_HANDLER_ID,
			       GLOBUS_TRUE,
			       GLOBUS_FALSE) != GLOBUS_SUCCESS) {
		globus_libc_printf("sp_group_create(): send_rsr failed\n");
		harness_abort();
		return (-1);
	    }
	}
	/* Cleanup */
	DebugSpg(globus_libc_printf("sp_group_create(): cleaning up\n"));
	for (x = 0; x < size; x++)
	    sp_list_done(&gather_lists[x]);
	globus_free(gather_lists);
	gather_lists = (sp_list_t *) NULL;
	/* Wait? */
	if (group != HARNESS_GROUP_ID) {
	    TraceSpg(globus_libc_printf("sp_group_create(): master zzz..\n"));
	    globus_mutex_lock(&(monitor.mutex));
	    while (monitor.done != 0)
		globus_cond_wait(&(monitor.cond),
				&(monitor.mutex));
	    globus_mutex_unlock(&(monitor.mutex));
	}
    } else { /* slave */
	TraceSpg(globus_libc_printf("sp_group_create(): slave zzzz..\n"));
	globus_mutex_lock(&(monitor.mutex));
	/* passive -> may have already received, so += */
	monitor.done ++;
	while (monitor.done != 0)
	    globus_cond_wait(&(monitor.cond),
			    &(monitor.mutex));
	globus_mutex_unlock(&(monitor.mutex));
    }
    TraceSpg(globus_libc_printf("sp_group_create(): created group %i\n",
			  group));
    return (group);
}

int
sp_group_delete(int gr_id)
{
    int		r = 0;
    
    VALID_GROUP("sp_group_delete", gr_id, 1);
    if (sp_list_destroy_sps(
	&(sp_group[gr_id].sp_list), 0,
	sp_list_get_size(&(sp_group[gr_id].sp_list)))
	!= 0) {
	globus_libc_printf("sp_group_delete(): destroy_sps failed\n");
	r = 1;
    }
    if (sp_list_done(&(sp_group[gr_id].sp_list)) != 0) {
	globus_libc_printf("sp_group_delete(): list_done failed\n");
	r = 1;
    }
    sp_group[gr_id].ep_id = -1;
    sp_group[gr_id].size  = -1;
    sp_group[gr_id].rank  = -1;
    return (r);
}

globus_nexus_startpoint_t **
sp_group_list(int gr_id)
{
    return (sp_group[gr_id].sp_list.sp);
}

/*
 *
 * HANDLERS
 *
 */

void
request_sp_copies_handler(globus_nexus_endpoint_t *endpoint,
			  globus_nexus_buffer_t *buffer,
			  globus_bool_t is_non_threaded_handler)
{
    globus_nexus_buffer_t	reply_buf;
    sp_list_t		sp_list;
    int		reply_to;
    int		ep_id;
    int		group_id;
    int		rank;
    int		n_copies;
    int		i, r, s;

    globus_nexus_get_int(buffer, &reply_to, 1);
    globus_nexus_get_int(buffer, &ep_id, 1);
    globus_nexus_get_int(buffer, &group_id, 1);
    globus_nexus_get_int(buffer, &rank, 1);
    globus_nexus_get_int(buffer, &n_copies, 1);
    DebugSpg(globus_libc_printf("request_sp_copies_handler(): \n"
			  "\treply_to = %i\n"
			  "\tep_id    = %i\n"
			  "\tgroup_id = %i\n"
			  "\trank     = %i\n"
			  "\tn_copies = %i\n",
			  reply_to, ep_id, group_id, rank, n_copies));
    sp_group[group_id].size = n_copies;
    sp_group[group_id].rank = rank;
    if (group_id == HARNESS_GROUP_ID)
	n_copies -= 1; /* master already has others */
    r = sp_list_init(&sp_list);
    if (r != 0) {
	globus_libc_printf("request_sp_copies_handler(): sp_list_init "
		     "failed\n");
	return;
    }
    r = sp_list_set_size(&sp_list, n_copies);
    if (r != 0) {
	globus_libc_printf("request_sp_copies_handler(): sp_list_set_size "
		     "failed\n");
	return;
    }
    r = sp_list_alloc_sps(&sp_list, 0, n_copies);
    if (r != 0) {
	globus_libc_printf("request_sp_copies_handler(): create_sps "
		     "failed\n");
	return;
    }
    for (i = 0; i < n_copies; i++) {
	r = globus_nexus_startpoint_bind(sp_list_get_sp(&sp_list, i),
				  ep_list_get_ep(ep_id));
	if (r != GLOBUS_SUCCESS) {
	    globus_libc_printf("request_sp_copies_handler(): "
			 "sp_bind failed\n");
	    return;
	}
    }
    r = sp_list_buffer_size(&sp_list, &s);
    if (r != 0) {
	globus_libc_printf("request_sp_copies_handler(): "
		     "buffer_size failed\n");
	return;
    }
    r = globus_nexus_buffer_init(&reply_buf, s, 0);
    if (r != GLOBUS_SUCCESS) {
	globus_libc_printf("request_sp_copies_handler(): buf_init failed\n");
	return;
    }
    globus_nexus_put_int(&reply_buf, &rank, 1);
    globus_nexus_put_int(&reply_buf, &group_id, 1);
    sp_list_buffer_put(&sp_list, &reply_buf);
    TraceSpg(globus_libc_printf("request_sp_copies_handler(): "
			  "sending rsr\n"));
    r = globus_nexus_send_rsr(&reply_buf,
		       sp_group_sp(HARNESS_GROUP_ID, reply_to),
		       COLLECT_SP_COPIES_HANDLER_ID,
		       GLOBUS_TRUE,
		       is_non_threaded_handler);
    if (r == GLOBUS_SUCCESS) {
	TraceSpg(globus_libc_printf("request_sp_copies_handler(): "
			      "sent rsr OK\n"));
    } else
	globus_libc_printf("request_sp_copies_handler(): send rsr failed\n");
}

void
collect_sp_copies_handler(globus_nexus_endpoint_t *endpoint,
			  globus_nexus_buffer_t *buffer,
			  globus_bool_t is_non_threaded_handler)
{
    int		sender_rank;
    int		group_id;
    sp_list_t	sp_list;
    int		r;

    r = sp_list_init(&sp_list);
    if (r != 0) {
	globus_libc_printf("collect_sp_copies_handler(): "
		     "sp_list init failed\n");
	return;
    }
    globus_nexus_get_int(buffer, &sender_rank, 1);
    globus_nexus_get_int(buffer, &group_id, 1);
    sp_list_buffer_get(&sp_list, buffer);
    DebugSpg(globus_libc_printf("collect_sp_copies_handler(): \n"
			  "\tsender   = %i\n"
			  "\tgroup_id = %i\n",
			  sender_rank, group_id));
    globus_mutex_lock(&(monitor.mutex));
    sp_list_append_list(&(gather_lists[sender_rank]), &sp_list);
    monitor.done--;
    DebugSpg(globus_libc_printf("collect_sp_copies_handler(): done = %i\n",
			  monitor.done));
    if (monitor.done == 0) {
	DebugSpg(globus_libc_printf("collect_sp_copies_handler(): "
			      "signalling\n"));
	globus_cond_signal(&(monitor.cond));
    }
    globus_mutex_unlock(&(monitor.mutex));
    TraceSpg(globus_libc_printf("collect_sp_copies_handler(): rcvd from %i\n",
			  sender_rank));
}

void scatter_sp_copies_handler(globus_nexus_endpoint_t *endpoint,
			       globus_nexus_buffer_t *buffer,
			       globus_bool_t is_non_threaded_handler)
{
    sp_list_t	sp_list;
    int		group_id;

    globus_nexus_get_int(buffer, &group_id, 1);
    DebugSpg(globus_libc_printf("scatter_sp_copies_handler(): got group %i\n",
			  group_id));
    if (sp_list_init(&sp_list) != 0) {
	globus_libc_printf("scatter_sp_copies_handler(): listinit failed\n");
	return;
    }
    globus_mutex_lock(&(monitor.mutex));
    sp_list_buffer_get(&sp_list, buffer);
    if (sp_list_append_list(&(sp_group[group_id].sp_list), &sp_list)
	!= 0)
	globus_libc_printf("scatter_sp_copies_handler(): append failed\n");
    monitor.done--;
    DebugSpg(globus_libc_printf("scatter_sp_copies_handler(): done = %i\n",
			  monitor.done));
    if (monitor.done == 0) {
	DebugSpg(globus_libc_printf("scatter_sp_copies_handler(): "
			      "signalling\n"));
	globus_cond_signal(&(monitor.cond));
    }
    globus_mutex_unlock(&(monitor.mutex));
    TraceSpg(globus_libc_printf("scatter_sp_copies_handler(): rcvd OK\n"));
}
