/*
 * harness_endpoint.c
 */

#include "globus_nexus.h"
#include "harness_endpoint.h"
#include "ep_list.h"
#include "sp_group.h"
#include "string.h"
#include "startup.h"

static globus_nexus_handler_t handler_table[] =
{
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	attach_startpoints_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	startup_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	shutdown_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	request_sp_copies_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	collect_sp_copies_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	scatter_sp_copies_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	ping_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	barrier_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	option_string_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,	set_debug_level_handler}
};
#define handler_table_size	(sizeof(handler_table) / \
				 sizeof(globus_nexus_handler_t))
static globus_nexus_endpoint_t harness_endpoint;

static struct
{
    globus_mutex_t	mutex;
    globus_cond_t	cond;
    int			done;
} monitor;

static int	next_ping_id = 1;

static int
unique_ping_id()
{
    int	x;

    x = next_ping_id++;
    x += 100000 * (sp_group_rank(HARNESS_GROUP_ID) + 1);
    return (x);
}

static void
cleanup_func(globus_nexus_endpoint_t *ep)
{
    globus_mutex_destroy(&(monitor.mutex));
    globus_cond_destroy(&(monitor.cond));
}

int
harness_endpoint_init()
{
    globus_nexus_endpointattr_t	ep_attr;

    if (globus_nexus_endpointattr_init(&ep_attr) != GLOBUS_SUCCESS) {
	globus_libc_printf("harness_endpoint_init(): attr_init failed\n");
	return (1);
    }
    if (globus_nexus_endpointattr_set_handler_table(&ep_attr,
						    handler_table,
						    handler_table_size)
	!= GLOBUS_SUCCESS) {
	globus_libc_printf("harness_endpoint_init(): set_table failed\n");
	return (1);
    }
    if (globus_nexus_endpoint_init(&harness_endpoint, &ep_attr)
	!= GLOBUS_SUCCESS) {
	globus_libc_printf("harness_endpoint_init(): ep_init failed\n");
	return (1);
    }
    globus_nexus_endpointattr_destroy(&ep_attr);
    if (ep_list_new_ep(&harness_endpoint,
		       GLOBUS_FALSE,
		       cleanup_func) != 0) {
	globus_libc_printf("harness_endpoint_init(): new_ep failed\n");
	return (1);
    }
    /* borrowing the test debug for just this one */
    TraceTst(globus_libc_printf("harness_endpoint_init(): success\n"));
    return (0);
}

void
harness_send_ping(int dest, int trips,
		  globus_bool_t is_non_threaded_handler)
{
    globus_nexus_buffer_t	buf;
    int				src, pid;

    if (trips > 0) {
	trips = trips * 2 - 1;	/* round trips */
	monitor.done = 1;
    } else
	trips = -1;	/* one way */
    src = sp_group_rank(HARNESS_GROUP_ID);
    pid = unique_ping_id();
    globus_nexus_buffer_init(&buf, 4 * nexus_sizeof_int(1), 0);
    globus_nexus_put_int(&buf, &src, 1);
    globus_nexus_put_int(&buf, &dest, 1);
    globus_nexus_put_int(&buf, &pid, 1);
    globus_nexus_put_int(&buf, &trips, 1);
    globus_libc_printf("Initiating ping: src=%i,pid=%i,dst=%i\n",
		 src, pid, dest);
    globus_nexus_send_rsr(&buf,
			  sp_group_sp(HARNESS_GROUP_ID, dest),
			  PING_HANDLER_ID,
			  GLOBUS_TRUE,
			  is_non_threaded_handler);
    if (trips > 0) {
	globus_mutex_lock(&(monitor.mutex));
	while (monitor.done != 0)
	    globus_cond_wait(&(monitor.cond),
			    &(monitor.mutex));
	globus_mutex_unlock(&(monitor.mutex));
	globus_libc_printf("ping %i complete\n", pid);
    }
}

void
harness_ping_test(int n_trips, int n_rounds)
{
    int		i, j;

    for (i = 0; i < n_rounds; i++)
	for (j = 0; j < sp_group_size(HARNESS_GROUP_ID); j++)
	    harness_send_ping(j, n_trips, GLOBUS_FALSE);
}

/*
 *
 * handlers
 *
 */

void
ping_handler(globus_nexus_endpoint_t *endpoint,
	     globus_nexus_buffer_t *buffer,
	     globus_bool_t is_non_threaded_handler)
{
    globus_nexus_buffer_t	reply_buf;
    int				src, dst, pid, ttl, x;

    globus_nexus_get_int(buffer, &src, 1);
    globus_nexus_get_int(buffer, &dst, 1);
    globus_nexus_get_int(buffer, &pid, 1);
    globus_nexus_get_int(buffer, &ttl, 1);
    globus_libc_printf("ping_send_handler(): dst=%i,pid=%i,src=%i,ttl=%i\n",
		 dst, pid, src, ttl);
    x = sp_group_rank(HARNESS_GROUP_ID);
    if (dst != x)
	globus_libc_printf("ping_send_handler(): I am %i, not %i\n",
		     x, dst);
    
    if (ttl == -1) {
	/* one way, let it die */
	return;
    } else if (ttl == 0) {
	/* end of trip, signal */
	globus_mutex_lock(&(monitor.mutex));
	monitor.done = 0;
	globus_cond_signal(&(monitor.cond));
	globus_mutex_unlock(&(monitor.mutex));
    } else {
	/* send reply */
	ttl--;
	globus_nexus_buffer_init(&reply_buf, 4 * nexus_sizeof_int(1), 0);
	globus_nexus_put_int(&reply_buf, &dst, 1); /* new src */
	globus_nexus_put_int(&reply_buf, &src, 1); /* new dst */
	globus_nexus_put_int(&reply_buf, &pid, 1);
	globus_nexus_put_int(&reply_buf, &ttl, 1);
	globus_nexus_send_rsr(&reply_buf,
			      sp_group_sp(HARNESS_GROUP_ID, src),
			      PING_HANDLER_ID,
			      GLOBUS_TRUE,
			      is_non_threaded_handler);
    }
}
