/*
 * generic_test.c
 */

/*
 *
 * options
 *
 */

#ifdef PrintHooks
#  define PrintEnterHook(func) \
	globus_libc_printf("-- Test hook, enter " func "() --\n")
#  define PrintLeaveHook(func) \
	globus_libc_printf("-- Test hook, leave " func "() --\n")
#else
#  define PrintEnterHook(func)
#  define PrintLeaveHook(func)
#endif

/*
 *
 * constants
 *
 */

#define PATTERN_ONE_TO_ALL		1
#define PATTERN_ALL_TO_ONE		2
#define PATTERN_ALL_TO_ALL		3
#define PATTERN_RANDOM			4

/*
 *
 * variables
 *
 */

static globus_nexus_handler_t	test_handler_table[] =
{
    Generic_HandlerTable
};

static globus_nexus_endpoint_t		test_endpoint;
static globus_bool_t			reliable_proto;
static int				the_debug_level;

/*
 *
 * functions
 *
 */

void
test_usage_string()
{
    printf("    Endpoint options:\n"
	   "\tproto=<type>\t\ttcp, udp (default)\n"
	   "    Test options:\n"
	   "\tpattern=<p>\t\tall_to_all, (one_to_all), "
	   "all_to_one, random\n"
	   "\tcount=<n>\t\t(1)\n"
	   "\trounds=<n>\t\t(1)\n"
	   "\trepeat=<n>\t\t(1)\n"
	   "\tself=<yes,no>\t\tallow send to self (yes)\n");
    Generic_UsageString();
}

int
test_startup(int dbg_lvl)
{
    PrintEnterHook("test_startup");
    the_debug_level = (dbg_lvl == -1 ? 1 : dbg_lvl);
    Generic_Startup();
    PrintLeaveHook("test_startup");
    return (0);
}

void
parse_option(arg_list_t *al,
	     char *tag_str,
	     char *fmt_str,
	     char *def_val,
	     char *dst_var)
{
    static char	buf[100];
    char	*s;
    s = arg_list_find(al, tag_str, def_val);
    if (sscanf(s, fmt_str, dst_var) != 1) {
	globus_libc_printf("parse_option(): bad format in "
		     "%s='%s'\n", tag_str, s);
	harness_abort();
    }
}

#define MAX_OPT_BUF_LEN		199

static char opt_buf[MAX_OPT_BUF_LEN + 1];
static int opt_buf_pos;

void
clear_opt_buf()
{
    int		i;

    for (i = 0; i < MAX_OPT_BUF_LEN; i++)
	opt_buf[i] = ' ';
    opt_buf_pos = 0;
}

char *
copy_opt_buf()
{
    char	*s;
    int		i;

    opt_buf[opt_buf_pos] = '\0';
    opt_buf_pos++;
    s = (char *) globus_malloc(opt_buf_pos * sizeof(char));
    if (s == NULL) {
	globus_libc_printf("copy_opt_buf(): malloc failed\n");
	harness_abort();
    }
    for (i = 0; i < opt_buf_pos; i++)
	s[i] = opt_buf[i];
    return (s);
}

void
add_option_str(char *str)
{
    int		i;

    /* leave a space between params */
    if (opt_buf_pos > 0)
    {
	opt_buf[opt_buf_pos] = ' ';
	opt_buf_pos++;
    }

    i = 0;
    while (str[i] != '\0')
    {
	if (opt_buf_pos == MAX_OPT_BUF_LEN)
	{
	    globus_libc_printf("add_option_str(): buffer overrun\n");
	    harness_abort();
	}
	opt_buf[opt_buf_pos] = str[i];
	opt_buf_pos++;
	i++;
    }
}

int
test_parse_options(arg_list_t *options,
		   int n_tests, arg_list_t *runtest,
		   char **parsed_options)
{
    char	tmp_buf[30];
    int		i;
    int		proto, pattern, self, master, count, rounds, repeat;
    char	*s;

    PrintEnterHook("test_parse_options");
    clear_opt_buf();
    /*
     * -ep options:
     *	proto=udp,tcp,(default)
     */
    s = arg_list_find(options, "proto", "default");
    if (strcmp(s, "tcp") == 0)
	proto = GLOBUS_NEXUS_PROTO_TYPE_TCP;
    else if (strcmp(s, "udp") == 0)
	proto = GLOBUS_NEXUS_PROTO_TYPE_UDP;
    else if (strcmp(s, "default") == 0)
	proto = -1;
    else {
	globus_libc_printf("bad option proto='%s'\n", s);
	return (1);
    }
    sprintf(tmp_buf, "proto=%i", proto);
    add_option_str(tmp_buf);
    parsed_options[0] = copy_opt_buf();
    /*
     * -run options
     */
    for (i = 0; i < n_tests; i++) {
	clear_opt_buf();
	/*
	 * pattern=
	 *	(one_to_all)
	 *	all_to_one
	 *	all_to_all
	 *	random
	 */
	s = arg_list_find(&(runtest[i]), "pattern", "one_to_all");
	if (strcmp(s, "one_to_all") == 0)
	    pattern = PATTERN_ONE_TO_ALL;
	else if (strcmp(s, "all_to_one") == 0)
	    pattern = PATTERN_ALL_TO_ONE;
	else if (strcmp(s, "all_to_all") == 0)
	    pattern = PATTERN_ALL_TO_ALL;
	else if (strcmp(s, "random") == 0)
	    pattern = PATTERN_RANDOM;
	else {
	    globus_libc_printf("bad option pattern='%s'\n", s);
	    harness_abort();
	}
	sprintf(tmp_buf, "pattern=%i", pattern);
	add_option_str(tmp_buf);
	/*
	 * master=<n>
	 */
	parse_option(&(runtest[i]), "master", "%i",
		     "0", (char *) &master);
	if (master < 0) master = 0;
	sprintf(tmp_buf, "master=%i", master);
	add_option_str(tmp_buf);
	/*
	 * self=(yes)/no
	 */
	s = arg_list_find(&(runtest[i]), "self", "default");
	if (strcmp(s, "yes") == 0)
	    self = 1;
	else if (strcmp(s, "no") == 0)
	    self = 0;
	else if (strcmp(s, "default") == 0)
	    self = 1;
	else {
	    globus_libc_printf("bad option self='%s'\n", s);
	    harness_abort();
	}
	sprintf(tmp_buf, "self=%i", self);
	add_option_str(tmp_buf);
	/* count (1) */
	parse_option(&(runtest[i]), "count", "%i",
		     "1", (char *) &count);
	if (count < 1) count = 1;
	sprintf(tmp_buf, "cnt=%i", count);
	add_option_str(tmp_buf);
	/* rounds (1) */
	parse_option(&(runtest[i]), "rounds", "%i",
		     "1", (char *) &rounds);
	if (rounds < 1) rounds = 1;
	sprintf(tmp_buf, "rnd=%i", rounds);
	add_option_str(tmp_buf);
	/* repeat (1) */
	parse_option(&(runtest[i]), "repeat", "%i",
		     "1", (char *) &repeat);
	if (repeat < 1) repeat = 1;
	sprintf(tmp_buf, "rep=%i", repeat);
	add_option_str(tmp_buf);
	/* end of generic options separator */
	add_option_str("$");
	/* test-specific options */
	if (Generic_ParseOptions(&(runtest[i])) != 0)
	    return (1);
	parsed_options[i + 1] = copy_opt_buf();
    }
    PrintLeaveHook("test_parse_options");
    return (0);
}

static void
cleanup_test_endpoint(globus_nexus_endpoint_t *ep)
{
    PrintEnterHook("cleanup_test_endpoint");
    Generic_CleanupEpFunc(ep);
    PrintLeaveHook("cleanup_test_endpoint");
}

int
test_create_endpoint(char *options,
		     globus_nexus_endpoint_t **ep,
		     void (**cleanup_func)(globus_nexus_endpoint_t *),
		     globus_bool_t *free_ep)
{
    globus_nexus_endpointattr_t	ep_attr;
    int				proto;

    PrintEnterHook("test_create_endpoint");
    if (sscanf(options, "proto=%i", &proto) != 1)
    {
	globus_libc_printf("test_create_endpoint(): options scan failed\n");
	return (1);
    }
    if (globus_nexus_endpointattr_init(&ep_attr) != GLOBUS_SUCCESS)
	return (1);
    if (globus_nexus_endpointattr_set_handler_table(&ep_attr,
					     test_handler_table,
					     sizeof(test_handler_table)
					     / sizeof(globus_nexus_handler_t))
	!= GLOBUS_SUCCESS)
	return (2);
    switch (proto)
    {
    case -1:
    case GLOBUS_NEXUS_PROTO_TYPE_TCP:
	reliable_proto = GLOBUS_TRUE;
	break;

    case GLOBUS_NEXUS_PROTO_TYPE_UDP:
	reliable_proto = GLOBUS_FALSE;
	break;

    default:
	globus_libc_printf("test_create_endpoint(): is %i a reliable "
		     "proto? (yes)\n", proto);
	reliable_proto = NEXUS_TRUE;
	break;
    }
    if (proto != -1)
    {
	if (globus_nexus_endpointattr_set_protocol(&ep_attr,
						   proto,
						   NULL,
						   0)
	    != GLOBUS_SUCCESS) {
	    globus_libc_printf("test_create_endpoint(): "
			 "set_proto(%i) failed\n", proto);
	    return (3);
	}
    }
    if (globus_nexus_endpoint_init(&test_endpoint, &ep_attr) != GLOBUS_SUCCESS)
	return (3);
    globus_nexus_endpointattr_destroy(&ep_attr);
    *ep = &test_endpoint;
    *cleanup_func = cleanup_test_endpoint;
    *free_ep = GLOBUS_FALSE;
    PrintEnterHook("test_create_endpoint");
    return (0);
}

void *
test_init(char *run_options,
	  int size, int rank,
	  globus_nexus_startpoint_t **sp)
{
    test_params_t	*tp;
    int			i;

    PrintEnterHook("test_init");
    /* allocate test params */
    tp = (test_params_t *) globus_malloc(sizeof(test_params_t));
    if (tp == NULL)
    {
	globus_libc_printf("test_init(): malloc failed\n");
	harness_abort();
    }
    /* fill in generic params */
    tp->endpoint = &test_endpoint;
    tp->sp_list = sp;
    tp->size = size;
    tp->rank = rank;
    tp->debug_level = the_debug_level;
    if (sscanf(run_options,
	       "pattern=%i master=%i self=%i cnt=%i rnd=%i rep=%i $",
	       &(tp->pattern), &(tp->master), &(tp->self),
	       &(tp->count), &(tp->rounds), &(tp->repeat)) != 6) {
	globus_libc_printf("test_init(): failed to scan generic params "
		     "'%s'\n", run_options);
	harness_abort();
    }
    /* skip generic params */
    for (i = 0; run_options[i] != '$'; i++) /* emtpy loop */;
    /* another space after the $ if the test has options */
    if (run_options[i + 1] == '\0')
	Generic_TestInit(&(run_options[i + 1]), tp);
    else
	Generic_TestInit(&(run_options[i + 2]), tp);
    PrintLeaveHook("test_init");
    return ((void *) tp);
}

void
test_run(void *test_arg)
{
    test_params_t	*tp;
    int			cnt, dst, rnd, rep;

    PrintEnterHook("test_run");
    tp = (test_params_t *) test_arg;
    switch (tp->pattern)
    {
    case PATTERN_ONE_TO_ALL:
	if (tp->debug_level == 1)
	    globus_libc_printf("RunTest(): starting %i reps\n", tp->repeat);
	for (rep = 1; rep <= tp->repeat; rep++)
	{
	    Generic_TestRun_BegRep(tp);
	    harness_barrier();

	    if (tp->rank == tp->master)
	    for (rnd = 1; rnd <= tp->rounds; rnd++)
	    {
		for (dst = 0; dst < tp->size; dst++)
		{
		    if ((dst != tp->master) || (tp->self == 1))
		    {
			if (tp->debug_level == 2)
			    globus_libc_printf("RunTest(): src=%i rep=%i "
					 "dst=%i: for cnt=%i\n",
					 tp->rank, rep, dst, tp->count);
			for (cnt = 1; cnt <= tp->count; cnt++)
			{
			    if (tp->debug_level == 3)
				globus_libc_printf("RunTest(): src=%i rep=%i "
					     "dst=%i cnt=%i:\n",
					     tp->rank, rep, dst, cnt);
			    Generic_TestRun_Once(tp, dst);
			    if (tp->debug_level == 3)
				globus_libc_printf("RunTest(): src=%i rep=%i "
					     "dst=%i cnt=%i: done\n",
					     tp->rank, rep, dst, cnt);
			}
			if (tp->debug_level == 2)
			    globus_libc_printf("RunTest(): src=%i rep=%i "
					 "dst=%i: done\n", tp->rank,
					 rep, dst);
		    }
		}
	    }
	    harness_barrier();
	    Generic_TestRun_EndRep(tp);
	    if (tp->debug_level == 1)
		globus_libc_printf("RunTest(): src=%i rep=%i: done\n",
			     tp->rank, rep);
	    harness_barrier();
	}
	break;

    case PATTERN_ALL_TO_ONE:
	dst = tp->master;
	if ((tp->debug_level == 1) || (tp->debug_level == 2))
	    globus_libc_printf("RunTest(): starting %i reps\n", tp->repeat);
 	for (rep = 1; rep <= tp->repeat; rep++)
	{
	    Generic_TestRun_BegRep(tp);
	    harness_barrier();
	    /* ignore rounds */
	    /* single dest */
	    if ((tp->rank != tp->master) || (tp->self == 1))
	    {
		if (tp->debug_level == 2)
		    globus_libc_printf("RunTest(): src=%i rep=%i dst=%i: "
				 "for count=%i\n", tp->rank, rep, dst,
				 tp->count);
		for (cnt = 1; cnt <= tp->count; cnt++)
		{
		    if (tp->debug_level == 3)
			globus_libc_printf("RunTest(): src=%i rep=%i dst=%i "
				     "cnt=%i:\n", tp->rank, rep, dst,
				     cnt);
		    Generic_TestRun_Once(tp, dst);
		    if (tp->debug_level == 3)
			globus_libc_printf("RunTest(): src=%i rep=%i dst=%i "
				     "cnt=%i: done\n", tp->rank, rep,
				     dst, cnt);
		}
		if (tp->debug_level == 2)
		    globus_libc_printf("RunTest(): src=%i rep=%i dst=%i: "
				       "done\n", tp->rank, rep, dst);
	    }
	    harness_barrier();
	    Generic_TestRun_EndRep(tp);
	    if ((tp->debug_level == 1) || (tp->debug_level == 2))
		globus_libc_printf("RunTest(): src=%i rep=%i: done\n",
			     tp->rank, rep);
	    harness_barrier();
	}
	break;

    case PATTERN_ALL_TO_ALL:
	if (tp->debug_level == 1)
	    globus_libc_printf("RunTest(): starting %i reps\n", tp->repeat);
	for (rep = 1; rep <= tp->repeat; rep++)
	{
	    Generic_TestRun_BegRep(tp);
	    harness_barrier();
	    for (rnd = 1; rnd <= tp->rounds; rnd++)
	    {
		for (dst = 0; dst < tp->size; dst++)
		{
		    if ((dst != tp->rank) || (tp->self != 0))
		    {
			if (tp->debug_level == 2)
			    globus_libc_printf("RunTest(): src=%i dst=%i "
					 "rnd=%i: for cnt=%i\n",
					 tp->rank, dst, rnd, tp->count);
			for (cnt = 1; cnt <= tp->count; cnt++)
			{
			    if (tp->debug_level == 3)
				globus_libc_printf("RunTest(): src=%i rep=%i "
					     "rnd=%i dst=%i cnt=%i:\n",
					     tp->rank, rep,
					     rnd, dst, cnt);
			    Generic_TestRun_Once(tp, dst);
			    if (tp->debug_level == 3)
				globus_libc_printf("RunTest(): src=%i rep=%i "
					     "rnd=%i dst=%i cnt=%i: "
					     "done\n", tp->rank, rep,
					     rnd, dst, cnt);
			}
		    }
		}
		if (tp->debug_level == 2)
		    globus_libc_printf("RunTest(): src=%i rep=%i rnd=%i: "
				 "done\n", tp->rank, rep, rnd);
	    }
	    if (tp->debug_level == 1)
		globus_libc_printf("RunTest(): src=%i rep=%i: done\n",
			     tp->rank, rep);
	    harness_barrier();
	    Generic_TestRun_EndRep(tp);
	    harness_barrier();
	}
	break;

    case PATTERN_RANDOM:
	globus_libc_printf("random pattern not implemented!\n");
	break;
    }
    PrintLeaveHook("test_run");
}

void
test_done(void *test_arg)
{
    PrintEnterHook("test_done");
    Generic_TestDone(test_arg);
    PrintLeaveHook("test_done");
}

void
test_cleanup()
{
    PrintEnterHook("test_cleanup");
    Generic_Cleanup();
    PrintLeaveHook("test_cleanup");
}
