/*
 * barrier.c
 */

#include "barrier.h"
#include "sp_group.h"
#include "startup.h"
#include "harness_options.h"
#include "harness_endpoint.h"

/*
 *
 * config
 *
 */

#define MAX_N_EVENTS	4

/*
 *
 * defines
 *
 */

#define STATE_DELETED		-1
#define STATE_NOTUSED		0
#define STATE_RUNNING		1
#define STATE_WAITING		2
#define STATE_SENDING		3
#define STATE_BLOCKED		4
#define STATE_LEAVING		5
#define STATE_PASSIVE		6

#define EVENT_NONE		0
#define EVENT_SIGNAL		1
#define EVENT_EXTERN		2

/*
 *
 * variables
 *
 */

typedef struct
{
    globus_mutex_t	mutex;
    int			state;
    int			stage_count;
    int			req_checkins;
    int			num_checkins;
    int			adv_checkins;
    int			group_id;
    int			wait_delay;
    /* int		*members; */
} l_barrier_t;

static l_barrier_t	l_barrier[MAX_N_BARRIERS];
static int		next_barrier_id;

typedef struct
{
    int		ev_type;
    int		ev_data;
} barrier_event_t;

static struct
{
    globus_mutex_t	mutex;
    globus_cond_t	cond;
    volatile int	q_size;
    barrier_event_t	the_q[MAX_N_EVENTS];
    int			q_take;
} event_q;

/*
 *
 * debug
 *
 */

#ifdef SKIP_CHECKS
#  define VALID_BARRIER(func, b_id, ret_err_val)
#else
#  define VALID_BARRIER(func, b_id, ret_err_val)		\
	if ((b_id < 0) || (b_id > next_barrier_id)) {		\
	    globus_libc_printf(func "(): illegal barrier id=%i\n",	\
			 b_id);					\
	    return (ret_err_val);				\
	}
#endif

/*
 *
 * functions
 *
 */

static int
unique_barrier_id()
{
    int	b = (next_barrier_id++);

    if (b >= MAX_N_BARRIERS) {
	globus_libc_printf("ran out of barriers!\n");
	next_barrier_id--;
	return (-1);
    }
    return (b);
}

static void
event_nq(int ev_type, int ev_data)
{
    int		p;

    globus_mutex_lock(&(event_q.mutex));
    if (event_q.q_size >= MAX_N_EVENTS) {
	globus_libc_printf("barrier event_nq(): overflow\n");
	harness_abort();
    }
    p = (event_q.q_take + event_q.q_size) % MAX_N_EVENTS;
    event_q.the_q[p].ev_type = ev_type;
    event_q.the_q[p].ev_data = ev_data;
    event_q.q_size++;
    if (event_q.q_size == 1)
	globus_cond_signal(&(event_q.cond));
    globus_mutex_unlock(&(event_q.mutex));
}

/*
 * Assumes caller has locked event_q mutex.
 */
static void
event_dq(int *ev_type, int *ev_data)
{
    if (event_q.q_size <= 0) {
	globus_libc_printf("event_dq(): q_size = %i\n", event_q.q_size);
	harness_abort();
	return;
    }
    *ev_type = event_q.the_q[event_q.q_take].ev_type;
    *ev_data = event_q.the_q[event_q.q_take].ev_data;
    event_q.q_size--;
    event_q.q_take = (event_q.q_take + 1) % MAX_N_EVENTS;
}

/*
 * Must be called by main thread.
 * No locked mutexes at entrance or exit.
 */
static void
event_loop(int barrier_id)
{
    int		ev_type, ev_data;

    globus_mutex_lock(&(event_q.mutex));
    while (event_q.q_size == 0)
	globus_cond_wait(&(event_q.cond), &(event_q.mutex));
    event_dq(&ev_type, &ev_data);
    switch (ev_type) {
    case EVENT_SIGNAL:
	if (ev_data != barrier_id) {
	    globus_libc_printf("event_loop(): expecting signal from "
			 "barrier %i, got %i\n",
			 barrier_id, ev_data);
	    globus_mutex_unlock(&(event_q.mutex));
	    harness_abort();
	    return;
	}
	break;
	
    case EVENT_EXTERN:
	if (barrier_id != -1) {
	    globus_libc_printf("event_loop(): unexpected external "
			 "event %i\n", ev_data);
	    globus_mutex_unlock(&(event_q.mutex));
	    harness_abort();
	}
    }
    globus_mutex_unlock(&(event_q.mutex));
}

int
barrier_startup()
{
    int		i;
    l_barrier_t	*b;

    HelloBar(globus_libc_printf("barrier_startup(): hello\n"));
    for (i = 0; i < MAX_N_BARRIERS; i++) {
	b = &(l_barrier[i]);
	if (globus_mutex_init(&(b->mutex), (globus_mutexattr_t *) NULL)
	    != GLOBUS_SUCCESS) {
	    globus_libc_printf("barrier_startup(): mutex_init[%i] failed\n",
			 i);
	    return (1);
	}
	b->state = STATE_NOTUSED;
	b->stage_count = 0;
	b->req_checkins = 654321; /* infinity or so */
	b->num_checkins = 0;
	b->adv_checkins = 0;
	b->group_id = -1;
	b->wait_delay = 0;
    }
    next_barrier_id = 0;
    if (globus_mutex_init(&(event_q.mutex), (globus_mutexattr_t *) NULL)
	!= GLOBUS_SUCCESS) {
	globus_libc_printf("barrier_startup(): mutex_init failed\n");
	return (1);
    }
    if (globus_cond_init(&(event_q.cond), (globus_condattr_t *) NULL)
	!= GLOBUS_SUCCESS) {
	globus_libc_printf("barrier_startup(): cond_init failed\n");
	return (1);
    }
    event_q.q_take = 0;
    event_q.q_size = 0;
    return (0);
}

void
barrier_shutdown(int aborted)
{
    int		i;

    HelloBar(globus_libc_printf("barrier_shutdown()\n"));
    for (i = 0; i < MAX_N_BARRIERS; i++)
	globus_mutex_destroy(&(l_barrier[i].mutex));
    globus_mutex_destroy(&(event_q.mutex));
    globus_cond_destroy(&(event_q.cond));
}

/*
 * Must already have barrier mutex.  Must not have
 * event_q mutex.  Checks count, adds an event, and
 * signals cond.
 */
static void
check_for_event(int barrier_id)
{
    l_barrier_t	*b;

    b = &(l_barrier[barrier_id]);
    DebugBar(globus_libc_printf("check_for_event(): id=%i,st=%i,sg=%i\n",
			  barrier_id, b->state, b->stage_count));
    DebugBar(globus_libc_printf("                 : num=%i,req=%i\n",
			  b->num_checkins, b->req_checkins));
    switch (b->state) {
    case STATE_DELETED:
	globus_libc_printf("check_for_event(): state=deleted\n");
	harness_abort();
	break;

    case STATE_NOTUSED:
    case STATE_RUNNING:
    case STATE_WAITING:
    case STATE_SENDING:
	/* no possible events */
	DebugBar(globus_libc_printf("                 : no pos events\n"));
	break;

    case STATE_BLOCKED:
	if (b->num_checkins == b->req_checkins) {
	    DebugBar(globus_libc_printf("\t\t: unblocking\n"));
	    b->num_checkins = b->adv_checkins;
	    b->adv_checkins = 0;
	    b->stage_count++;
	    event_nq(EVENT_SIGNAL, barrier_id);
	} else {
	    DebugBar(globus_libc_printf("\t\t: still blocked\n"));
	}
	break;

    case STATE_LEAVING:
	if (b->num_checkins == b->req_checkins) {
	    b->num_checkins = b->adv_checkins;
	    b->adv_checkins = 0;
	    if (b->num_checkins == 0) {
		b->state = STATE_PASSIVE;
	    }
	}
	break;

    case STATE_PASSIVE:
	globus_libc_printf("check_for_event(): state=passive ???\n");
	break;
    }
}

int
barrier_create(int group, int *who)
{
    l_barrier_t	*b;
    int		barrier_id;

    if (who != NULL) {
	globus_libc_printf("barrier_create(): who not implemented\n");
	return (1);
    }
    barrier_id = unique_barrier_id();
    DebugBar(globus_libc_printf("barrier_create(): creating %i\n",
			  barrier_id));
    b = &(l_barrier[barrier_id]);
    globus_mutex_lock(&(b->mutex));
    b->group_id = group;
    b->state = STATE_SENDING;
    b->req_checkins = sp_group_size(group);
    globus_mutex_unlock(&(b->mutex));
    barrier_block(HARNESS_BARRIER_ID);
    return (barrier_id);
}

int
barrier_delete(int barrier_id)
{
    VALID_BARRIER("barrier_delete", barrier_id, 1);
    globus_mutex_lock(&(l_barrier[barrier_id].mutex));
    l_barrier[barrier_id].state = STATE_DELETED;
    globus_mutex_unlock(&(l_barrier[barrier_id].mutex));
    return (0);
}

int
barrier_passive(int barrier_id)
{
    l_barrier_t	*b;

    VALID_BARRIER("barrier_passive", barrier_id, -1);
    b = &(l_barrier[barrier_id]);
    globus_mutex_lock(&(b->mutex));
    b->state = STATE_LEAVING;
    while (b->num_checkins != 0) {
	globus_mutex_unlock(&(b->mutex));
	barrier_block(barrier_id);
	globus_mutex_lock(&(b->mutex));
    }
    b->state = STATE_PASSIVE;
    globus_mutex_unlock(&(b->mutex));
    barrier_block(HARNESS_BARRIER_ID);
    return (0);
}

int
barrier_set_udelay(int barrier_id,
		   int udelay)
{
    VALID_BARRIER("barrier_set_udelay", barrier_id, -1);
    globus_mutex_lock(&(l_barrier[barrier_id].mutex));
    l_barrier[barrier_id].wait_delay = udelay;
    globus_mutex_unlock(&(l_barrier[barrier_id].mutex));
    return (0);
}

/*
 * May only be called by main thread.
 * Assumes caller has locked barrier mutex.
 * It is released while sending, but locked
 * again before returning.
 */
void
send_messages(int barrier_id)
{
    globus_nexus_buffer_t	buf;
    l_barrier_t			*b;
    int				i, j;
    int				my_rank,
				no_rank,
				my_stage_count,
				my_state;
    int				group_id;

    no_rank = -23;
    b = &(l_barrier[barrier_id]);
    DebugBar(globus_libc_printf("barrier send_messages(): id=%i,sg=%i\n",
			  barrier_id, b->stage_count));
    j = sp_group_size(b->group_id);
    my_rank = sp_group_rank(b->group_id);
    my_stage_count = b->stage_count;
    my_state = b->state;
    group_id = b->group_id;
    globus_mutex_unlock(&(b->mutex));
    for (i = 0; i < j; i++) {
	DebugBar(globus_libc_printf("send_messages(): to %i\n", i));
	if (globus_nexus_buffer_init(&buf, 5 * globus_nexus_sizeof_int(1), 0)
	    != GLOBUS_SUCCESS) {
	    globus_libc_printf("barrier send_messages(): "
			 "buffer_init failed\n");
	    harness_abort();
	    return;
	}

	/* barrier_id */
	globus_nexus_put_int(&buf, &barrier_id, 1);
	/* sender id */
	if (i == my_rank)
	    globus_nexus_put_int(&buf, &no_rank, 1);
	else
	    globus_nexus_put_int(&buf, &my_rank, 1);
	/* receiver_id */
	globus_nexus_put_int(&buf, &i, 1);
	/* stage */
	globus_nexus_put_int(&buf, &my_stage_count, 1);
	/* state */
	globus_nexus_put_int(&buf, &my_state, 1);

	if (sp_group_sp(group_id, i) == NULL)
	    globus_libc_printf("sp == NULL\n");
	else if
	    (globus_nexus_startpoint_is_null(sp_group_sp(group_id, i)))
	    globus_libc_printf("sp is null\n");
	else {
	    DebugBar(globus_libc_printf("sp is OK\n"));
	}
	TraceBar(globus_libc_printf("SEND BARRIER MSG %i -> %i\n",
			      my_rank, i));
	if (globus_nexus_send_rsr(&buf,
			   sp_group_sp(HARNESS_GROUP_ID, i),
			   BARRIER_HANDLER_ID,
			   GLOBUS_TRUE,
			   GLOBUS_FALSE) != GLOBUS_SUCCESS) {
	    globus_libc_printf("barrier send_msgs(): send_rsr failed\n");
	    harness_abort();
	    return;
	}
	DebugBar(globus_libc_printf("SEND MSG OK\n"));
    }
    globus_mutex_lock(&(b->mutex));
}

int
barrier_block(int barrier_id)
{
    l_barrier_t	*b;
    int		r;

    VALID_BARRIER("barrier_block", barrier_id, -1);
    b = &(l_barrier[barrier_id]);
    globus_mutex_lock(&(b->mutex));
    TraceBar(globus_libc_printf("barrier_block(): enter:bid=%i, stg=%i\n",
			  barrier_id, b->stage_count));
    b->state = STATE_SENDING;
    r = b->wait_delay;
    if (r > 0) {
	globus_mutex_unlock(&(b->mutex));
	while (r >= 10000) {
	    /* XXX -- Replace with gettimeofday() */
	    globus_libc_usleep(10000);
	    globus_poll();
	    r -= 10000;
	}
	globus_mutex_lock(&(b->mutex));
    }
    send_messages(barrier_id);
    b->state = STATE_BLOCKED;
    b->num_checkins++;
    check_for_event(barrier_id);
    TraceBar(globus_libc_printf("barrier_block(): waiting: bid=%i\n",
			  barrier_id));
    globus_mutex_unlock(&(b->mutex));
    event_loop(barrier_id);
    globus_mutex_lock(&(b->mutex));
    r = b->stage_count;
    TraceBar(globus_libc_printf("barrier_block(): leave: bid=%i, stg=%i\n",
			  barrier_id, b->stage_count));
    globus_mutex_unlock(&(b->mutex));
    return (r);
}

int
barrier_get_stage(int barrier_id)
{
    int		r;

    VALID_BARRIER("barrier_get_stage", barrier_id, -1);
    globus_mutex_lock(&(l_barrier[barrier_id].mutex));
    r = l_barrier[barrier_id].stage_count;
    globus_mutex_unlock(&(l_barrier[barrier_id].mutex));
    return (r);
}

int
barrier_validate(int barrier_id, int stage)
{
    l_barrier_t	*b;
    int		r;

    VALID_BARRIER("barrier_validate", barrier_id, BARRIER_STAGE_ERROR);
    b = &(l_barrier[barrier_id]);
    globus_mutex_lock(&(b->mutex));
    r = BARRIER_STAGE_ERROR;
    switch (b->state) {
    case STATE_RUNNING:
    case STATE_WAITING:
	/* have not sent yet, nobody could be ahead */
	if (stage == b->stage_count)
	    r = BARRIER_STAGE_CURRENT;
	else if (stage < b->stage_count)
	    r = BARRIER_STAGE_OLD;
	else
	    r = BARRIER_STAGE_ERROR;
	break;
	
    case STATE_SENDING:
    case STATE_BLOCKED:
	/* have sent, others may be running again */
	if (stage == b->stage_count)
	    r = BARRIER_STAGE_CURRENT;
	else if (stage == (b->stage_count + 1))
	    r = BARRIER_STAGE_EARLY;
	else if (stage < b->stage_count)
	    r = BARRIER_STAGE_OLD;
	else
	    r = BARRIER_STAGE_ERROR;
	break;

    case STATE_LEAVING:
    case STATE_PASSIVE:
	r = BARRIER_STAGE_UNKNOWN;
	break;
    }
    globus_mutex_unlock(&(b->mutex));
    if (r == BARRIER_STAGE_ERROR)
	globus_libc_printf("barrier_validate(): error\n");
    return (r);
}

/*
 *
 * handlers
 *
 */

void barrier_handler(globus_nexus_endpoint_t *endpoint,
		     globus_nexus_buffer_t *buffer,
		     globus_bool_t is_non_threaded_handler)
{
    l_barrier_t	*b;
    int		barrier_id;
    int		sender_id;
    int		receiver_id;
    int		stage_count;
    int		state;
    int		my_rank;

    globus_nexus_get_int(buffer, &barrier_id, 1);
    globus_nexus_get_int(buffer, &sender_id, 1);
    globus_nexus_get_int(buffer, &receiver_id, 1);
    globus_nexus_get_int(buffer, &stage_count, 1);
    globus_nexus_get_int(buffer, &state, 1);

    b = &(l_barrier[barrier_id]);
    globus_mutex_lock(&(b->mutex));
    my_rank = (b->group_id == -1 ? -1 : sp_group_rank(b->group_id));
    TraceBar(globus_libc_printf("RECV BARRIER MSG (%i)    %i <- %i\n",
			  my_rank, receiver_id, sender_id));
    DebugBar(globus_libc_printf("barrier_handler(%i): id=%i,src=%i,stg=%i\n",
			  my_rank,
			  barrier_id,
			  sender_id,
			  stage_count));
    if (sender_id == -23) {
	TraceBar(globus_libc_printf("barrier_handler(): "
			      "ignoring own msg\n"));
	globus_mutex_unlock(&(b->mutex));
	return;
    }
    switch (b->state) {
    case STATE_NOTUSED:
	b->num_checkins++;
	check_for_event(barrier_id);
	break;

    case STATE_RUNNING:
	if (stage_count == b->stage_count) {
	    b->num_checkins++;
	    check_for_event(barrier_id);
	} else
	    globus_libc_printf("barrier_handler(): received bad stage count"
			 "=%i\n", stage_count);
	break;

    case STATE_WAITING:
    case STATE_SENDING:
    case STATE_BLOCKED:
	if (stage_count == b->stage_count) {
	    if (state == STATE_PASSIVE)
		b->req_checkins--;
	    else
		b->num_checkins++;
	    check_for_event(barrier_id);
	} else if (stage_count == (b->stage_count + 1))
	    b->adv_checkins++;
	else
	    globus_libc_printf("barrier_handler(): received bad stage count"
			 "=%i\n", stage_count);
	break;

    case STATE_LEAVING:
	globus_libc_printf("barrier state=leaving???\n");
	harness_abort();
	if (stage_count == b->stage_count) {
	    b->num_checkins++;
	    check_for_event(barrier_id);
	    break;
	}
	/* else fall thru */
    case STATE_PASSIVE:
    {
	globus_nexus_buffer_t	buffer;
	int			i;

	/* XXX */
	if (globus_nexus_buffer_init(&buffer, 4 * globus_nexus_sizeof_int(1), 0)
	    != GLOBUS_SUCCESS) {
	    globus_libc_printf("barrier_handler(): buffer_init failed\n");
	    break;
	}
	globus_nexus_put_int(&buffer, &barrier_id, 1);
	i = sp_group_rank(b->group_id);
	globus_nexus_put_int(&buffer, &i, 1); /* sender_id */
	globus_nexus_put_int(&buffer, &stage_count, 1);
	i = STATE_PASSIVE;
	globus_nexus_put_int(&buffer, &i, 1);
	globus_mutex_unlock(&(b->mutex));
	if (globus_nexus_send_rsr(&buffer,
			   sp_group_sp(b->group_id, sender_id),
			   BARRIER_HANDLER_ID,
			   GLOBUS_TRUE,
			   is_non_threaded_handler) != GLOBUS_SUCCESS) {
	    globus_libc_printf("barrier_handler(): send reply failed\n");
	}
	return; /* already unlocked mutex, exit immediately */
    }

    case STATE_DELETED:
	globus_libc_printf("barrier_handler(): "
		     "received unexpected message: "
		     "\n\tbarrier_id   = %i"
		     "\n\tsender_id    = %i"
		     "\n\tstage_count  = %i"
		     "\n\tsender_state = %i"
		     "\n\tmy_state     = %i\n",
		     barrier_id, sender_id,
		     stage_count, state, b->state);
    }
    globus_mutex_unlock(&(b->mutex));
}
