/*
 * wrappers.c
 */

#include "wrappers.h"
#include "debug.h"
#include "handlers.h"

#include "everything.h"

/*
 * -- SendContextDestroyMsg() --
 *
 * Send a message to a startpoint telling it to destroy its context.
 */
int SendContextDestroyMsg(nexus_startpoint_t *startpoint)
{
    nexus_buffer_t	buffer;
    
    nexus_buffer_init(&buffer, 0, 0);
    nexus_send_rsr(&buffer, startpoint,
		   SHUTDOWN_CONTEXT_HANDLER_ID,
		   NEXUS_TRUE, NEXUS_FALSE);
    return(1);	
}
/* -- SendContextDestroyMsg() -- */

/*
 * -- InitTesting() --
 *
 * Set up master and slave nodes for testing.
 *	o make sure the slave_sp is valid
 *	o get the endpoint user pointer, make sure valid
 *	o try to get a node to be master
 *	o bind a startpoint to the master endpoint
 *	o put the master node_num and startpoint in a buffer
 *	o send the buffer, wait for reply
 *	o point the master to the slave
 */
int InitTesting(nexus_endpoint_t *master, nexus_startpoint_t *slave,
		test_node_t **nd, int is_non_threaded_handler)
{
    nexus_startpoint_t	startpoint;
    nexus_buffer_t	buffer;
    int			result;
    ep_data_t		*epd;
    int			node_num;

#ifdef TRC_InitTesting
    nexus_printf("InitTesting(): hello\n");
#endif

    if (nexus_startpoint_is_null(slave))
    {
	nexus_printf("InitTesting(): *slave_sp is null\n");
	return(0);
    }

    epd = (ep_data_t *)nexus_endpoint_get_user_pointer(master);
    if (epd == NULL)
    {
	nexus_printf("InitTesting(): user_pointer = NULL\n");
	return(0);
    }

    if (EPD_AllocNode(epd, NT_MASTER, &node_num) == 0)
    {
	nexus_printf("InitTesting(): AllocMasterNode failed\n");
	return(0);
    }
    *nd = EPD_GetNodePtr(epd, node_num);
    
    nexus_startpoint_bind(&startpoint, master);

#ifdef DBG_InitTesting
    nexus_printf("InitTesting(): startpoint bound\n");
#endif
    
    if (nexus_buffer_init(&buffer,
			  nexus_sizeof_int(1) +
			  nexus_sizeof_startpoint(&startpoint, 1), 0)
	!= NEXUS_SUCCESS)
    {
	nexus_printf("InitTesting(): buffer_init failed\n");
	return(0);
    }

    nexus_put_int(&buffer, &node_num, 1);
    nexus_put_startpoint_transfer(&buffer, &startpoint, 1);

#ifdef DBG_InitTesting
    nexus_printf("InitTesting(): sending rsr\n");
#endif

    TN_PreWait((*nd), CONNECT_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, slave,
		       CONNECT_HANDLER_ID,
		       NEXUS_TRUE, is_non_threaded_handler) != NEXUS_SUCCESS)
    {
	nexus_printf("InitTesting(): send_rsr failed\n");
	return(0);
    }
#ifdef DBG_InitTesting
    nexus_printf("InitTesting(): sent rsr\n");
#endif

    if (TN_WaitResult(*nd, &result) == 0)
    {
	nexus_printf("InitTesting(): TN_WaitResult failed\n");
	EPD_FreeNode(epd, NT_MASTER, node_num);
	return(0);
    }

    if (result < 0)
    {
	nexus_printf("InitTesting(): failed to get slave\n");
	EPD_FreeNode(epd, NT_MASTER, node_num);
	return(0);
    }

    /* point the master node to the slave */
    (*nd)->startpoint = *slave;
    (*nd)->node_num = result;
    
#ifdef DBG_InitTesting
    nexus_printf("InitTesting(): done\n");
#endif
    
    return(1);
}
/* -- InitTesting() -- */

/*
 * -- DoneTesting() --
 *
 * Set the slave free, release master node.
 *	o make sure node is valid master
 *	o send message to slave
 *	o free master node
 */
int DoneTesting(ep_data_t *epd, test_node_t **nd, int is_non_threaded_handler)
{
    nexus_buffer_t	buffer;
    int r;
    
#ifdef TRC_DoneTesting
    nexus_printf("DoneTesting(): hello\n");
#endif

    if (*nd == NULL)
    {
	nexus_printf("DoneTesting(): node_data = NULL");
	return(0);
    }

    if ((*nd)->node_type != NT_MASTER)
    {
	nexus_printf("DoneTesting(): expecting master, got %i",
		     (*nd)->node_type);
	return(0);
    }

    if (nexus_buffer_init(&buffer, nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("DoneTesting(): buffer_init failed\n");
	return(0);
    }
    
    nexus_put_int(&buffer, &((*nd)->node_num), 1);
    
    TN_PreWait((*nd), DISCONNECT_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &( (*nd)->startpoint ),
		       DISCONNECT_HANDLER_ID,
		       NEXUS_TRUE, is_non_threaded_handler) != NEXUS_SUCCESS)
    {
	nexus_printf("DoneTesting(): send_rsr failed\n");
	return(0);
    }

#ifdef DBG_DoneTesting
    nexus_printf("DoneTesting(): sent rsr\n");
#endif

    if (TN_WaitResult(*nd, &r))
    {
	if (r)
	{
	    nexus_startpoint_set_null(&((*nd)->startpoint));
	    EPD_FreeNode(epd, NT_MASTER, (*nd)->my_num);
	}
	else
	{
	    nexus_printf("TN_WaitResult(): slave thought it wasn't\n");
	    return(0);	   
	}
    }
    else
    {
	nexus_printf("DoneTesting(): TN_WaitResult failed\n");
	return(0);
    }
    
#ifdef DBG_DoneTesting
    nexus_printf("DoneTesting(): done\n");
#endif
   
    return(1);
}
/* -- DoneTesting() -- */

/*
 * -- WaitForStopWaitingMsg() --
 *
 * Wait on condition so main thread of context 0 doesn't waste cycles.
 * Wake up after message from remote context to StopWaiting.
 *	o lock endpoint mutex
 *	o set up flags saying we are waiting
 *	o wait
 *	o clean up flags
 *	o unlock mutex
 */
int WaitForStopWaitingMsg(ep_data_t *epd)
{
    nexus_printf("WaitForStopWaitingMsg(): hello me\n");

    EPD_MainLoop(epd);
    
    nexus_printf("WaitForStopWaitingMsg(): bye\n");
    return(1);
}
/* -- WaitForStopWaitingMsg() -- */

/*
 * -- SendStopWaitingMsg() --
 *
 * Send a message to a remote context to tell the main thread
 * to stop waiting.
 *	o send an empty buffer
 */

int SendStopWaitingMsg(nexus_startpoint_t *startpoint)
{
    nexus_buffer_t	buffer;

    nexus_buffer_init(&buffer, 0, 0);

    nexus_send_rsr(&buffer, startpoint,
		   STOP_WAITING_HANDLER_ID,
		   NEXUS_TRUE, NEXUS_FALSE);
    return(1);
}
/* -- SendStopWaitingMsg() -- */

/*
 * -- WaitForReply() --
 */

int WaitForReply(test_node_t *nd, int *result)
{
    return(TN_WaitResult(nd, result));
}
/* -- WaitForReply() -- */

/*
 * -- ONE_MSG_CHECKS --
 *
 * Macro for wrapper functions
 *	o make sure node is non-NULL and is a master
 *	o make sure msg_num is in range
 */

#define ONE_MSG_CHECKS(func_name)\
    if (nd == NULL)\
    {\
	nexus_printf(#func_name "(): nd = NULL\n");\
	return(0);\
    }\
    \
    if (nd->node_type != NT_MASTER)\
    {\
	nexus_printf(#func_name "(): not a master\n");\
	return(0);\
    }\
    \
    if ((msg_num < 0) || (msg_num > MAX_MSG_NUM))\
    {\
	nexus_printf(#func_name "(): illegal msg_num %i\n", msg_num);\
        return(0);\
    }

/*
 * -- TWO_MSG_CHECKS --
 *
 * Macro for wrapper functions
 *	o make sure node is non-NULL and is a master
 *	o make sure ctrl_msg_num is in range
 *	o make sure test_msg_num is in range
 */

#define TWO_MSG_CHECKS(func_name)\
    if (nd == NULL)\
    {\
	nexus_printf(#func_name "(): nd = NULL\n");\
	return(0);\
    }\
    \
    if (nd->node_type != NT_MASTER)\
    {\
	nexus_printf(#func_name "(): not a master\n");\
	return(0);\
    }\
    \
    if ((ctrl_msg_num < 0) || (ctrl_msg_num > MAX_MSG_NUM))\
    {\
	nexus_printf(#func_name "(): illegal ctrl_msg_num %i\n",\
		     ctrl_msg_num);\
        return(0);\
    }\
    \
    if ((test_msg_num < 0) || (test_msg_num > MAX_MSG_NUM))\
    {\
	nexus_printf(#func_name "(): illegal test_msg_num %i\n",\
		     test_msg_num);\
        return(0);\
    }

/************************************************
 * X_InitMsg()					*
 *						*
 *  >>	MessageInit(), blocking			*
 ************************************************/
  
/*
 * -- M_InitMsg() --
 */

int M_InitMsg(test_node_t *nd, int msg_num)
{
    ONE_MSG_CHECKS(M_InitMsg);

    return(MessageInit(TN_GetMsgPtr(nd, msg_num)));
}
/* -- M_InitMsg() -- */

/*
 * -- S_InitMsg() --
 */

int S_InitMsg(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;
    
    ONE_MSG_CHECKS(S_InitMsg);

#ifdef TRC_S_InitMsg
    nexus_printf("S_InitMsg(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 2 * nexus_sizeof_int(1), 0)
	!= NEXUS_SUCCESS)
    {
	nexus_printf("S_InitMsg(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 2 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);

#ifdef DBG_S_InitMsg
    nexus_printf("S_InitMsg(): ready to send\n");
#endif
    
    TN_PreWait(nd, INIT_MSG_HANDLER_ID);

    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		   INIT_MSG_HANDLER_ID,
		   NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_InitMsg(): send_rsr failed\n");
	return(0);
    }

#ifdef DBG_S_InitMsg
    nexus_printf("S_InitMsg(): sent rsr\n");
#endif
    
    return(TN_WaitResult(nd, NULL));
}
/* -- S_InitMsg() -- */

/************************************************
 * X_KillMsg()					*
 *						*
 *  >>	MessageDestroy(), blocking		*
 ************************************************/

/*
 * -- M_KillMsg() --
 */

int M_KillMsg(test_node_t *nd, int msg_num)
{
    ONE_MSG_CHECKS(M_KillMsg);

    return(MessageDestroy(TN_GetMsgPtr(nd, msg_num)));
}
/* -- M_KillMsg() -- */

/*
 * -- S_KillMsg() --
 */

int S_KillMsg(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;
    
    ONE_MSG_CHECKS(S_KillMsg);

#ifdef TRC_S_KillMsg
    nexus_printf("S_KillMsg(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 2 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_KillMsg(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 2 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);

    TN_PreWait(nd, KILL_MSG_HANDLER_ID);

    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       KILL_MSG_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_KillMsg(): send_rsr failed\n");
	return(0);
    }

    return(TN_WaitResult(nd, NULL));
}
/* -- S_KillMsg() -- */

/************************************************
 * X_MallocData()				*
 *						*
 *  >>	MessageDataMalloc(), blocking		*
 ************************************************/

/*
 * -- M_MallocData() --
 */

int M_MallocData(test_node_t *nd, int msg_num)
{
    ONE_MSG_CHECKS(M_MallocData);

    return(MessageDataMalloc(TN_GetMsgPtr(nd, msg_num)));
}
/* -- M_MallocData() -- */

/*
 * -- S_MallocData() --
 */

int S_MallocData(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;

    ONE_MSG_CHECKS(S_MallocData);

#ifdef TRC_S_MallocData
    nexus_printf("S_MallocData(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 2 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_MallocData(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 2 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);

    TN_PreWait(nd, MALLOC_DATA_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       MALLOC_DATA_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_MallocData(): send_rsr failed\n");
	return(0);
    }

    return(TN_WaitResult(nd, NULL));    
}
/* -- S_MallocData() -- */

/************************************************
 * X_FreeData()					*
 *						*
 *  >>	MessageDataFree(), blocking		*
 ************************************************/

/*
 * -- M_FreeData() --
 */

int M_FreeData(test_node_t *nd, int msg_num)
{
    ONE_MSG_CHECKS(M_FreeData);

    return(MessageDataFree(TN_GetMsgPtr(nd, msg_num)));
}
/* -- M_FreeData() -- */

/*
 * -- S_FreeData() --
 */

int S_FreeData(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;
    
    ONE_MSG_CHECKS(S_FreeData);

#ifdef TRC_S_FreeData
    nexus_printf("S_FreeData(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 2 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_FreeData(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 2 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);

    TN_PreWait(nd, FREE_DATA_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       FREE_DATA_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_FreeData(): send_rsr failed\n");
	return(0);	    
    }

    return(TN_WaitResult(nd, NULL));
}
/* -- S_FreeData() -- */

/************************************************
 * X_SendType()					*
 *						*
 *	transmit type, blocking			*
 ************************************************/

/*
 * -- M_SendType() --
 *
 * Send type to slave, wait for reply.
 */

int M_SendType(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;
    int			msg_sz;

    ONE_MSG_CHECKS(M_SendType);

#ifdef TRC_M_SendType
    nexus_printf("M_SendType(): hello\n");
#endif

    if (MessageGetTypeSize(TN_GetMsgPtr(nd, msg_num), &msg_sz) == 0)
    {
	nexus_printf("M_SendType(): MessageGetTypeSize failed\n");
	return(0);
    }

    if (nexus_buffer_init(&buffer, msg_sz + nexus_sizeof_int(1), 0)
	!= NEXUS_SUCCESS)
    {
	nexus_printf("M_SendType(): buffer_init failed\n");
	return(0);
    }

    ReportAddType(TN_GetReportPtr(nd), msg_sz + nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);

    if (PutMessageType(&buffer, TN_GetMsgPtr(nd, msg_num)) == 0)
    {
	nexus_printf("M_SendType(): PutMessageType failed\n");
	return(0);
    }

    TN_PreWait(nd, S_RECV_TYPE_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       S_RECV_TYPE_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("M_SendType(): send_rsr failed\n");
	return(0);
    }

    return(TN_WaitResult(nd, NULL));
}
/* -- M_SendType() -- */

/*
 * -- S_SendType() --
 *
 * Send request to slave for type, wait until received.
 */

int S_SendType(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;

    ONE_MSG_CHECKS(S_SendType);

#ifdef TRC_S_SendType
    nexus_printf("S_SendType(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 2 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_SendType(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 2 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);

    TN_PreWait(nd, S_SEND_TYPE_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       S_SEND_TYPE_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_SendType(): send_rsr failed\n");
	return(0);
    }

    return(TN_WaitResult(nd, NULL));
}
/* -- S_SendType() -- */

/************************************************
 * X_SendData()					*
 *						*
 *	transmit data, blocking			*
 ************************************************/

/*
 * -- M_SendData() --
 *
 * Send data buffer to slave, wait for reply.
 */

int M_SendData(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;
    int			msg_sz;

    ONE_MSG_CHECKS(M_SendData);

#ifdef TRC_M_SendData
    nexus_printf("M_SendData(): hello\n");
#endif

    if (MessageGetDataSize( TN_GetMsgPtr(nd, msg_num), &msg_sz) == 0)
    {
	nexus_printf("M_SendData(): GetDataSize failed\n");
	return(0);
    }

    if (nexus_buffer_init(&buffer, msg_sz + nexus_sizeof_int(1), 0)
	!= NEXUS_SUCCESS)
    {
	nexus_printf("M_SendData(): buffer_init failed\n");
	return(0);
    }

    ReportAddData(TN_GetReportPtr(nd), msg_sz + nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    
    if (PutMessageData(&buffer, TN_GetMsgPtr(nd, msg_num)) == 0)
    {
	nexus_printf("M_SendData(): PutData failed\n");
	return(0);
    }

    TN_PreWait(nd, S_RECV_DATA_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       S_RECV_DATA_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("M_SendData(): send_rsr failed\n");
	return(0);
    }

    return(TN_WaitResult(nd, NULL));    
}
/* -- M_SendData() -- */

/*
 * -- S_SendData()
 *
 * Send request to slave, wait for data to be received.
 */

int S_SendData(test_node_t *nd, int msg_num, int inth)
{
    nexus_buffer_t	buffer;

    ONE_MSG_CHECKS(S_SendData);

#ifdef TRC_S_SendData
    nexus_printf("S_SendData(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 2 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_SendData(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 2 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);

    TN_PreWait(nd, S_SEND_DATA_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       S_SEND_DATA_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_SendData(): send_rsr failed\n");
	return(0);
    }

    return(TN_WaitResult(nd, NULL));
}
/* -- S_SendData() -- */

/************************************************
 * X_GenType()					*
 *						*
 *  >>	UserGenType(), non-blocking		*
 ************************************************/

/*
 * -- M_GenType() --
 */

int M_GenType(test_node_t *nd, int msg_num, int type_g, int x)
{
    ONE_MSG_CHECKS(M_GenType);
    
    return(UserGenType(TN_GetMsgPtr(nd, msg_num), type_g, x));
}
/* -- M_GenType() -- */

/*
 * -- S_GenType() --
 */

int S_GenType(test_node_t *nd, int msg_num, int type_g, int x, int inth)
{
    nexus_buffer_t	buffer;

    ONE_MSG_CHECKS(S_GenType);

#ifdef TRC_S_GenType
    nexus_printf("S_GenType(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 4 * nexus_sizeof_int(1), 0) !=
	NEXUS_SUCCESS)
    {
	nexus_printf("S_GenType(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 4 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);
    nexus_put_int(&buffer, &type_g, 1);
    nexus_put_int(&buffer, &x, 1);

    TN_PreWait(nd, GEN_TYPE_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       GEN_TYPE_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_GenType(): send_rsr failed\n");
	return(0);
    }

    return(1);
}
/* -- S_GenType() -- */

/************************************************
 * X_GenData()					*
 *						*
 *  >>	UserGenData(), non-blocking		*
 ************************************************/

/*
 * -- M_GenData() --
 */

int M_GenData(test_node_t *nd, int msg_num, int data_g, int x)
{
    ONE_MSG_CHECKS(M_GenData);
    
    return(UserGenData(TN_GetMsgPtr(nd, msg_num), data_g, x));
}
/* -- M_GenData() -- */

/*
 * -- S_GenData() --
 */

int S_GenData(test_node_t *nd, int msg_num, int data_g, int x, int inth)
{
    nexus_buffer_t	buffer;

    ONE_MSG_CHECKS(S_GenData);

#ifdef TRC_S_GenData
    nexus_printf("S_GenData(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 4 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_GenData(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 4 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &msg_num, 1);
    nexus_put_int(&buffer, &data_g, 1);
    nexus_put_int(&buffer, &x, 1);

    TN_PreWait(nd, GEN_DATA_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       GEN_DATA_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_GenData(): send_rsr failed\n");
	return(0);
    }
    
    return(1);  
}
/* -- S_GenData() -- */


/************************************************
 * X_CmpType()					*
 *                                              *
 *  >>	MessageCheckType(), non-blocking	*
 ************************************************/

/*
 * -- M_CmpType() --
 */

int M_CmpType(test_node_t *nd, int ctrl_msg_num, int test_msg_num, int *errors)
{
    TWO_MSG_CHECKS(M_CmpType);

    return(CheckMessageType(TN_GetMsgPtr(nd, ctrl_msg_num),
			    TN_GetMsgPtr(nd, test_msg_num), errors));
}
/* -- M_CmpType() -- */

/*
 * -- S_CmpType() --
 */

int S_CmpType(test_node_t *nd, int ctrl_msg_num, int test_msg_num, int inth)
{
    nexus_buffer_t	buffer;
    
    TWO_MSG_CHECKS(M_CmpType);

#ifdef TRC_S_CmpType
    nexus_printf("S_CmpType(): hello\n");
#endif

    if (nexus_buffer_init(&buffer, 3 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_CmpType(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 3 * nexus_sizeof_int(1));

    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &ctrl_msg_num, 1);
    nexus_put_int(&buffer, &test_msg_num, 1);

    TN_PreWait(nd, CMP_TYPE_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       CMP_TYPE_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_CmpType(): send_rsr failed\n");
	return(0);
    }

    return(1);
}
/* -- S_CmpType() -- */

/************************************************
 * X_CmpData()					*
 *						*
 *  >>	MessageCheckData(), non-blocking	*
 ************************************************/

/*
 * -- M_CmpData() --
 */

int M_CmpData(test_node_t *nd, int ctrl_msg_num, int test_msg_num,
	      int *errors, FILE *f)
{
    TWO_MSG_CHECKS(M_CmpData);

    return(CheckMessageData(TN_GetMsgPtr(nd, ctrl_msg_num),
			    TN_GetMsgPtr(nd, test_msg_num),
			    errors, f));
}
/* -- M_CmpData() -- */

/*
 * -- S_CmpData() --
 */

int S_CmpData(test_node_t *nd, int ctrl_msg_num, int test_msg_num, int inth)
{
    nexus_buffer_t	buffer;

    TWO_MSG_CHECKS(S_CmpData);

#ifdef TRC_S_CmpData
    nexus_printf("S_CmpData(): hello\n");
#endif
    
    if (nexus_buffer_init(&buffer, 3 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("S_CmpData(): buffer_init failed\n");
	return(0);
    }

    ReportAddMisc(TN_GetReportPtr(nd), 3 * nexus_sizeof_int(1));
    
    nexus_put_int(&buffer, &(nd->node_num), 1);
    nexus_put_int(&buffer, &ctrl_msg_num, 1);
    nexus_put_int(&buffer, &test_msg_num, 1);

    TN_PreWait(nd, CMP_DATA_HANDLER_ID);
    
    if (nexus_send_rsr(&buffer, &(nd->startpoint),
		       CMP_DATA_HANDLER_ID,
		       NEXUS_TRUE, inth) != NEXUS_SUCCESS)
    {
	nexus_printf("S_CmpData(): send_rsr failed\n");
	return(0);
    }

    return(1);
}
/* -- S_CmpData() -- */

