/*
 * message.c
 *
 * Todo:
 *	allow tolerance when checking floats and doubles
 */

#include "message.h"

#ifdef WITHOUT_NEXUS
#define nexus_printf	printf
#define nexus_malloc	malloc
#define nexus_free	free
#endif /* WITHOUT_NEXUS */

/*
 * -- VERIFY_MESSAGE() --
 *
 * Check and make sure all fields are valid, numbers add up, etc.
 * Great way to catch problems in communication or program logic.
 */
#ifdef SKIP_CONSISTENCY_CHECK

#define VERIFY_MESSAGE(msg, func, action)

#else /* SKIP_CONSISTENCY_CHECK */

#define VERIFY_MESSAGE(msg, func, action) \
    if (MessageConsistent(msg) == 0) \
    { \
	nexus_printf(#func "(): Message failed consistency check\n"); \
	action; \
    }

/*
 * -- MessageConsistent --
 *
 * Function to do the actual checks.
 */
static int MessageConsistent(message_t *msg)
{
    data_array_t	*da;
    int			i;
    int			ok;

    ok = 1;
    
    if (msg == NULL)
    {
	nexus_printf("MessageConsistent(): msg == NULL");
	return(0);
    }

    if ((msg->num_arrays < 0) || (msg->num_arrays > MAX_TYPES_PER_MSG))
    {
	nexus_printf("MessageConsistent(): illegal num_arrays %i\n",
		msg->num_arrays);
	return(0);
    }

    if (msg->data_array == NULL)
    {
	nexus_printf("MessageConsistent(): data_array == NULL");
	return(0);
    }

    for (i = 0; i < MAX_TYPES_PER_MSG; i++)
    {
	da = &(msg->data_array[i]);
	
	if (i < msg->num_arrays)
	{
	    if ((da->data_type < MIN_VALID_TYPE) || (da->data_type > MAX_VALID_TYPE))
	    {
		nexus_printf(
			"MessageConsistent(): array[%i] has illegal type %i\n",
			i, da->data_type);
		ok = 0;
	    }

	    if ((da->data_count < 0) || (da->data_count > MAX_DATA_COUNT))
	    {
		nexus_printf(
			"MessageConsistent(): array[%i] has illegal count %i\n",
			i, da->data_count);
		ok = 0;
	    }

	    if (msg->have_data)
	    {
		if (da->data_ptr == NULL)
		{
		    nexus_printf(
			    "MessageConsistent(): array[%i] has NULL ptr\n", i);
		    ok = 0;
		}
	    }
	}
	else /* if (i < num_arrays) */
	{
	    if (da->data_type != VOID_DATA)
	    {
		nexus_printf(
			"MessageConsistent(): array[%i] has non-void type %i\n",
			i, da->data_type);
		ok = 0;
	    }

	    if (da->data_count != 0)
	    {
		nexus_printf(
			"MessageConsistent(): array[%i] has non-zero count %i\n",
			i, da->data_count);
		ok = 0;
	    }

	    if (da->data_ptr != NULL)
	    {
		nexus_printf(
			"MessageConsistent(): array[%i] has non-NULL ptr\n", i);
		ok = 0;
	    }
	}
    } /* for */

    return(ok);
}
/* -- MessageConsistent() -- */

#endif /* SKIP_CONSISTENCY */
/* -- VERIFY_MESSAGE -- */

/*
 * -- MessageInit() --
 */
int MessageInit(message_t *msg)
{
    int i;
    
    msg->num_arrays = 0;
    msg->have_data = 0;
    
    msg->data_array = (data_array_t *)nexus_malloc(MAX_TYPES_PER_MSG *
						   sizeof(data_array_t));
    if (msg->data_array == NULL)
    {
	nexus_printf("MessageInit(): nexus_malloc returned NULL\n");
	return(0);
    }

    for (i = 0; i < MAX_TYPES_PER_MSG; i++)
    {
	msg->data_array[i].data_type = VOID_DATA;
	msg->data_array[i].data_count = 0;
	msg->data_array[i].as_array = 1;
	msg->data_array[i].test_info = 0;
	msg->data_array[i].data_ptr = NULL;
    }

    return(1);
}
/* -- MessageInit() -- */

/*
 * -- MessageAddArray() --
 */
int MessageAddArray(message_t *msg, int type, int count)
{
    VERIFY_MESSAGE(msg, MessageAddArray, return(0));

    if (msg->have_data)
    {
	nexus_printf("MessageAddArray(): already has data\n");
	return(0);
    }

    if (msg->num_arrays < MAX_TYPES_PER_MSG)
    {
	msg->data_array[msg->num_arrays].data_type = type;
	msg->data_array[msg->num_arrays].data_count = count;
	msg->num_arrays++;
	
	return(1);
    }
    else
    {
	nexus_printf("MessageAddArray(): no room for more\n");
	return(0);
    }
}
/* -- MessageAddArray() -- */

/*
 * -- MessageDataMalloc() --
 */
int MessageDataMalloc(message_t *msg)
{
    int i, s;

    VERIFY_MESSAGE(msg, MessageDataMalloc, return(0));

    if (msg->have_data)
    {
	nexus_printf("MessageDataMalloc(): already have data\n");
	return(0);
    }

    for (i = 0; i < msg->num_arrays; i++)
    {
	s = SizeofType(msg->data_array[i].data_type);
	
	msg->data_array[i].data_ptr =
	    nexus_malloc(s * msg->data_array[i].data_count);
	
	if (msg->data_array[i].data_ptr == NULL)
	{
	    
	    nexus_printf("MsgDataMalloc(): nexus_malloc returned NULL");
	    
	    return(0);
	}
    } /* for */

    msg->have_data = 1;
    
    return(1);
}
/* -- MessageDataMalloc() -- */

/*
 * -- MessageDataFree() --
 */
int MessageDataFree(message_t *msg)
{
    int i;

    VERIFY_MESSAGE(msg, MessageDataFree, nexus_printf("attempting to free anyway\n"));

    if (msg->data_array == NULL)
    {
	nexus_printf("MessageDataFree(): data_array == NULL\n");
	return(0);
    }

    if ((msg->num_arrays < 0) || (msg->num_arrays > MAX_TYPES_PER_MSG))
    {
	nexus_printf("MessageDataFree(): corrupt num_arrays = %i\n",
		msg->num_arrays);
	return(0);
    }
    
    for (i = 0; i < msg->num_arrays; i++)
    {
	if (msg->data_array[i].data_ptr == NULL)
	{
	    nexus_printf("MessageDataFree(): data_ptr[%i] == NULL\n", i);
	}
	else
	{
	    nexus_free(msg->data_array[i].data_ptr);
	    msg->data_array[i].data_ptr = NULL;
	}
    } /* for */
    return(1);
}
/* -- MessageDataFree() -- */

/*
 * -- MessageDestroy() --
 */
int MessageDestroy(message_t *msg)
{
    nexus_free(msg->data_array);
    msg->data_array = NULL;
    msg->num_arrays = 0;
    msg->have_data = 0;
    return(1);
}
/* -- MessageDestroy() -- */

/************************************************
 *                                              *
 *                                              *
 ************************************************/

/*
 * -- MessageResetOptions() --
 */
int MessageResetOptions(message_t *msg)
{
    int i;

    VERIFY_MESSAGE(msg, MessageResetOptions, return(0));

    for (i = 0; i < msg->num_arrays; i++)
    {
	msg->data_array[i].as_array = 1;
	msg->data_array[i].test_info = 0;
    }

    return(1);
}
/* -- MessageResetOptions() -- */

/*
 * -- MessageSetXmitOption() --
 */
int MessageSetXmitOption(message_t *msg, int array_num, int send_as_array)
{
    VERIFY_MESSAGE(msg, MessageSetXmitOption, return(0));

    if ((array_num < 0) || (array_num >= msg->num_arrays))
    {
	nexus_printf("MessageSetXmitOption(): illegal array_num %i\n", array_num);
	return(0);
    }

    msg->data_array[array_num].as_array = send_as_array;

    return(1);
}
/* MessageSetXmitOption() -- */

/*
 * -- MessageSetTestInfo() --
 */
int MessageSetTestInfo(message_t *msg, int array_num, int test_info)
{
    VERIFY_MESSAGE(msg, MessageSetTestInfo, return(0));

    if ((array_num < 0) || (array_num >= msg->num_arrays))
    {
	nexus_printf("MessageSetTestInfo(): illegal array_num %i\n", array_num);
	return(0);
    }

    msg->data_array[array_num].test_info = test_info;

    return(1);
}
/* -- MessageSetTestInfo() -- */

/*
 * -- MessageGetTestInfo() --
 */
int MessageGetTestInfo(message_t *msg, int array_num, int *test_info)
{
    VERIFY_MESSAGE(msg, MessageGetTestInfo, return(0));

    if ((array_num < 0) || (array_num >= msg->num_arrays))
    {
	nexus_printf("MessageGetTestInfo(): illegal array_num %i\n", array_num);
	return(0);
    }

    *test_info = msg->data_array[array_num].test_info;

    return(1);
}
/* -- MessageGetTestInfo() -- */

/*
 * -- MessageGetDataPtr() --
 */
int MessageGetDataPtr(message_t *msg, int array_num, void* *data_ptr)
{
    VERIFY_MESSAGE(msg, MessageGetDataPtr, return(0));

    if (msg->have_data == 0)
    {
	nexus_printf("MessageGetDataPtr(): don't have data\n");
	return(0);
    }

    if ((array_num < 0) || (array_num >= msg->num_arrays))
    {
	nexus_printf("MessageGetDataPtr(): illegal array_num %i\n", array_num);
	return(0);
    }

    *data_ptr = msg->data_array[array_num].data_ptr;

    return(1);
}
/* -- MessageGetDataPtr() -- */

/*
 * -- MessageGetDataCount() --
 */
int MessageGetDataCount(message_t *msg, int array_num, int *data_count)
{
    VERIFY_MESSAGE(msg, MessageGetDataCount, return(0));
    
    if (msg->have_data == 0)
    {
	nexus_printf("MessageGetDataCount(): don't have data\n");
	return(0);
    }

    if ((array_num < 0) || (array_num >= msg->num_arrays))
    {
	nexus_printf("MessageGetDataCount(): illegal array_num %i\n", array_num);
	return(0);
    }

    *data_count = msg->data_array[array_num].data_count;

    return(1);
}
/* -- MessageGetDataCount() -- */

/*
 * -- MessageGetDataType() --
 */
int MessageGetDataType(message_t *msg, int array_num, int *data_type)
{
    VERIFY_MESSAGE(msg, MessageGetDataType, return(0));

    if (msg->have_data == 0)
    {
	nexus_printf("MessageGetDataCount(): don't have data\n");
	return(0);
    }

    if ((array_num < 0) || (array_num >= msg->num_arrays))
    {
	nexus_printf("MessageGetDataCount(): illegal array_num %i\n",
		     array_num);
	return(0);
    }

    *data_type = msg->data_array[array_num].data_type;

    return(1);    
}
/* -- MessageGetDataType() -- */

/************************************************
 *                                              *
 *                                              *
 ************************************************/

/*
 * -- CheckArray() --
 *
 * Check a single array of type and length to see if they are the same.
 * Keep track of errors, optionally print them out to a file.
 * Return non-zero if the message was valid and checked.
 */

/*
 * Macro to test for a single type.
 */
#define CHECK_DATA(type, abbrev) \
    abbrev ## _p1 = (type *)(ctrl_array->data_ptr); \
    abbrev ## _p2 = (type *)(test_array->data_ptr); \
    if (f != NULL) \
    { \
	nexus_stdio_lock(); \
	for (i = 0; i < c; i++) \
	{ \
	    if (abbrev ## _p1[i] != abbrev ## _p2[i]) \
	    { \
		(*errors)++; \
		fprintf(f, "expected " #abbrev "[%i] = %i, got %i\n", \
			i, abbrev ## _p1[i], abbrev ## _p2[i]); \
	    } \
	} \
	nexus_stdio_unlock(); \
    } \
    else \
    { \
	for (i = 0; i < c; i++) \
	    if (abbrev ## _p1[i] != abbrev ## _p2[i]) \
		(*errors)++; \
    } \
    break;

static int CheckArray(data_array_t *ctrl_array, data_array_t *test_array,
		      int *errors, FILE *f)
{
    char		*char_p1, *char_p2;
    unsigned char	*uchar_p1, *uchar_p2;
    short		*short_p1, *short_p2;
    unsigned short	*ushort_p1, *ushort_p2;
    int			*int_p1, *int_p2;
    unsigned int	*uint_p1, *uint_p2;
    long		*long_p1, *long_p2;
    unsigned long	*ulong_p1, *ulong_p2;
    float		*float_p1, *float_p2;
    double		*double_p1, *double_p2;
    int			i, c, t;

    *errors = 0;

    if (test_array->data_type != ctrl_array->data_type)
	return(0);

    t = ctrl_array->data_type;

    if (ctrl_array->data_count != test_array->data_count)
	return(0);

    c = ctrl_array->data_count;
    
    switch (t)
    {
    case CHAR_DATA:
	CHECK_DATA(char, char);
    case UCHAR_DATA:
	CHECK_DATA(unsigned char, uchar);
    case SHORT_DATA:
	CHECK_DATA(short, short);
    case USHORT_DATA:
	CHECK_DATA(unsigned short, ushort);
    case INT_DATA:
	CHECK_DATA(int, int);
    case UINT_DATA:
	CHECK_DATA(unsigned int, uint);
    case LONG_DATA:
	CHECK_DATA(long, long);
    case ULONG_DATA:
	CHECK_DATA(unsigned long, ulong);
    case FLOAT_DATA:
	CHECK_DATA(float, float);
    case DOUBLE_DATA:
	CHECK_DATA(double, double);
    default:
	nexus_printf("CheckArray(): unknown data type %i\n", t);
	return(0);
    } /* switch(data_type) */

    return(1);
}
/* -- CheckArray() -- */

/*
 * -- CheckMessageData() --
 *
 * Check every data array in two messages to see if they differ.
 * Keep track of error count, optionally print to a file.
 * Return non-zero if the message was valid and checked.
 */
int CheckMessageData(message_t *ctrl_msg, message_t *test_msg,
		     int *errors, FILE *f)
{
    int i, e, r;
    
    *errors = 0;
    r = 1;

    VERIFY_MESSAGE(ctrl_msg, CheckMessageData, return(0));
    VERIFY_MESSAGE(test_msg, CheckMessageData, return(0));

    for (i = 0; i < ctrl_msg->num_arrays; i++)
    {
	if (CheckArray(&(ctrl_msg->data_array[i]),
		       &(test_msg->data_array[i]), &e, f) == 0)
	    r = 0;
	*errors += e;
    }

    return(r);    
}
/* -- CheckMessageData() -- */

/*
 * -- CheckMessageType() --
 *
 * Check two messages to see if the types differ.
 * Count errors and print them out.
 */
int CheckMessageType(message_t *ctrl_msg, message_t *test_msg,
		     int *errors)
{
    int i;
    
    *errors = 0;

    VERIFY_MESSAGE(ctrl_msg, CheckMessageData, return(0));
    VERIFY_MESSAGE(test_msg, CheckMessageData, return(0));

    if (ctrl_msg->num_arrays != test_msg->num_arrays)
    {
	nexus_printf(
		"CheckMessageType(): num_arrays ctrl = %i, test = %i\n",
		ctrl_msg->num_arrays, test_msg->num_arrays);

	return(0);		
    }
    
    for (i = 0; i < ctrl_msg->num_arrays; i++)
    {
	if (test_msg->data_array[i].data_type != ctrl_msg->data_array[i].data_type)
	{
	    nexus_printf(
		    "CheckMessageType(): type[%i] ctrl = %i, test = %i\n", i,
		    ctrl_msg->data_array[i].data_type,
		    test_msg->data_array[i].data_type);

	    (*errors)++;
	}

	if (test_msg->data_array[i].data_count != ctrl_msg->data_array[i].data_count)
	{
	    nexus_printf("CheckMessageType(): count[%i] ctrl = %i, test = %i\n", i,
		    ctrl_msg->data_array[i].data_count,
		    test_msg->data_array[i].data_count);

	    (*errors)++;
	}
    } /* for */

    return(1);
}
/* -- CheckMessageType() -- */

/************************************************
 *                                              *
 *                                              *
 ************************************************/

#ifndef WITHOUT_NEXUS

/*
 * -- MessageGetTypeSize() --
 *
 * Calculate the nexus_buffer_t size required to send the message type.
 */
int MessageGetTypeSize(message_t *msg, int *size)
{
    *size = 0;

    VERIFY_MESSAGE(msg, MessageGetTypeSize, return(0));
    
    *size = (MAX_TYPES_PER_MSG * 2  + 2) * nexus_sizeof_int(1);
    return(1);
}
/* -- MessageGetTypeSize() -- */

/*
 * -- PutMessageType() --
 *
 * Put the message type into a nexus_buffer_t.
 */
int PutMessageType(nexus_buffer_t *buffer, message_t *msg)
{
    int i;
    int c;

    VERIFY_MESSAGE(msg, PutMessageType, return(0));

    c = 0;
    
    for (i = 0; i < MAX_TYPES_PER_MSG; i++)
    {
	nexus_put_int(buffer, &(msg->data_array[i].data_type), 1);
	nexus_put_int(buffer, &(msg->data_array[i].data_count), 1);
	c = (c + msg->data_array[i].data_type
	     + msg->data_array[i].data_count) % 12345;
#ifdef SHOW_MSG_TYPE
	nexus_printf("::PutType[%i] type = %i, count = %i\n", i,
		     msg->data_array[i].data_type,
		     msg->data_array[i].data_count);
#endif
    }

    nexus_put_int(buffer, &(msg->num_arrays), 1);
    nexus_put_int(buffer, &c, 1);

#ifdef SHOW_MSG_TYPE
    nexus_printf("::PutType num = %i, checksum = %i\n", msg->num_arrays, c);
#endif

    return(1);
}
/* -- PutMessageType() -- */

/*
 * GetMessageType()
 *
 * Get message type from a nexus_buffer_t.
 */
int GetMessageType(nexus_buffer_t *buffer, message_t *msg)
{
    int i, c;

    VERIFY_MESSAGE(msg, GetMessageType, return(0));

    if (msg->have_data)
    {
	nexus_printf("GetMessageType(): old message has data\n");
	return(0);
    }
    
    c = 0;
    
    for (i = 0; i < MAX_TYPES_PER_MSG; i++)
    {
	nexus_get_int(buffer, &(msg->data_array[i].data_type), 1);
	nexus_get_int(buffer, &(msg->data_array[i].data_count), 1);
	c = (c + msg->data_array[i].data_type
	     + msg->data_array[i].data_count) % 12345;
#ifdef SHOW_MSG_TYPE
	nexus_printf("::GetType[%i] type = %i, count = %i\n", i,
		     msg->data_array[i].data_type,
		     msg->data_array[i].data_count);
#endif
    }

    nexus_get_int(buffer, &(msg->num_arrays), 1);
    nexus_get_int(buffer, &i, 1); /* checksum */

#ifdef SHOW_MSG_TYPE
    nexus_printf("::GetType num = %i, mysum = %i correct = %i\n",
		 msg->num_arrays, c, i);
#endif

    if (i != c)
    {
	nexus_printf("GetMessageType(): failed checksum\n");
	return(0);
    }

    if (MessageConsistent(msg) == 0)
    {
	nexus_printf("GetMessageType(): new message failed consistency check\n");
	return(0);
    }

    return(1);
}
/* -- GetMessageType() -- */

/***************************************
 *                                     *
 ***************************************/

/*
 * -- MessageGetDataSize() --
 *
 * Calculate the size required to put the message data into a nexus_buffer_t.
 */
int MessageGetDataSize(message_t *msg, int *size)
{
    int			i, j;

    *size = 0;
    
    VERIFY_MESSAGE(msg, MessageGetDataSize, return(0));

    if (msg->have_data == 0)
    {
	nexus_printf("MessageGetDataSize(): no data\n");
	return(0);
    }

    j = 0;
    for (i = 0; i < msg->num_arrays; i++)
    {
	if (msg->data_array[i].as_array)
	    j += NexusSizeofType(msg->data_array[i].data_type,
				 msg->data_array[i].data_count);
	else
	    j += msg->data_array[i].data_count *
		NexusSizeofType(msg->data_array[i].data_type, 1);
    }

    *size = j;
    
    return(1);
}
/* -- MessageGetDataSize() -- */

/*
 * -- PutMessageData() --
 *
 * Put message data into a nexus_buffer_t.
 */

/*
 * Macro to put a single type into a buffer.
 */
#define PUT_DATA_CASE(case_lbl, put_lbl, type) \
	case case_lbl ## _DATA: \
	    if (da->as_array) \
		nexus_put_ ## put_lbl (buffer, (type *)da->data_ptr, da->data_count); \
	    else \
		for (j = 0; j < da->data_count; j++) \
		    nexus_put_ ## put_lbl (buffer, &(((type *)da->data_ptr)[j]), 1); \
	    break;

int PutMessageData(nexus_buffer_t *buffer, message_t *msg)
{
    data_array_t	*da;
    int			i, j;

    VERIFY_MESSAGE(msg, PutMessageData, return(0));

    if (msg->have_data == 0)
    {
	nexus_printf("PutMessageData(): no data\n");
	return(0);
    }

    for (i = 0; i < msg->num_arrays; i++)
    {
	da = &(msg->data_array[i]);
	       
	switch (da->data_type)
	{
	    PUT_DATA_CASE(CHAR, char, char);
	    PUT_DATA_CASE(UCHAR, u_char, unsigned char);
	    PUT_DATA_CASE(SHORT, short, short);
	    PUT_DATA_CASE(USHORT, u_short, unsigned short);
	    PUT_DATA_CASE(INT, int, int);
	    PUT_DATA_CASE(UINT, u_int, unsigned int);
	    PUT_DATA_CASE(LONG, long, long);
	    PUT_DATA_CASE(ULONG, u_long, unsigned long);
	    PUT_DATA_CASE(FLOAT, float, float);
	    PUT_DATA_CASE(DOUBLE, double, double);	
	default:
	    nexus_printf("PutMessageData(): unknown data type %i\n",
			 da->data_type);
	    return(0);
	} /* switch (data_type) */
    } /* for */

    return(1);
}
/* -- PutMessageData() -- */

/*
 * -- GetMessageData() --
 *
 * Get message data from a nexus_buffer_t.
 */

/*
 * Macro to get a single type.
 */
#define GET_DATA_CASE(case_lbl, get_lbl, type) \
        case case_lbl ## _DATA: \
	    if (da->as_array) \
		nexus_get_ ## get_lbl (buffer, (type *)da->data_ptr, da->data_count); \
	    else \
		for (j = 0; j < da->data_count; j++) \
		    nexus_get_ ## get_lbl (buffer, &(((type *)da->data_ptr)[j]), 1); \
	    break;	    

int GetMessageData(nexus_buffer_t *buffer, message_t *msg)
{
    data_array_t	*da;
    int			i, j;

    VERIFY_MESSAGE(msg, GetMessageData, return(0));

    if (msg->have_data == 0)
    {
	nexus_printf("GetMessageData(): don't have data\n");
	return(0);
    }

    for (i = 0; i < msg->num_arrays; i++)
    {
	da = &(msg->data_array[i]);
	       
	switch (da->data_type)
	{
	    GET_DATA_CASE(CHAR, char, char);
	    GET_DATA_CASE(UCHAR, u_char, unsigned char);
	    GET_DATA_CASE(SHORT, short, short);
	    GET_DATA_CASE(USHORT, u_short, unsigned short);
	    GET_DATA_CASE(INT, int, int);
	    GET_DATA_CASE(UINT, u_int, unsigned int);
	    GET_DATA_CASE(LONG, long, long);
	    GET_DATA_CASE(ULONG, u_long, unsigned long);
	    GET_DATA_CASE(FLOAT, float, float);
	    GET_DATA_CASE(DOUBLE, double, double);	

	default:
	    nexus_printf("GetMessageData(): unknown type %i\n", da->data_type);
	    return(0);
	} /* switch (data_type) */
    } /* for */

    return(1);
}
/* -- GetMessageData() -- */

#endif /* !WITHOUT_NEXUS */

/************************************************
 *                                              *
 *                                              *
 ************************************************/

/*
 * -- SizeofType() --
 *
 * Return sizeof type in memory.
 */
int SizeofType(int data_type)
{
    switch (data_type)
    {
    case CHAR_DATA:
	return sizeof(char);
    case UCHAR_DATA:
	return sizeof(unsigned char);
    case SHORT_DATA:
	return sizeof(short);
    case USHORT_DATA:
	return sizeof(unsigned short);
    case INT_DATA:
	return sizeof(int);
    case UINT_DATA:
	return sizeof(unsigned int);
    case LONG_DATA:
	return sizeof(long);
    case ULONG_DATA:
	return sizeof(unsigned long);
    case FLOAT_DATA:
	return sizeof(float);
    case DOUBLE_DATA:
	return sizeof(double);
    default:
	nexus_printf("SizeofType(): unknown data_type %i\n", data_type);
	return(0);
    }
}
/* -- SizeofType() -- */

#ifndef WITHOUT_NEXUS

/*
 * -- NexusSizeofType() --
 *
 * Return sizeof type, count in nexus_buffer_t.
 */
int NexusSizeofType(int data_type, int data_count)
{
    switch (data_type)
    {
    case SHORT_DATA:
	return nexus_sizeof_short(data_count);
    case USHORT_DATA:
	return nexus_sizeof_u_short(data_count);
    case INT_DATA:
	return nexus_sizeof_int(data_count);
    case UINT_DATA:
	return nexus_sizeof_u_int(data_count);
    case LONG_DATA:
	return nexus_sizeof_long(data_count);
    case ULONG_DATA:
	return nexus_sizeof_u_long(data_count);
    case CHAR_DATA:
	return nexus_sizeof_char(data_count);
    case UCHAR_DATA:
	return nexus_sizeof_u_char(data_count);
    case FLOAT_DATA:
	return nexus_sizeof_float(data_count);
    case DOUBLE_DATA:
	return nexus_sizeof_double(data_count);
    default:
	nexus_printf("NexusSizeofType(): unknown data_type %i\n", data_type);
	return(0);
    } /* switch (data_type) */
}
/* -- NexusSizeofInt() -- */

#endif /* !WITHOUT_NEXUS */

/*
 * -- ZeroMessageData()
 *
 * Fill in all the data with 0.
 */
int ZeroMessageData(message_t *msg)
{
    int i, j;

    VERIFY_MESSAGE(msg, ZeroMessageData, return(0));

    if (msg->have_data == 0)
    {
	nexus_printf("ZeroMessageData(): no data\n");
	return(0);
    }

    for (i = 0; i < msg->num_arrays; i++)
    {
	switch(msg->data_array[i].data_type)
	{
	case CHAR_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((char *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case UCHAR_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((unsigned char *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case SHORT_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((short *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case USHORT_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((unsigned short *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case INT_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((int *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case UINT_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((unsigned int *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case LONG_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((long *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case ULONG_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((unsigned long *) msg->data_array[i].data_ptr)[j] = 0;
	    break;
	case FLOAT_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((float *) msg->data_array[i].data_ptr)[j] = 0.0;
	    break;
	case DOUBLE_DATA:
	    for (j = 0; j < msg->data_array[i].data_count; j++)
		((double *) msg->data_array[i].data_ptr)[j] = 0.0;
	    break;
	default:
	    nexus_printf("ZeroMessageData(): unknown data type %i\n",
		    msg->data_array[i].data_type);
	    return(0);
	} /* switch */
    } /* for */

    return(1);
}
/* -- ZeroMessageData() -- */

/*
 * -- MessageFilePut() --
 */
int MessageFilePut(FILE *f, message_t *msg)
{
    char *p;
    int a, i;

    
    
    return(1);
}
/* -- MessageFilePut() -- */

/*
 * -- MessageFileGet() --
 */
int MessageFileGet(FILE *f, int dc_format, message_t *msg)
{
}
/* -- MessageFileGet() -- */
