/*
 * handlers.c
 *
 * Contains all rsr handler code, and
 * init/cleanup routines for endpoints.
 */

#include "big_test.h"
#include "handlers.h"
#include "wrappers.h"
#include "debug.h"
#include "group.h"
#include "run_test.h"

#include "everything.h"

/*
 * -- SendResult() --
 *
 * This function sends a status and result from a slave handler
 * back to the master node_num at the given startpoint.  Handler_id
 * of the sender is checked by the master for debugging.
 */

static void SendResult(nexus_startpoint_t *startpoint,
		       int node_num,
		       int handler_id,
		       int status, int result,
		       nexus_bool_t is_non_threaded_handler)
{
    nexus_buffer_t	buffer;
    
#ifdef TRC_SendResult
    nexus_printf("SendResult(): node %i, id %i, stat %i, res %i\n",
		 node_num, handler_id, status, result);
#else
#ifdef TRC_NEG_RESULT
    if (status == 0)
	nexus_printf("SendResult(): status = 0, node = %i, id %i, "
		     "result = %i\n", node_num, handler_id, result);
#endif /* TRC_NEG_RESULT */
#endif /* TRC_SendResult */

    if (nexus_buffer_init(&buffer, 4 * nexus_sizeof_int(1), 0) != NEXUS_SUCCESS)
    {
	nexus_printf("SendResult(): buffer_init failed\n");
	return;
    }
    nexus_put_int(&buffer, &node_num, 1);
    nexus_put_int(&buffer, &handler_id, 1);
    nexus_put_int(&buffer, &status, 1);
    nexus_put_int(&buffer, &result, 1);

#ifdef DBG_SendResult
    nexus_printf("SendResult(): sending rsr\n");
#endif
    
    if (nexus_send_rsr(&buffer, startpoint,
		       RESULT_HANDLER_ID,
		       NEXUS_TRUE, is_non_threaded_handler) != NEXUS_SUCCESS)
    {
	nexus_printf("SendResult(): send_rsr failed\n");
	return;
    }

#ifdef DBG_SendResult
    nexus_printf("SendResult(): sent rsr\n");
#endif
    
    return;
}
/* -- SendResult() -- */

/* prototypes for handler functions */

static void result_handler(nexus_endpoint_t *endpoint,
			   nexus_buffer_t *buffer,
			   nexus_bool_t is_non_threaded_handler);
static void connect_handler(nexus_endpoint_t *endpoint,
			    nexus_buffer_t *buffer,
			    nexus_bool_t is_non_threaded_handler);
static void disconnect_handler(nexus_endpoint_t *endpoint,
			       nexus_buffer_t *buffer,
			       nexus_bool_t is_non_threaded_handler);
static void init_msg_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler);
static void kill_msg_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler);
static void malloc_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void free_data_handler(nexus_endpoint_t *endpoint,
			      nexus_buffer_t *buffer,
			      nexus_bool_t is_non_threaded_handler);
static void cmp_type_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler);
static void cmp_data_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler);
static void s_send_type_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void m_recv_type_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void s_recv_type_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void s_send_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void m_recv_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void s_recv_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler);
static void gen_type_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler);
static void gen_data_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler);
static void shutdown_context_handler(nexus_endpoint_t *endpoint,
				     nexus_buffer_t *buffer,
				     nexus_bool_t is_non_threaded_handler);
static void stop_waiting_handler(nexus_endpoint_t *endpoint,
				 nexus_buffer_t *buffer,
				 nexus_bool_t is_non_threaded_handler);
static void join_group_handler(nexus_endpoint_t *endpoint,
			       nexus_buffer_t *buffer,
			       nexus_bool_t is_non_threaded_handler);
static void run_test_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler);
static void report_handler(nexus_endpoint_t *endpoint,
				 nexus_buffer_t *buffer,
				 nexus_bool_t is_non_threaded_handler);

/* handler table */

static nexus_handler_t handlers[] =
{
    {NEXUS_HANDLER_TYPE_NON_THREADED, result_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, connect_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, disconnect_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, init_msg_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, kill_msg_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, malloc_data_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, free_data_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, cmp_type_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, cmp_data_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, s_send_type_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, m_recv_type_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, s_recv_type_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, s_send_data_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, m_recv_data_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, s_recv_data_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, gen_type_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, gen_data_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, shutdown_context_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, stop_waiting_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, join_group_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, run_test_handler},
    {NEXUS_HANDLER_TYPE_NON_THREADED, report_handler}
};

/*
 * -- InitEndpoint() --
 *
 * Initialize an endpoint for use in testing.  Num_nodes sets the
 * limit for how many tests this endpoint may be in at once.
 *	o set the handler table
 *	o create ep_data
 *	o set the user pointer
 */

int InitEndpoint(nexus_endpoint_t *ep, int num_nodes, int verbose)
{
    nexus_endpointattr_t	ep_attr;
    ep_data_t			*epd;

    /* set the handler table */    
    nexus_endpointattr_init(&ep_attr);
    nexus_endpointattr_set_handler_table(&ep_attr, handlers,
					 sizeof(handlers) /
					 sizeof(nexus_handler_t));
    nexus_endpoint_init(ep, &ep_attr);
    nexus_endpointattr_destroy(&ep_attr);

    /* create endpoint_data */
    if (EPD_Create(&epd, num_nodes, verbose) == 0)
    {
	nexus_printf("InitEndpoint(): EPD_Create failed\n");
	return(0);
    }

    epd->endpoint = ep;
    
    /* set the endpoint's user pointer */
    nexus_endpoint_set_user_pointer(ep, epd);
    
    return(1);
}
/* -- InitEndpoint() -- */

/*
 * -- DoneEndpoint() --
 */

int DoneEndpoint(nexus_endpoint_t *ep)
{
    ep_data_t		*epd;

#ifdef TRC_DoneEndpoint
    nexus_printf("DoneEndpoint(): hello\n");
#endif
    
    epd = (ep_data_t *)nexus_endpoint_get_user_pointer(ep);
    if (epd == NULL)
    {
	nexus_printf("DoneEndpoint(): epd == NULL\n");
	return(0);
    }

    /* print the report */
    nexus_printf("DoneEndpoint(): report:\n");
    ReportPrint(stdout, &(epd->report));

    /* destroy the ep_data */
    EPD_Destroy(&epd);

    /* set the user pointer to NULL */
    nexus_endpoint_set_user_pointer(ep, NULL);

    return(1);
}
/* -- DoneEndpoint() -- */

/*
 * -- HANDLER_CHECK_FIRST --
 *
 * Macro to do the common checks for the test rsr's that
 * are sent to a specific node_num.
 *	o get the endpoint user pointer, make sure non-NULL
 *	o get the node_num from the buffer, make sure in range
 *	o get the pointer to the test node
 *	o make sure the node is the right type
 */

#define HANDLER_CHECK_FIRST(func_name, right_type)\
    ep_data_t		*epd;\
    test_node_t		*nd;\
    int			node_num;\
    \
    epd = (ep_data_t*)nexus_endpoint_get_user_pointer(endpoint);\
    if (epd == NULL)\
    {\
        nexus_printf(#func_name "(): user_pointer = NULL\n");\
	return;\
    }\
    nexus_get_int(buffer, &node_num, 1);\
    if ((node_num < 0) || (node_num >= epd->nodes.num_nodes))\
    {\
        nexus_printf(#func_name "(): illegal node_num %i\n", node_num);\
	return;\
    }\
    nd = EPD_GetNodePtr(epd, node_num);\
    if (nd->node_type != right_type) \
    { \
        nexus_printf(#func_name "(): node_num %i is of type %i, expecting %i\n", \
	             node_num, nd->node_type, right_type); \
    }
    
/*
 * -- HANDLER_CHECK_MSG --
 *
 * make sure the message is valid
 */

#define HANDLER_CHECK_MSG(func_name, msg)\
     if (MessageConsistent(msg) == 0)\
     {\
	 nexus_printf(#func_name "(): message not consistent\n");\
	 return;\
     }

/*
 * -- result_handler() --
 *
 * Returns status and result to a master node.
 *	o make sure the result is from the handler the master is waiting for
 *	o get the status and result from the buffer
 *	o signal the master thread
 */

static void result_handler(nexus_endpoint_t *endpoint,
			   nexus_buffer_t *buffer,
			   nexus_bool_t is_non_threaded_handler)
{
    int wait_op, status, result;
    
    HANDLER_CHECK_FIRST(result_handler, NT_MASTER);
    
#ifdef TRC_result_h
    nexus_printf("result_handler(): hello\n");
#endif

    nexus_get_int(buffer, &wait_op, 1);    
    nexus_get_int(buffer, &status, 1);
    nexus_get_int(buffer, &result, 1);

#ifdef DBG_result_h
    nexus_printf("result_handler(): node = %i, id = %i, stat = %i, res = %i\n",
		 node_num, wait_op, nd->status, nd->result);
#endif

#ifdef TRC_NEG_RESULT
    if (nd->status == 0)
	nexus_printf("result_handler(): status = 0: node %i, id %i, res %i\n",
		     node_num, wait_op, nd->result);
#endif

    TN_SignalResult(nd, wait_op, status, result);

#ifdef DBG_result_h
    nexus_printf("result_handler(): done\n");
#endif
    
    return;
}
/* -- result_handler() -- */

/*
 * -- connect_handler() --
 *
 * Handle a master's request to allocate a slave.
 *	o get the master's node_num from the buffer
 *	o reply with result = -1 if can't allocate a slave node
 *	o point the slave node to the master
 *	o reply with result = slave's node_num
 */

static void connect_handler(nexus_endpoint_t *endpoint,
			    nexus_buffer_t *buffer,
			    nexus_bool_t is_non_threaded_handler)
{
    nexus_startpoint_t	startpoint;
    ep_data_t		*epd;
    test_node_t		*nd;
    int			l_node_num; /* local (slave) */
    int			r_node_num; /* remote (master) */
    
#ifdef TRC_connect_h
    nexus_printf("connect_handler(): hello\n");
#endif

    nexus_get_int(buffer, &r_node_num, 1);
#ifdef DBG_connect_h
    nexus_printf("connect_handler(): request from node %i\n", r_node_num);
#endif
    nexus_get_startpoint(buffer, &startpoint, 1);

    /* get the ep_data */
    epd = (ep_data_t*)nexus_endpoint_get_user_pointer(endpoint);
    if (epd == NULL)
    {
	nexus_printf("connect_handler(): epd == NULL\n");
	return;
    }

    /* try to allocate a test_node */
    if (EPD_AllocNode(epd, NT_SLAVE, &l_node_num) == 0)
    {
	/* failed, send result */
	nexus_printf("connect_handler(): AllocNode failed\n");
	SendResult(&startpoint, r_node_num,
		   CONNECT_HANDLER_ID, 1, -1,
		   is_non_threaded_handler);
	return;
    }

#ifdef DBG_connect_h
    nexus_printf("connect_handler(): got node %i\n", l_node_num);
#endif
    /* point this slave to its master */
    nd = EPD_GetNodePtr(epd, l_node_num);
    nd->node_num = r_node_num;
    nd->startpoint = startpoint;

    /* send result (index to slave) */
    SendResult(&startpoint, r_node_num,
	       CONNECT_HANDLER_ID, 1, l_node_num,
	       is_non_threaded_handler);
}
/* -- connect_handler() -- */

/*
 * -- disconnect_handler() --
 *
 * Called by a master when it wants to free a slave.
 *	o get reply info from the node (startpoint, node_num)
 *	o free the node
 *	o send the reply
 *	o destroy the startpoint
 */

static void disconnect_handler(nexus_endpoint_t *endpoint,
			       nexus_buffer_t *buffer,
			       nexus_bool_t is_non_threaded_handler)
{
    nexus_startpoint_t	startpoint;
    int			m_node_num;

    HANDLER_CHECK_FIRST(disconnect_handler, NT_SLAVE);

#ifdef TRC_disconnect_h
    nexus_printf("disconnect_handler(): hello\n");
#endif

    /* get info so can reply to master */
    startpoint = nd->startpoint;
    m_node_num = nd->node_num;
    
    /* free the node */
    EPD_FreeNode(epd, NT_SLAVE, node_num);

    /* notify the master */
    SendResult(&startpoint, m_node_num,
	       DISCONNECT_HANDLER_ID, 1, 1,
	       is_non_threaded_handler);

    /* destroy the startpoint to the master */
    nexus_startpoint_destroy(&startpoint);
}
/* -- disconnect_handler() -- */

/*
 * -- init_msg_handler() --
 *
 * Initialize a slave's message.
 *	o get the message number from the buffer
 *	o initialize the message
 *	o send reply
 */

static void init_msg_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler)
{
    int		m, s;

    HANDLER_CHECK_FIRST(init_msg_handler, NT_SLAVE);
    
#ifdef TRC_init_msg_h
    nexus_printf("init_msg_handler(): hello\n");
#endif

    nexus_get_int(buffer, &m, 1);

    s = MessageInit(&(nd->message[m]));

    SendResult(&(nd->startpoint), nd->node_num,
	       INIT_MSG_HANDLER_ID, s, 0,
	       is_non_threaded_handler);
}
/* -- init_msg_handler() -- */

/*
 * -- kill_msg_handler() --
 *
 * Destroy a slave's message.
 *	o get message number from buffer
 *	o destroy message
 *	o send reply
 */

static void kill_msg_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler)
{
    int status, r;

    HANDLER_CHECK_FIRST(kill_msg_handler, NT_SLAVE);

#ifdef TRC_kill_msg_h
    nexus_printf("kill_msg_handler(): hello\n");
#endif
    
    nexus_get_int(buffer, &r, 1);

    status = MessageDestroy(&(nd->message[r]));

    SendResult(&(nd->startpoint), nd->node_num,
	       KILL_MSG_HANDLER_ID, status, 0,
	       is_non_threaded_handler);
}
/* -- kill_msg_handler() -- */

/*
 * -- malloc_data_handler() --
 *
 * Malloc memory to hold message's data.
 *	o get message_num from buffer
 *	o malloc memory
 *	o send result
 */

static void malloc_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler)
{
    int status, m;

    HANDLER_CHECK_FIRST(malloc_data_handler, NT_SLAVE);

#ifdef TRC_malloc_data_h
    nexus_printf("malloc_data_handler(): hello\n");
#endif
    
    nexus_get_int(buffer, &m, 1);

    status = MessageDataMalloc(&(nd->message[m]));

    SendResult(&(nd->startpoint), nd->node_num,
	       MALLOC_DATA_HANDLER_ID, status, 0,
	       is_non_threaded_handler);
}
/* -- malloc_data_handler() -- */

/*
 * -- free_data_handler() --
 *
 * Free a message's data memory.
 *	o get message num from buffer
 *	o free the memory
 *	o send result
 */

static void free_data_handler(nexus_endpoint_t *endpoint,
			      nexus_buffer_t *buffer,
			      nexus_bool_t is_non_threaded_handler)
{
    int status, m;

    HANDLER_CHECK_FIRST(free_data_handler, NT_SLAVE);

#ifdef TRC_free_data_h
    nexus_printf("free_data_handler(): hello\n");
#endif
    
    nexus_get_int(buffer, &m, 1);

    status = MessageDataFree(&(nd->message[m]));

    SendResult(&(nd->startpoint), nd->node_num,
	       FREE_DATA_HANDLER_ID, status, 0,
	       is_non_threaded_handler);
}
/* -- free_data_handler() -- */

/*
 * -- cmp_type_handler() --
 *
 * Compare the types of two messages.
 *	o get the control and test message nums
 *	o compare the types
 *	o send the result
 */

static void cmp_type_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler)
{
    int status, result;
    int c, t;

    HANDLER_CHECK_FIRST(cmp_type_handler, NT_SLAVE);

#ifdef TRC_cmp_type_h
    nexus_printf("cmp_type_handler(): hello\n");
#endif
    
    nexus_get_int(buffer, &c, 1);
    nexus_get_int(buffer, &t, 1);
    
    status = CheckMessageType(&(nd->message[c]), &(nd->message[t]), &result);
    
    SendResult(&(nd->startpoint), nd->node_num,
	       CMP_TYPE_HANDLER_ID, status, result,
	       is_non_threaded_handler);
}
/* -- cmp_type_handler() -- */

/*
 * -- cmp_data_handler() --
 *
 * Compare the data of two messages.
 *	o get the control and test message nums
 *	o compare the data
 *	o send the result
 */

static void cmp_data_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler)
{
    int status, result;
    int c, t;

    HANDLER_CHECK_FIRST(cmp_data_handler, NT_SLAVE);

#ifdef TRC_cmp_data_h
    nexus_printf("cmp_data_handler(): hello\n");
#endif
    
    nexus_get_int(buffer, &c, 1);
    nexus_get_int(buffer, &t, 1);

    status = CheckMessageData(&(nd->message[c]), &(nd->message[t]),
			      &result, NULL);

    SendResult(&(nd->startpoint), nd->node_num,
	       CMP_DATA_HANDLER_ID, status, result,
	       is_non_threaded_handler);
}
/* -- cmp_data_handler() -- */

/*
 * -- s_send_type_handler() --
 *
 * Tell the slave to send the type of a message back to the master.
 *	o get the message num from the buffer
 *	o calculate the size of the reply buffer
 *	o put the destination node_num into the buffer
 *	o put the message type into the buffer
 *	o send the buffer
 */

static void s_send_type_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler)
{
    nexus_buffer_t	reply_buf;
    int			m;
    int			msg_sz;
    
    HANDLER_CHECK_FIRST(s_send_type_handler, NT_SLAVE);

    nexus_get_int(buffer, &m, 1);

#ifdef TRC_s_send_type_h
    nexus_printf("s_send_type_handler(): node %i, msg %i requested\n",
		 node_num, m);
#endif
    
    /* compute the size of the message */
    if (MessageGetTypeSize(&(nd->message[m]), &msg_sz) == 0)
    {
	nexus_printf("s_send_type_handler(): GetTypeSize failed\n");
	return;
    }

    /* init buffer */
    if (nexus_buffer_init(&reply_buf, msg_sz + nexus_sizeof_int(1), 0) !=
	NEXUS_SUCCESS)
    {
	nexus_printf("s_send_type_handler(): buffer_init failed\n");
	return;
    }

    /* accounting */
    ReportAddType(&(nd->report), msg_sz + nexus_sizeof_int(1));

    /* put the message into buffer */
    nexus_put_int(&reply_buf, &(nd->node_num), 1);
    PutMessageType(&reply_buf, &(nd->message[m]));

    /* send the message */
    if (nexus_send_rsr(&reply_buf, &(nd->startpoint),
		       M_RECV_TYPE_HANDLER_ID,
		       NEXUS_TRUE, is_non_threaded_handler) != NEXUS_SUCCESS)
    {
	nexus_printf("s_send_type_handler_id(): send_rsr failed\n");
	return;
    }

#ifdef DBG_s_send_type_h
    nexus_printf("s_send_type_handler(): reply rsr sent\n");
#endif
    
    return;
}
/* -- s_send_type_handler() -- */

/*
 * -- m_recv_type_handler() --
 *
 * Receive the message type sent by the slave.
 *	o lock the mutex
 *	o make sure the master is expecting
 *	o get the message type (always into message[0])
 *	o signal the master
 *	o unlock mutex
 */

static void m_recv_type_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler)
{
    int status;

    HANDLER_CHECK_FIRST(m_recv_type_handler, NT_MASTER);

#ifdef TRC_m_recv_type_h
    nexus_printf("m_recv_type_handler(): hello\n");
#endif

    status = GetMessageType(buffer, &(nd->message[0]));

    TN_SignalResult(nd, S_SEND_TYPE_HANDLER_ID, status, 0);

#ifdef DBG_m_recv_type_h
    nexus_printf("m_recv_type_handler(): done\n");
#endif
    
    return;
}
/* -- m_recv_type_handler() -- */

/*
 * -- s_recv_type_handler() --
 *
 * Receive message type sent by the master.
 *	o get the message type (always into message[0])
 *	o sent the result
 */

static void s_recv_type_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler)
{
    int status;

    HANDLER_CHECK_FIRST(s_recv_type_handler, NT_SLAVE);

#ifdef TRC_s_recv_type_h
    nexus_printf("s_recv_type_handler(): hello\n");
#endif
    
    status = GetMessageType(buffer, &(nd->message[0]));
    
    SendResult(&(nd->startpoint), nd->node_num,
	       S_RECV_TYPE_HANDLER_ID, status, 0,
	       is_non_threaded_handler);
    
    return;
}
/* -- s_recv_type_handler() -- */

/*
 * -- s_send_data_handler() --
 *
 * Tell a slave to sent data back to the master.
 *	o get the message num from the buffer
 *	o calculate the reply buffer size
 *	o put the master's node_num in the buffer
 *	o put the data in the buffer
 *	o send to the master
 */

static void s_send_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler)
{
    nexus_buffer_t	reply_buf;
    int			status, m;
    int			msg_sz;

    HANDLER_CHECK_FIRST(s_send_data_handler, NT_SLAVE);

#ifdef TRC_s_send_data_h
    nexus_printf("s_send_data_handler(): hello\n");
#endif
    
    nexus_get_int(buffer, &m, 1);

    if (MessageGetDataSize(&(nd->message[m]), &msg_sz) == 0)
    {
	nexus_printf("s_send_data_handler(): GetDataSize failed\n");
	return;
    }

    if (nexus_buffer_init(&reply_buf, nexus_sizeof_int(1) + msg_sz, 0) !=
	NEXUS_SUCCESS)
    {
	nexus_printf("s_send_data_handler(): buffer_init failed\n");
	return;
    }

    ReportAddData(&(nd->report), msg_sz + nexus_sizeof_int(1));
    
    nexus_put_int(&reply_buf, &(nd->node_num), 1);    
    if (PutMessageData(&reply_buf, &(nd->message[m])) == 0)
    {
	nexus_printf("s_send_data_handler(): PutMessage failed\n");
	return;
    }

    if (nexus_send_rsr(&reply_buf, &(nd->startpoint),
		       M_RECV_DATA_HANDLER_ID,
		       NEXUS_TRUE, is_non_threaded_handler) != NEXUS_SUCCESS)
    {
	nexus_printf("s_send_data_handler(): send_rsr failed\n");
	return;
    }
    
    return;
}
/* -- s_send_data_handler() -- */

/*
 * -- m_recv_data_handler() --
 *
 * Receive the data sent by the slave.
 *	o lock the mutex
 *	o make sure the master is waiting for data
 *	o get the data (always into message[0])
 *	o signal the master
 *	o unlock the mutex
 */

static void m_recv_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler)
{
    int status;
    
    HANDLER_CHECK_FIRST(m_recv_data_handler, NT_MASTER);

#ifdef TRC_m_recv_data_h
    nexus_printf("m_recv_data_handler(): hello\n");
#endif

    status = GetMessageData(buffer, &(nd->message[0]));

    TN_SignalResult(nd, S_SEND_DATA_HANDLER_ID, status, 0);

#ifdef DBG_m_recv_data_h
    nexus_printf("m_recv_data_handler(): locked mutex\n");
#endif
}
/* -- m_recv_data_handler() -- */

/*
 * -- s_recv_data_handler() --
 *
 * Receive data sent by the master.
 *	o get the data (always into message[0])
 *	o send a reply
 */

static void s_recv_data_handler(nexus_endpoint_t *endpoint,
				nexus_buffer_t *buffer,
				nexus_bool_t is_non_threaded_handler)
{
    int status, r;

    HANDLER_CHECK_FIRST(s_recv_data_handler, NT_SLAVE);
    
#ifdef TRC_s_recv_data_h
    nexus_printf("s_recv_data_handler(): hello\n");
#endif
    
    status = GetMessageData(buffer, &(nd->message[0]));
    SendResult(&(nd->startpoint), nd->node_num,
	       S_RECV_DATA_HANDLER_ID, status, 0,
	       is_non_threaded_handler);
    return;
}
/* -- s_recv_data_handler() -- */

/********************************************
 *                                          *
 ********************************************/

/*
 * -- gen_type_handler() --
 *
 * Service a request for a slave to generate the message type.
 *	o get the message_num, gen_type, and x
 *	o generate the type
 *	o return the result
 */

static void gen_type_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler)
{
    int status, m, g, x;

    HANDLER_CHECK_FIRST(gen_type_handler, NT_SLAVE);
    
#ifdef TRC_gen_type_h
    nexus_printf("gen_type_handler(): hello\n");
#endif

    nexus_get_int(buffer, &m, 1);
    nexus_get_int(buffer, &g, 1);
    nexus_get_int(buffer, &x, 1);

    status = UserGenType(&(nd->message[m]), g, x);
    
    SendResult(&(nd->startpoint), nd->node_num,
	       GEN_TYPE_HANDLER_ID, status, 0,
	       is_non_threaded_handler);

    return;
}
/* -- gen_type_handler() -- */

/*
 * -- gen_data_handler() --
 *
 * Service a request for a slave to generate data.
 *	o get the message_num, gen_type, and x
 *	o generate the data
 *	o return the result
 */

static void gen_data_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler)
{
    int status, m, g, x;

    HANDLER_CHECK_FIRST(gen_data_handler, NT_SLAVE);

#ifdef TRC_gen_data_h
    nexus_printf("gen_data_handler(): hello\n");
#endif
    
    nexus_get_int(buffer, &m, 1);
    nexus_get_int(buffer, &g, 1);
    nexus_get_int(buffer, &x, 1);

    status = UserGenData(&(nd->message[m]), g, x);

    SendResult(&(nd->startpoint), nd->node_num,
	       GEN_DATA_HANDLER_ID, status, 0,
	       is_non_threaded_handler);

    return;
}
/* -- gen_data_handler() -- */

/*
 * -- shutdown_context_handler() --
 *
 * Service a request to destroy the context.
 */

static void shutdown_context_handler(nexus_endpoint_t *endpoint,
				     nexus_buffer_t *buffer,
				     nexus_bool_t is_non_threaded_handler)
{
#ifdef TRC_shutdown_context_h
    nexus_printf("shutdown_context_handler(): goodbye!\n");
#endif
   
    nexus_context_destroy(is_non_threaded_handler);

    return;
}
/* -- shutdown_context_handler() -- */

/*
 * -- stop_waiting_handler() --
 *
 * Signal a waiting thread.
 *	o lock mutex
 *	o signal cond
 *	o unlock mutex
 */

void stop_waiting_handler(nexus_endpoint_t *endpoint,
			  nexus_buffer_t *buffer,
			  nexus_bool_t is_non_threaded_handler)
{
    ep_data_t	*epd;

    epd = nexus_endpoint_get_user_pointer(endpoint);
    if (epd == NULL)
    {
	nexus_printf("stop_waiting_handler(): epd == NULL\n");
	return;
    }

    EPD_StopWaiting(epd);

    return;
}
/* -- stop_waiting_handler() -- */

/*
 * -- join_group_handler() --
 *
 * Connect to a group leader with a list of startpoints to run tests on.
 *	o make sure endpoint is a leader
 *	o get the number of startpoints
 *	o get the startpoints
 */

static void join_group_handler(nexus_endpoint_t *endpoint,
			       nexus_buffer_t *buffer,
			       nexus_bool_t is_non_threaded_handler)
{
    nexus_startpoint_t	*startpoint;
    ep_data_t		*epd;    
    int			num_sps;
    int			i;
    
    epd = nexus_endpoint_get_user_pointer(endpoint);
    if (epd == NULL)
    {
	nexus_printf("join_group_handler(): epd = NULL\n");
	return;
    }
    if (epd->group.am_leader == 0)
    {
	nexus_printf("join_group_handler(): not a leader\n");
	return;
    }
    /* get # SPs */
    nexus_get_int(buffer, &num_sps, 1);
    nexus_printf("join_group_handler(): getting %i SP's\n", num_sps);
    /* allocate room to keep them */
    startpoint = AllocFollowerSPs(epd, num_sps);
    /* get SPs */
    nexus_get_startpoint(buffer, startpoint, num_sps);
    /* step through SPs so they can be seen with debugger */
    for(i = 0; i < num_sps; i++)
	startpoint = GetFollowerSP(epd, num_sps);
    /* notify main thread */
    GotFollowers(epd);
    nexus_printf("join_group_handler(): bye\n");
}
/* -- join_group_handler() -- */

/*
 * -- run_test_handler() --
 *
 * Connect to the given startpoint and run some tests.
 *	o get the test_id, reps, verbose level, slaveSP and leaderSP
 *	o queue the test
 */

static void run_test_handler(nexus_endpoint_t *endpoint,
			     nexus_buffer_t *buffer,
			     nexus_bool_t is_non_threaded_handler)
{
    test_info_t	*ti;
    ep_data_t	*epd;

    epd = nexus_endpoint_get_user_pointer(endpoint);
    if (epd == NULL)
    {
	nexus_printf("run_test_handler(): epd = NULL\n");
	return;
    }
    
    nexus_printf("run_test_handler(): hello\n");
    
    ti = (test_info_t *)nexus_malloc(sizeof(test_info_t));
    if (ti == NULL)
    {
	nexus_printf("run_test_handler(): malloc failed\n");
	return;
    }

    /* get test_id, reps, verbose, slaveSP, & leaderSP */
    nexus_get_int(buffer, &(ti->test_id), 1);
    nexus_get_int(buffer, &(ti->reps), 1);
    nexus_get_int(buffer, &(ti->verb), 1);
    nexus_get_startpoint(buffer, &(ti->slaveSP), 1);
    nexus_get_startpoint(buffer, &(ti->leaderSP), 1);
    
    EPD_NQ_Test(epd, ti);
    
    nexus_printf("run_test_handler(): bye\n");
}
/* -- run_test_handler() -- */

/*
 * -- report_handler() --
 *
 * Get the statistics from a test that's been run.
 *	o get test_id and startpoint
 *	o get the report and add to total
 */

static  void report_handler(nexus_endpoint_t *endpoint,
				 nexus_buffer_t *buffer,
				 nexus_bool_t is_non_threaded_handler)
{
    int			test_id;
    nexus_startpoint_t	*startpoint;
    ep_data_t		*epd;
    report_t		report;

    nexus_printf("report_handler(): hello\n");
    epd = nexus_endpoint_get_user_pointer(endpoint);
    if (epd == NULL)
    {
	nexus_printf("report_handler(): epd = NULL\n");
	return;
    }
    if (epd->group.am_leader == 0)
    {
	nexus_printf("report_handler(): not a leader\n");
	return;
    }
    nexus_get_int(buffer, &test_id, 1);
    startpoint = GetFollowerSP(epd, test_id);
    if (startpoint == NULL)
    {
	nexus_printf("report_handler(): FollowerSP = NULL\n");
	return;
    }
    if (nexus_startpoint_is_null(startpoint) == 0)
    {
	nexus_printf("report_handler(): startpoint not null\n");
	return;
    }
    nexus_get_startpoint(buffer, startpoint, 1);
    ReportInit(&report);
    GetReport(buffer, &report);

    /* Check in, signal if last one */
    nexus_mutex_lock(&(epd->group.mutex));
    epd->group.num_left--;
    if (epd->group.num_left == 0)
    {
	nexus_mutex_lock(&(epd->wait.mutex));
	epd->wait.done++;
	epd->wait.stop_waiting = 1;
	nexus_printf("stop_waiting = 1\n");
	nexus_cond_signal(&(epd->wait.cond));
	nexus_mutex_unlock(&(epd->wait.mutex));
    }
    nexus_mutex_unlock(&(epd->group.mutex));
    
    ReportDone(&report);
}
/* -- report_handler() -- */
