/*
 * group.c
 */

#include "group.h"
#include "matches.h"
#include "handlers.h"

#include "everything.h"

#define TRC_LeaderInit
#define TRC_LeaderDone
#define TRC_JoinGroup
#define TRC_AllocFollowerSPs
#define DBG_AllocFollowerSPs
#define TRC_GotFollowers
#define DBG_GotFollowers

/*
 * -- LeaderInit() --
 *
 * Initialize endpoint to be a group leader.
 *	o init mutex and cond
 *	o malloc for lists of follower startpoints
 *	o init counts of eps, sps
 */
int LeaderInit(ep_data_t *epd, int max_eps)
{
#ifdef TRC_LeaderInit
    nexus_printf("LeaderInit(): hello\n");
#endif
    if (epd->group.am_leader)
    {
	nexus_printf("LeaderInit(): already am leader\n");
	return(0);
    }
    if (nexus_mutex_init(&(epd->group.mutex), NULL) != NEXUS_SUCCESS)
    {
	nexus_printf("LeaderInit(): mutex_init failed\n");
	return(0);
    }
    epd->group.sps_per_ep = (int*)nexus_malloc(max_eps * sizeof(int));
    if (epd->group.sps_per_ep == NULL)
    {
	nexus_printf("LeaderInit(): malloc(ints) failed\n");
	return(0);
    }
    epd->group.follower = (nexus_startpoint_t**)
	                  nexus_malloc(max_eps * sizeof(nexus_startpoint_t*));
    if (epd->group.follower == NULL)
    {
	nexus_printf("LeaderInit(): malloc(startpoint**) failed\n");
	return(0);
    }
    ReportInit(&(epd->group.report));
    epd->group.num_left = max_eps;
    epd->group.num_eps = 0;
    epd->group.num_sps = 0;
    epd->group.am_leader = 1;
    return(1);
}
/* -- LeaderInit() -- */

/*
 * -- LeaderDone() --
 *
 * Clean up after group leader.
 *	o destroy condition
 *	o destroy follower startpoints
 *	o free follower lists
 *	o destroy mutex
 */
int LeaderDone(ep_data_t *epd)
{
    int i, ok;

#ifdef TRC_LeaderDone
    nexus_printf("LeaderDone(): hello\n");
#endif
    ok = 1;
    if (epd->group.am_leader == 0)
    {
	nexus_printf("LeaderDone(): not a leader\n");
	ok = 0;
    }
    nexus_mutex_lock(&(epd->group.mutex));

    for (i = 0; i < epd->group.num_sps; i++)
	nexus_startpoint_destroy(GetFollowerSP(epd, i));
    for (i = 0; i < epd->group.num_eps; i++)
    {
	nexus_free(epd->group.follower[i]);
    }
    nexus_free(epd->group.follower);
    epd->group.follower = NULL;
    nexus_free(epd->group.sps_per_ep);
    epd->group.sps_per_ep = NULL;
    epd->group.am_leader = 0;
    nexus_mutex_unlock(&(epd->group.mutex));
    if (nexus_mutex_destroy(&(epd->group.mutex)) != NEXUS_SUCCESS)
    {
	nexus_printf("LeaderDone(): mutex_destroy failed\n");
	ok = 0;
    } 
    return(ok);
} 
/* -- LeaderDone() -- */

/*
 * -- WaitForFollowers() --
 *
 * Wait on condition until all followers have reported in.
 */
int WaitForFollowers(ep_data_t *epd)
{
    nexus_mutex_lock(&(epd->wait.mutex));
    while (epd->wait.stop_waiting == 0)
    {
	nexus_cond_wait(&(epd->wait.cond), &(epd->wait.mutex));
    }
    epd->wait.stop_waiting = 0;
    nexus_printf("stop_waiting = 0\n");
    nexus_mutex_unlock(&(epd->wait.mutex));
    return(1);
}
/* -- WaitForFollowers() -- */

/*
 * -- JoinGroup() --
 */
int JoinGroup(nexus_endpoint_t *ep, int n_nodes, nexus_node_t *nodes,
	      nexus_startpoint_t *leader)
{
    nexus_buffer_t	buffer;
    int			buf_size;
    nexus_startpoint_t	*sp;
    int			i;
    nexus_startpoint_t	*src_sp;
    nexus_startpoint_t	*dst_sp;

#ifdef TRC_JoinGroup
    nexus_printf("JoinGroup(): hello\n");
#endif
    if (leader == NULL)
    {
	nexus_printf("JoinGroup(): leaderSP == NULL\n");
	return(0);
    }
    else
	nexus_printf("JoinGroup(): leaderSP != NULL\n");
    if (nexus_startpoint_is_null(leader))
    {
	nexus_printf("JoinGroup(): leaderSP is null\n");
	return(0);
    }
    else
	nexus_printf("JoinGroup(): leaderSP is not null\n");

    /* Make a list of SPs to be sent */
    sp = (nexus_startpoint_t *)nexus_malloc(n_nodes *
					    sizeof(nexus_startpoint_t));
    if (sp == NULL)
	nexus_printf("JoinGroup(): malloc failed\n");
    else
	nexus_printf("JoinGroup(): malloc'd memory\n");
    /* sp[0] is the master context */
    dst_sp = &(sp[0]);
    nexus_startpoint_bind(dst_sp, ep);
    nexus_printf("JoinGroup(): Startpoint bound, %i nodes\n", n_nodes);
    /* sp[1..N] are slave contexts */
    for (i = 1; i < n_nodes; i++)
    {
	dst_sp = &(sp[i]);
	src_sp = &(nodes[i].startpoint);
	nexus_startpoint_copy(dst_sp, src_sp);
    }
    nexus_printf("JoinGroup(): startpoints copied\n");
    /* buffer_init */
    nexus_buffer_init(&buffer, nexus_sizeof_startpoint(sp, n_nodes) +
		      nexus_sizeof_int(1), 0);
    nexus_printf("JoinGroup(): buffer init'd\n");
    /* put num_startpoints */
    nexus_put_int(&buffer, &n_nodes, 1);
    /* put startpoints */
    nexus_put_startpoint_transfer(&buffer, sp, n_nodes);
    nexus_printf("JoinGroup(): RSR ready to go\n");
    /* send_rsr */
    nexus_send_rsr(&buffer,
		   leader,
		   JOIN_GROUP_HANDLER_ID,
		   NEXUS_TRUE, NEXUS_FALSE);
    nexus_printf("JoinGroup(): RSR sent\n");
    return(1);
}
/* -- JoinGroup() -- */

/*
 * -- BasicGroupTest() --
 */
int BasicGroupTest(nexus_endpoint_t *ep, int verb, int reps, report_t *report)
{
    int		i, j, n, l;
    match_t	*m;
    tourn_t	t;
    ep_data_t	*epd;

    nexus_printf("BasicGroupTest(): hello\n");
    
    epd = nexus_endpoint_get_user_pointer(ep);

    n = epd->group.num_sps;
    if (n < 2)
    {
	nexus_printf("BasicGroupTest(): can't test with only %i nodes\n", n);
	return(0);
    }
    
    InitTourn(&t, n);
    nexus_stdio_lock();
    PrintTourn(&t);
    nexus_stdio_unlock();

    for (i = 0; i < t.rounds; i++)
    {
	l = MatchesThisRound(&t, i);
	SetNumPairs(epd, l);
	nexus_printf("=== Fwd Round %i, %i matches ===\n", i, l);
	for (j = 0; j < t.mpr; j++)
	{
	    m = GetMatch(&t, i, j);
	    if (m->t1 != -1)
		ScheduleTest(ep, m->t1, m->t2, verb, reps);
	}
	WaitForTests(epd);
    }
    for (i = 0; i < t.rounds; i++)
    {
	l = MatchesThisRound(&t, i);
	nexus_printf("=== Rev Round %i, %i matches ===\n", i, l);
	SetNumPairs(epd, l);
	for (j = 0; j < t.mpr; j++)
	{
	    m = GetMatch(&t, i, j);
	    if (m->t1 != -1)
		ScheduleTest(ep, m->t2, m->t1, verb, reps);
	}
	WaitForTests(epd);
    }
    return(1);    
}
/* -- BasicGroupTest() -- */

/*
 * -- SetNumPairs() --
 */
int SetNumPairs(ep_data_t *epd, int n)
{
    epd->group.num_left = n;
}
/* -- SetNumPairs() -- */

/*
 * -- ScheduleTest() --
 */
int ScheduleTest(nexus_endpoint_t *ep,
		 int master, int slave,
		 int verb, int reps)
{
    nexus_buffer_t	buffer;
    int			buf_size;
    nexus_startpoint_t	leader_sp;
    nexus_startpoint_t	*master_sp;
    nexus_startpoint_t	*slave_sp;
    ep_data_t		*epd;
    nexus_printf("ScheduleTest(): master = %i, slave = %i\n", master, slave);
    epd = nexus_endpoint_get_user_pointer(ep);    

    nexus_startpoint_bind(&leader_sp, ep);
    master_sp = GetFollowerSP(epd, master);
    slave_sp = GetFollowerSP(epd, slave);
    
    buf_size = nexus_sizeof_startpoint(slave_sp, 1);
    buf_size += nexus_sizeof_startpoint(&leader_sp, 1);
    buf_size += 3 * nexus_sizeof_int(1);
    nexus_buffer_init(&buffer, buf_size, 0);
    /* test_id, reps, verbose, slaveSP, leaderSP */
    nexus_put_int(&buffer, &slave, 1);
    nexus_put_int(&buffer, &reps, 1);
    nexus_put_int(&buffer, &verb, 1);
    nexus_put_startpoint_transfer(&buffer, slave_sp, 1);
    nexus_put_startpoint_transfer(&buffer, &leader_sp, 1);
/*    nexus_printf("ScheduleTest(): RTG\n"); */
    nexus_send_rsr(&buffer,
		   master_sp,
		   RUN_TEST_HANDLER_ID,
		   NEXUS_TRUE, NEXUS_FALSE);
    nexus_printf("ScheduleTest(): bye\n");
    return(1);
}
/* -- ScheduleTest() -- */

/*
 * -- WaitForTests() --
 */
int WaitForTests(ep_data_t *epd)
{
    nexus_printf("WaitForTests(): hello\n");
    EPD_MainLoop(epd);
    nexus_printf("WaitForTests(): bye\n");
    return(1);
}
/* -- WaitForTests() -- */

/*
 * -- AllocFollowerSPs() --
 *
 * Create room for more followers.
 *	o malloc memory
 *	o refigure counts
 *	o return pointer to first new startpoint
 */
nexus_startpoint_t *AllocFollowerSPs(ep_data_t *epd, int num_sps)
{
    nexus_startpoint_t	*startpoint;
#ifdef TRC_AllocFollowerSPs
    nexus_printf("AllocFollowerSPs(): hello\n");
#endif
    /* make sure really am a leader */
    if (epd->group.am_leader == 0)
    {
	nexus_printf("AllocFollowerSPs(): not a leader\n");
	return(NULL);
    }
    /* lock the group mutex */
    nexus_mutex_lock(&(epd->group.mutex));
    if (epd->group.num_left <= 0)
    {
	nexus_printf("AllocFollowerSPs(): not expecting more\n");
	nexus_mutex_unlock(&(epd->group.mutex));
	return(NULL);
    }
    /* make sure leader_init did its job */
    if (epd->group.sps_per_ep == NULL)
	nexus_printf("AllocFollowerSPs(): sps_per_ep = NULL\n");
    if (epd->group.follower == NULL)
	nexus_printf("AllocFollowerSPs(): follower = NULL\n");
#ifdef DBG_AllocFollowerSPs
    nexus_printf("AllocFollowerSPs(): checks OK\n");
#endif
    /* save # new sps, alloc memory */
    epd->group.sps_per_ep[epd->group.num_eps] = num_sps;
    epd->group.follower[epd->group.num_eps] = (nexus_startpoint_t*)
	nexus_malloc(num_sps * sizeof(nexus_startpoint_t));
    if (epd->group.follower[epd->group.num_eps] == NULL)
    {
	nexus_printf("AllocFollowerSPs(): malloc failed\n");
	nexus_mutex_unlock(&(epd->group.mutex));
	return(NULL);
    }
    /* return pointer to first allocated startpoint */
    startpoint = epd->group.follower[epd->group.num_eps];
    /* update totals */
    epd->group.num_eps += 1;
    epd->group.num_sps += num_sps;
    
    nexus_mutex_unlock(&(epd->group.mutex));
#ifdef DBG_AllocFollowerSPs
    nexus_printf("AllocFollowerSPs(): bye\n");
#endif
    return (startpoint);
}
/* -- AllocFollowerSPs() -- */

/*
 * -- GotFollowers() --
 *
 * Notify endpoint that previously alloc'd followers
 * have been filled in.
 *	o num_left to wait for decreases
 *	o signal condition if no more to wait for
 */
void GotFollowers(ep_data_t *epd)
{
#ifdef TRC_GotFollowers
    nexus_printf("GotFollowers(): hello\n");
#endif
    if (epd->group.am_leader == 0)
    {
	nexus_printf("GotFollowers(): not a leader\n");
	return;
    }
    nexus_mutex_lock(&(epd->group.mutex));
    epd->group.num_left--;
    if (epd->group.num_left == 0)
    {
#ifdef DBG_GotFollowers
	nexus_printf("GotFolowers(): last ones\n");
#endif
	/* signal main thread */
	nexus_mutex_lock(&(epd->wait.mutex));
	epd->wait.done++;
	epd->wait.stop_waiting = 1;
	nexus_printf("stop_waiting = 1\n");
	nexus_cond_signal(&(epd->wait.cond));
	nexus_mutex_unlock(&(epd->wait.mutex));
    }
    nexus_mutex_unlock(&(epd->group.mutex));
}
/* -- GotFollowers() -- */

/*
 * -- GetFollowerSP() --
 *
 * Step through lists of followers to find the n'th one.
 */
nexus_startpoint_t *GetFollowerSP(ep_data_t *epd, int n)
{
    int i = 0;
/*    nexus_printf("GetFollowerSP(%i): hello\n", n); */
    if ((n < 0) || (n >= epd->group.num_sps))
    {
	nexus_printf("GetFollowerSP(): illegal n %i\n", n);
	return(NULL);
    }
    while (n >= epd->group.sps_per_ep[i])
    {
	n -= epd->group.sps_per_ep[i];
	i++;
    }
/*    nexus_printf("GetFolowerSP(): i = %i, n = %i\n", i, n); */
    return(&(epd->group.follower[i][n]));
}
/* -- GetFollowerSP() -- */
