/*
 * epdata.c
 */

#include "epdata.h"
#include "debug.h"
#include "handlers.h"
#include "run_test.h"

/*
 * -- EPD_Create() --
 */
int EPD_Create(ep_data_t **epd, int num_nodes, int verbose)
{
    test_node_t		*nd;
    int			i;
    
    /* allocate memory */
    *epd = (ep_data_t*)nexus_malloc(sizeof(ep_data_t));
    if (*epd == NULL)
	return(0);
    /*   nodes section: */
    (*epd)->nodes.test_node = (test_node_t*)
	nexus_malloc(sizeof(test_node_t) * num_nodes);
    (*epd)->nodes.num_nodes = num_nodes;
    (*epd)->nodes.nodes_used = 0;
    if (nexus_mutex_init(&((*epd)->nodes.mutex), NULL) != NEXUS_SUCCESS)
    {
	nexus_printf("EndpointInit(): nodes.mutex_init failed\n");
	return(0);
    }

    for (i = 0; i < num_nodes; i++)
    {
	nd = EPD_GetNodePtr(*epd, i);
	/* initialize the test node, type = available for use */
	nd->my_num = i;
	nd->node_type = NT_FREE;
    }

    /*   group section: */
    (*epd)->group.am_leader = 0;
    
    /*   wait section: */
    if (nexus_mutex_init(&((*epd)->wait.mutex), NULL) != NEXUS_SUCCESS)
    {
	nexus_printf("EndpointInit(): wait.mutex_inti failed\n");
	return(0);
    }
    if (nexus_cond_init(&((*epd)->wait.cond), NULL) != NEXUS_SUCCESS)
    {
	nexus_printf("EndpointInit(): wait.cond_init failed\n");
	return(0);
    }
    (*epd)->wait.done = 0;
    (*epd)->wait.stop_waiting = 0;
    nexus_printf("stop_waiting = 0\n");
    (*epd)->wait.service_queue = 0;
    (*epd)->wait.context_destroy = 0;

    /* test_Q section: */
    if (nexus_mutex_init(&((*epd)->test_Q.mutex), NULL) != NEXUS_SUCCESS)
    {
	nexus_printf("EndpointInit(): q.mutex_init failed\n");
	return(0);
    }
    (*epd)->test_Q.first = NULL;
    (*epd)->test_Q.last = NULL;
    
    /* set the minimum verbose level for this context */
    (*epd)->verbose = verbose;
    
    /* initialize the report for this context */
    if (ReportInit(&((*epd)->report)) == 0)
    {
	nexus_printf("EndpointInit(): ReportInit failed\n");
	return(0);
    }
}
/* -- EPD_Create() -- */

/*
 * -- EPD_Destroy() --
 */
int EPD_Destroy(ep_data_t **epd)
{
    test_node_t		*nd;
    int			i;

    /* nodes section: */
    for (i = 0; i < (*epd)->nodes.num_nodes; i++)
    {
	/* check to make sure all test_nodes are free */
	nd = &((*epd)->nodes.test_node[i]);
	EPD_FreeNode((*epd), NT_FREE, nd->my_num);
    }
    nexus_mutex_destroy(&((*epd)->nodes.mutex));

    /* wait section: */
    nexus_mutex_destroy(&((*epd)->wait.mutex));
    nexus_cond_destroy(&((*epd)->wait.cond));

    /* queue section: */
    nexus_mutex_destroy(&((*epd)->test_Q.mutex));
    
    ReportDone(&((*epd)->report));
    
    nexus_free(*epd);
    *epd = NULL;
}
/* -- EPD_Destroy() -- */

/*
 * -- EPD_GetNodePtr() --
 */
test_node_t *EPD_GetNodePtr(ep_data_t *epd, int node_num)
{
    if ((node_num < 0) || (node_num >= epd->nodes.num_nodes))
    {
	nexus_printf("EPD_GetNodePtr(): illegal node_num %i\n", node_num);
	return(NULL);
    }
    return(&(epd->nodes.test_node[node_num]));				 
}
/* -- EPD_GetNodePtr() -- */

/*
 * -- EPD_AllocNode() --
 *
 * Allocate a node of some type, initialize for use.
 *	o make sure type is valid
 *	o lock mutex
 *	o get a free node, mark as used
 *	o unlock mutex
 *	o initialize node
 */
int EPD_AllocNode(ep_data_t *epd, int node_type, int *node_num)
{
    int		i;
    test_node_t *nd;

    *node_num = -1;

    /* make sure requested type is valid */
    if ((node_type < 0) || (node_type > NT_MAX))
    {
	nexus_printf("AllocNode(): illegal node_type %i\n", node_type);
	return(0);
    }

    /* make sure not trying to allocate type FREE */
    if (node_type == NT_FREE)
    {
	nexus_printf("EPD_AllocNode(): trying to alloc NT_FREE?\n");
	return(0);
    }
    
#ifdef TRC_AllocNode
    nexus_printf("AllocNode(): hello\n");
#endif

    /* lock the mutex to keep other threads out while looking */
    nexus_mutex_lock(&(epd->nodes.mutex));
    
    if (epd->nodes.nodes_used < epd->nodes.num_nodes)
    {
	/* OK: at least one is still free */
	for (i = 0; i < epd->nodes.num_nodes; i++)
	{
	    /* step through the list until we find the first */
	    nd = EPD_GetNodePtr(epd, i);
	    if (nd->node_type == NT_FREE)
	    {
		/* grab this one and unlock the mutex */
		nd->node_type = node_type;
		nd->wait_op = -1;
		epd->nodes.nodes_used++;
		nexus_mutex_unlock(&(epd->nodes.mutex));

		/* set *node_num, initialize node, and return */
		*node_num = i;
#ifdef DBG_AllocNode
		nexus_printf("AllocNode(): got node %i\n", i);
#endif
		return(TN_Init(nd));
	    }
	}

	/* error: the count said one is still free */
	nexus_mutex_unlock(&(epd->nodes.mutex));
	nexus_printf("AllocNode(): A node should have been free\n");
	return(0);
    }
    else
    {
	/* error: no more left */
	nexus_printf("AllocNode(): All nodes in use\n");
	nexus_mutex_unlock(&(epd->nodes.mutex));
	return(0);
    }
}
/* -- EPD_AllocNode() -- */

/*
 * -- EPD_FreeNode() --
 *
 * Clean up a node's resources and mark as free.
 *	o lock endpoint mutex
 *	o make sure node is valid
 *	o add report to total
 *	o mark as free, add one to free count
 *	o unlock endpoint mutex
 */

int EPD_FreeNode(ep_data_t *epd, int node_type, int node_num)
{
    test_node_t *nd;
    int		ok;

#ifdef TRC_FreeNode
    nexus_printf("FreeNode(): hello\n");
#endif
    ok = 1;

    /* get a pointer to the test_node */
    nd = EPD_GetNodePtr(epd, node_num);
    if (nd == NULL)
    {
	nexus_printf("EPD_FreeNode(): illegal node num\n");
	return(0);
    }

    /* lock the mutex while freeing node */
    nexus_mutex_lock(&(epd->nodes.mutex));
    
    /* print an error if the types is not as expected */
    if (nd->node_type != node_type)
    {
	nexus_printf("FreeNode(): expecting node_type %i, got %i\n",
		     node_type, nd->node_type);
    }

    /* if the node thinks it's free, don't try to free it again */
    if (nd->node_type == NT_FREE)
    {
	nexus_mutex_unlock(&(epd->nodes.mutex));
	return(1);
    }
    
    /* add to the context total report */
    ReportSum(&(epd->report), &(nd->report));

    /* finished with the node */
    TN_Done(nd);

    epd->nodes.nodes_used--;
    nexus_mutex_unlock(&(epd->nodes.mutex));
    return(ok);
}
/* -- EPD_FreeNode() -- */

/*
 * -- EPD_StopWaiting() --
 */
int EPD_StopWaiting(ep_data_t *epd)
{
    nexus_printf("EPD_StopWaiting(): hello\n");
    nexus_mutex_lock(&(epd->wait.mutex));
    epd->wait.stop_waiting = 1;
    nexus_printf("stop_waiting = 1\n");
    epd->wait.done = 1;
    nexus_cond_signal(&(epd->wait.cond));
    nexus_mutex_unlock(&(epd->wait.mutex));
    return(1);
}
/* EPD_StopWaiting() -- */

/*
 * -- ServiceQ() --
 *
 * Grab the first test_info and run the test.
 */
static void ServiceQ(ep_data_t *epd)
{
    nexus_buffer_t	reply_buf;
    report_t		report;
    test_info_t		*ti;

    nexus_printf("ServiceQ(): going to run a test\n");
    
    EPD_DQ_Test(epd, &ti);

    nexus_printf("ServiceQ(): ti->verb = %i, ep->verb = %i\n",
		 ti->verb, epd->verbose);
    
    if (ti->verb < epd->verbose)
	ti->verb = epd->verbose;

    ReportInit(&report);

    if (nexus_startpoint_is_null(&(ti->slaveSP)))
	nexus_printf("ServiceQ(): slaveSP is null\n");
    else
	nexus_printf("ServiceQ(): slaveSP is OK\n");
    
    if (nexus_startpoint_is_null(&(ti->leaderSP)))
	nexus_printf("ServiceQ(): leaderSP is null\n");
    else
	nexus_printf("ServiceQ(): leaderSP is OK\n");

    RunTests(epd->endpoint, &(ti->slaveSP), ti->verb,
	     NEXUS_FALSE, &report);

    nexus_buffer_init(&reply_buf,
		      nexus_sizeof_int(1) +
		      nexus_sizeof_startpoint(&(ti->slaveSP), 1) +
		      ReportSize(&report), 0);
    nexus_put_int(&reply_buf, &(ti->test_id), 1);
    nexus_put_startpoint_transfer(&reply_buf, &(ti->slaveSP), 1);
    PutReport(&reply_buf, &report);
    nexus_send_rsr(&reply_buf,
		   &(ti->leaderSP),
		   REPORT_HANDLER_ID,
		   NEXUS_TRUE, NEXUS_FALSE);
    nexus_startpoint_destroy(&(ti->leaderSP));
    nexus_free(ti);
    ReportDone(&report);

    nexus_printf("ServiceQ(): finished the test\n");
}
/* -- ServiceQ() -- */

/*
 * -- EPD_MainLoop() --
 *
 * Main thread of slave just calls this.
 * Main thread of leader calls this after sending out all tests.
 * Main thread of follower calls this while in the group.
 */
int EPD_MainLoop(ep_data_t *epd)
{
    nexus_printf("EPD_MainLoop(): hello\n");
    
    for(;;)
    {
	nexus_printf("EPD_MainLoop(): for\n");
	nexus_mutex_lock(&(epd->wait.mutex));
	while (epd->wait.done == 0)
	{
	    nexus_cond_wait(&(epd->wait.cond), &(epd->wait.mutex));
	}
	
	epd->wait.done--;

	/* Exit now if ContextDestroyMsg was received. */
	if (epd->wait.context_destroy != 0)
	{
	    nexus_printf("EPD_MainLoop(): got context_destroy_msg\n");
	    nexus_mutex_unlock(&(epd->wait.mutex));
	    return(2);
	}
	
	/* Only stop waiting if the queue is empty. */
	if ((epd->wait.stop_waiting != 0) && (epd->wait.done == 0))
	{
	    nexus_printf("EPD_MainLoop(): got stop_waiting_msg\n");
	    epd->wait.stop_waiting = 0;
	    nexus_printf("stop_waiting = 0\n");
	    nexus_mutex_unlock(&(epd->wait.mutex));
	    return(1);
	}

	/* Process one item from queue. */
	if (epd->wait.service_queue != 0)
	{
	    nexus_printf("EPD_MainLoop(): got a test to run\n");
	    epd->wait.service_queue--;
	    nexus_mutex_unlock(&(epd->wait.mutex));
	    ServiceQ(epd);
	}
	else
	    nexus_mutex_unlock(&(epd->wait.mutex));
    }
}
/* -- EPD_MainLoop() -- */

/*
 * -- EPD_NQ_Test() --
 */
int EPD_NQ_Test(ep_data_t *epd, test_info_t *ti)
{
    /* stick the test_info on the end of the Q */
    nexus_mutex_lock(&(epd->test_Q.mutex));
    if (epd->test_Q.first == NULL)
    {
	epd->test_Q.first = ti;
	epd->test_Q.last = ti;
    }
    else
    {
	epd->test_Q.last->next = ti;
	epd->test_Q.last = ti;
    }
    ti->next = NULL;
    nexus_mutex_unlock(&(epd->test_Q.mutex));

    /* tell the main thread about it */
    nexus_mutex_lock(&(epd->wait.mutex));
    epd->wait.service_queue++;
    epd->wait.done++;
    nexus_cond_signal(&(epd->wait.cond));
    nexus_mutex_unlock(&(epd->wait.mutex));
}
/* -- EPD_NQ_Test() -- */

/*
 * -- EPD_DQ_Test() --
 */
int EPD_DQ_Test(ep_data_t *epd, test_info_t **ti)
{
    nexus_mutex_lock(&(epd->test_Q.mutex));
    if (epd->test_Q.first == epd->test_Q.last)
    {
	/* last thing in the Q */
	*ti = epd->test_Q.first;
	epd->test_Q.first = NULL;
	epd->test_Q.last = NULL;
    }
    else
    {
	/* more stuff left */
	*ti = epd->test_Q.first;
	epd->test_Q.first = (*ti)->next;
	(*ti)->next = NULL;
    }
    nexus_mutex_unlock(&(epd->test_Q.mutex));
    
    if (*ti == NULL)
	return(0);
    else
	return(1);
}
/* -- EPD_DQ_Test() -- */

/*
 * -- EPD_ContextDestroy() --
 */
void EPD_ContextDestroy(ep_data_t *epd)
{
    nexus_mutex_lock(&(epd->wait.mutex));
    epd->wait.done++;
    epd->wait.context_destroy = 1;
    nexus_cond_signal(&(epd->wait.cond));
    nexus_mutex_unlock(&(epd->wait.mutex));
}
/* -- EPD_ContextDestroy() -- */
