/*
 * big_test.c
 */

#include <nexus.h>
#include <stdio.h>
#include "big_test.h"
#include "wrappers.h"
#include "handlers.h"
#include "remote.h"
#include "run_test.h"
#include "group.h"

#include "everything.h"

/* define to get messages when the required nexus functions are called */
#undef	TRC_nexus_funcs

/*******************************************************
 * Required Nexus Functions                            *
 *******************************************************/

/* The endpoint for this context */
static nexus_endpoint_t	GlobalEndpoint;

/*
 * -- NexusBoot() --
 *
 * Required nexus function.  Set up the endpoint for this context
 * and return return the startpoint to it.
 */
int NexusBoot(nexus_startpoint_t *startpoint)
{
    if (InitEndpoint(&GlobalEndpoint, 10, 0))
	return (nexus_startpoint_bind(startpoint, &GlobalEndpoint));
    else
	return(NEXUS_FAILURE);
}
/* -- NexusBoot() -- */

/*
 * -- NexusAcquiredAsNode() --
 *
 * Required nexus function.
 */
int NexusAcquiredAsNode(nexus_startpoint_t *startpoint)
{
#ifdef TRC_nexus_funcs
    nexus_printf("NexusAcquiredAsNode(): hello\n");
#endif
    if (nexus_startpoint_bind(startpoint, &GlobalEndpoint) == 0)
    {
	nexus_printf("NexusAcquiredAsNode(): success\n");
	return NEXUS_ACQUIRE_THIS_CONTEXT;
    }
    else
    {
	nexus_printf("NexusAcquiredAsNode(): failure\n");
	return NEXUS_ACQUIRE_NEW_INITIAL_CONTEXT;
    }
}
/* --NexusAcquiredAsNode() -- */

/*
 * -- NexusExit() --
 */
void NexusExit(void)
{
#ifdef TRC_nexus_funcs
    nexus_printf("NexusExit():\n");
#endif
    DoneEndpoint(&GlobalEndpoint);
}
/* -- NexusExit() -- */

/*******************************************************
 * main()                                              *
 *******************************************************/

/*
 * -- GetOpt() --
 *
 * Look to see if an option is present in the command line
 * somewhere.  If it is, return its position, otherwise 0.
 *
 * Order of options is unimportant to keep things simple.
 */
int GetOpt(int argc, char *argv[], const char *opt)
{
    int i, x;

    for (i = 1; i < argc; i++)
    {
	nexus_stdio_lock();
	x = strcmp(argv[i], opt);
	nexus_stdio_unlock();
	if (x == 0)
	    return(i);
    }

    return(0);
}
/* -- GetOpt() -- */

/*
 * -- ContextIsSlave() --
 */

void ContextIsSlave()
{
}

/*
 * -- main() --
 */
int main(int argc, char *argv[])
{
    nexus_node_t	*nodes;
    int			n_nodes;

    /******************
     * Initialization *
     ******************/    
    
    nexus_init(&argc, &argv,    /* command line parms   */
               "NEXUS_ARGS",    /* args_env_var         */
               "nx",            /* pkg designator       */
               NULL,            /* pkg args init func   */
               NULL,            /* usage msg func       */
               NULL,            /* new proc parms func  */
               NULL,            /* module list          */
               &nodes, &n_nodes); /* node list & count  */
    
    nexus_start_nonblocking();

    /* check if a slave (master has n_nodes >= 1) */
    if (n_nodes == 0)
    {
	nexus_printf("I am a slave!\n");

	/* Wait in a loop for signal */
	WaitForStopWaitingMsg(nexus_endpoint_get_user_pointer(&GlobalEndpoint));
    }
    else
    {
	nexus_startpoint_t	remote_sp;
	int			i, j, ok;
	char		*my_url;
	char		*attach_url;
	int			reps;
	int			verbose;
	my_url = NULL;
	attach_url = NULL;

	nexus_printf("I am the master!\n");
	
	/* no errors yet... */
	ok = 0;
    
	nexus_printf("main(): got %i nodes\n", n_nodes);

	/*
	 * No options?
	 *   Print usage.  Nothing else will get done with no options
	 *   given on the command line.
	 */
	if (argc == 1)
	{
	    nexus_stdio_lock();
	    printf("usage: %s <test options> <nexus options>\n", argv[0]);
	    printf("  test options:\n");
	    printf("    -v <lev>  verbose level 0..4 (1)\n");
	    printf("    -x <reps> times to repeat each test (3)\n");
	    printf("    -n        normal testing\n");
	    printf("    -l        listen\n");
	    printf("    -w        wait for attach\n");
	    printf("    -a <url>  attach to nexus-url\n");
	    printf("    -a self   attach to self\n");
	    printf("    -r        remote test\n");
	    printf("    -m        send stop waiting message\n");
	    printf("    -g <size> group server, include self if attached\n");
	    printf("    -j        join group\n");
	    printf("  defaults are in ()\n");
	    nexus_stdio_unlock();	
	}

	/***********
	 * Options *
	 ***********/
    
	/*
	 * -v = verbose level
	 *    how much output to give (0 is none, 4 is all)
	 *    1 is default,
	 *    2 gives enough output to show that something is happening
	 */    
	verbose = 1;
	if (i = GetOpt(argc, argv, "-verb"))
	{
	    int x;

	    nexus_stdio_lock();
	    x = sscanf(argv[i + 1], "%i", &verbose);
	    nexus_stdio_unlock();
	    if (x == 0)
	    {
		nexus_printf("main(): couldn't scan verbose level\n");
		verbose = 1;
	    }
	}
    
	/*
	 * -x = repetitions
	 *    number of times to repeat the test with each node
	 *    default = 3
	 */
	reps = 3;
	if (i = GetOpt(argc, argv, "-reps"))
	{
	    int x;

	    nexus_stdio_lock();
	    x = sscanf(argv[i + 1], "%i", &reps);
	    nexus_stdio_unlock();
	    if (x)
	    {
		if (verbose > 0)
		    nexus_printf("repetitions = %i\n", reps);
	    }
	    else
		nexus_printf("can't scan reps, using default 3\n");
	}

	/***********
	 * Actions *
	 ***********/
    
	/*
	 * -n = normal testing
	 *    The master connects to each startpoint in the list of
	 *    nodes returned by nexus_init() and runs a single test.
	 *    Goes through the whole list -r times.
	 */
	if (GetOpt(argc, argv, "-normal"))
	{
	    if (verbose > 1)
		nexus_printf("*** normal testing ***\n");
	
	    if (n_nodes < 2)
	    {
		nexus_printf("main(): not enough nodes for normal test\n");
	    }
	    else
	    {
		report_t	report;
		ReportInit(&report);	    
		for (j = 1; j <= reps; j++)
		{
		    for (i = 1; i < n_nodes; i++)
		    {
			if (RunTests(&GlobalEndpoint, &(nodes[i].startpoint),
				     verbose, NEXUS_FALSE, &report) == 0)
			{
			    nexus_printf("main(): normal pass %i, node %i bad\n",
					 j, i);
			    ok = 1;
			}
			else
			    if (verbose > 2)
				nexus_printf("main(): normal pass %i, node %i"
					     " OK\n", j, i);
		    } /* for i = {node} */
		    if (verbose > 1)
			nexus_printf("main(): normal pass %i completed\n", j);
		} /* for j = {pass} */
	    
		if (verbose > 0)
		    nexus_printf("%i normal passes completed\n", reps);
		ReportPrint(stdout, &report);
		ReportDone(&report);
	    }
	    if (verbose > 1)
		nexus_printf("*** end normal test ***\n");
	}

	/*
	 * -l = listen
	 *    Perform a nexus_allow_attach() and print out the URL.
	 */
	if (GetOpt(argc, argv, "-listen"))
	{
	    if (verbose > 1)
		nexus_printf("*** Listen ***\n");
	
	    my_url = (char *) nexus_malloc (200 * sizeof(char));
	    if (RemoteListen(&GlobalEndpoint, my_url))
		nexus_printf("Listening, URL = \"%s\"\n", my_url);
	    else
	    {
		nexus_printf("Listen failed\n");
		nexus_free(my_url);
		my_url = NULL;
		ok = 1;
	    }
	}

	/*
	 * -a = attach
	 *    Do a nexus_attach() to the given URL.
	 *    If URL == "self", use the URL obtained from nexus_allow_attach
	 *    for a loopback.
	 */
	if (i = GetOpt(argc, argv, "-attach"))
	{
	    int x;
	
	    if (verbose > 1)
		nexus_printf("*** Attach ***\n");

	    nexus_stdio_lock();
	    x = strcmp(argv[i + 1], "self");
	    nexus_stdio_unlock();
	    if (x == 0)
	    {
		if (my_url == NULL)
		{
		    attach_url = NULL;
		    nexus_printf("Can't attach to self--I'm not listening\n");
		}
		else
		    attach_url = my_url;		
	    }
	    else
		attach_url = argv[i + 1];

	    if (RemoteAttach(attach_url, &remote_sp) == 0)
	    {
		nexus_printf("main(): RemoteAttach to %s failed\n", attach_url);
		ok = 1;
		attach_url = NULL;
	    }
	    else
	    {
		if (verbose > 0)
		    nexus_printf("main(): attach succeeded\n");
	    }	
	}

	/*
	 * -r = remote test
	 */
	if (GetOpt(argc, argv, "-remote"))
	{
	    report_t	report;
	    ReportInit(&report);
	    if (verbose > 1)
		nexus_printf("*** remote test ***\n");
		
	    for (j = 1; j <= reps; j++)
	    {
		if (RunTests(&GlobalEndpoint, &remote_sp,
			     verbose, NEXUS_FALSE, &report) == 0)
		{
		    nexus_printf("main(): remote pass %i failed\n", j);
		    ok = 1;
		}
		else
		    if (verbose > 1)
			nexus_printf("main(): remote pass %i OK\n", j);
	    }
	    if (verbose > 0)
		nexus_printf("%i remote passes completed\n", reps);
	    ReportPrint(stdout, &report);
	    ReportDone(&report);
	
	    if (verbose > 1)
		nexus_printf("*** remote test done ***\n");
	}

	/*
	 * -g = group server
	 */
	if (i = GetOpt(argc, argv, "-group"))
	{
	    int		grp_size;
	    ep_data_t	*epd;
	    report_t	report;

	    if (verbose > 1)
		nexus_printf("*** group server start ***\n");
	
	    if ((my_url != NULL) && (attach_url != NULL))
	    {
		int x;

		nexus_stdio_lock();
		x = sscanf(argv[i + 1], "%i", &grp_size);
		nexus_stdio_unlock();
		if (x != 0)
		{
		    epd = nexus_endpoint_get_user_pointer(&GlobalEndpoint);
		    LeaderInit(epd, grp_size);
		    nexus_printf("main(): LeaderInit'd\n");
		    if ((attach_url != NULL) && (attach_url == my_url))
			JoinGroup(&GlobalEndpoint, n_nodes, nodes, &remote_sp);
		    nexus_printf("main(): GroupJoin'd\n");
		    WaitForFollowers(epd);
		    nexus_printf("main(): Wait'd\n");
		    ReportInit(&report);
		
		    BasicGroupTest(&GlobalEndpoint, verbose, reps, &report);
		
		    ReportDone(&report);
		    LeaderDone(epd);
		}
		else
		{
		    nexus_printf("main(): can't get group size\n");
		    ok = 1;
		}
	    }
	    else
	    {
		nexus_printf("Can't be group server--bad URL's\n");
		ok = 1;
	    }

	    if (verbose > 1)
		nexus_printf("*** group server done ***\n");
	}

	/*
	 * -j = join group
	 */
	if (i = GetOpt(argc, argv, "-join"))
	{
	    if (attach_url != NULL)
	    {
		if (attach_url == my_url)
		{
		    nexus_printf("Can't join own group with -j\n");
		    ok = 1;
		}
		else
		{
		    ep_data_t	*epd;

		    epd = nexus_endpoint_get_user_pointer(&GlobalEndpoint);
		    JoinGroup(&GlobalEndpoint, n_nodes, nodes, &remote_sp);
		    EPD_MainLoop(epd);
		}
	    }
	    else
	    {
		nexus_printf("Can't join group--not attached\n");
		ok = 1;
	    }
	}
    
	/*
	 * -m = send stop waiting message.
	 */
	if (GetOpt(argc, argv, "-msg"))
	{
	    if (attach_url != NULL)
	    {
		SendStopWaitingMsg(&remote_sp);
		nexus_startpoint_destroy(&remote_sp);
	    }
	    else
	    {
		nexus_printf("main(): not attached, can't send stop waiting "
			     "message.\n");
	    }
	}
	else
	{
	    if (attach_url != NULL)
		nexus_startpoint_destroy(&remote_sp);
	}
    
	/*
	 * -w = wait
	 *    Main thread waits on a condition until a message
	 *    to stop waiting is received.  Without this option,
	 *    this context will exit and other programs will
	 *    not have a chance to attach.
	 */ 
	if (GetOpt(argc, argv, "-wait"))
	{
	    nexus_printf("*** Waiting for stop waiting message ***\n");
	    if (my_url == NULL)
	    {
		nexus_printf("main(): can't wait; I'm not listening\n");
	    }
	    else
	    {
		WaitForStopWaitingMsg(nexus_endpoint_get_user_pointer(
		    &GlobalEndpoint));
	    }
	    nexus_printf("*** Waiting over ***\n");
	}    

	/* no more options to check */
	if (verbose > 1)
	    nexus_printf("*** End options ***\n");
    
	/***********
	 * Cleanup *
	 ***********/

	/* Attach_url is pointing same place as either my_url */
	/* or argv[?] so don't need to free it.		  */
	if (my_url != NULL)
	    nexus_free(my_url);
    
	/* send a message to all nodes telling them to exit */
	for (i = 1; i < n_nodes; i++)
	    SendContextDestroyMsg(&(nodes[i].startpoint));
	if (verbose > 1)
	    nexus_printf("main(): Contexts destroyed\n");
    
	/* Free nodes list */
	for (i = 0; i < n_nodes; i++)
	{
	    nexus_free(nodes[i].name);
	    nexus_startpoint_destroy(&(nodes[i].startpoint));
	}
	nexus_free(nodes);
	if (verbose > 1)
	    nexus_printf("main(): Nodes freed.\n");

	/*
	 * Print out the status and exit.
	 */
    
	/* Print out the last message, telling how testing went */
	if (ok)
	    nexus_printf("Testing did not complete successfully\n");
	else
	    nexus_printf("Testing completed successfully\n");

	/* Destroy this context & exit */
	/* Need a way for the program to return error code in ok */
	nexus_context_destroy(NEXUS_FALSE);

	/* nexus_context_destroy should not return */
	nexus_printf("main(): shouldn't get here!\n");
    }
}
/* -- main() -- */
