/*
 * ppmp.c
 *
 * Message Passing ping-pong
 * Supports IBM SP2 MPL, Intel Paragon NX, and MPI
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/performance/ppmp.c,v 1.2 1998/06/12 21:58:07 tuecke Exp $";

#include <stdio.h>
#include "globus_utp.h"

extern char *getenv(const char *);

/*
#ifndef DISPLAY_TIMES
#define DISPLAY_TIMES
#endif
*/

#define N_TESTS		1
#define TEST_RUN_TIME	5.0
#define CALIBRATE_ITERS 100

int msg_size_array[] =
{
    0,
    10,
    100,
    1000,
    5000,
    10000,
    50000,
    100000,
    500000,
    1000000,
    -1,
    0,
    10,
    20,
    30,
    40,
    50,
    60,
    70,
    80,
    90,
    100,
    200,
    300,
    400,
    500,
    600,
    700,
    800,
    900,
    1000,
    2000,
    3000,
    4000,
    5000,
    6000,
    7000,
    8000,
    9000,
    10000,
    20000,
    30000,
    40000,
    50000,
    60000,
    70000,
    80000,
    90000,
    100000,
    200000,
    300000,
    400000,
    500000,
    600000,
    700000,
    800000,
    900000,
    1000000,
    -1
};

#define PP_MAX(V1,V2) (((V1) > (V2)) ? (V1) : (V2))
#define PP_MIN(V1,V2) (((V1) < (V2)) ? (V1) : (V2))


/******************************************************************
 *		MPL Support
 ******************************************************************/
#ifdef PP_ARCH_MPL
#include <mpproto.h>

#define PPTYPE "ppmpl"
#define DEFAULT_FILENAME "ppmpl.utp"

#define MP_INIT_NODE_INFO(My_node, N_nodes, Ping_node, Pong_node) \
{ \
    mpc_environ(&(N_nodes), &(My_node)); \
    (Ping_node) = 0; \
    (Pong_node) = 1; \
}

#define MP_SHUTDOWN()

#define MP_SEND(Dest, Buf, Len) \
{ \
    mpc_bsend(Buf, Len, Dest, 0); \
}

#define MP_RECV(Source, Buf, Len) \
{ \
    int __source = DONTCARE; \
    int __type = DONTCARE; \
    size_t __n_bytes; \
    mpc_brecv(Buf, Len, &__source, &__type, &__n_bytes); \
}
/*
#define MP_SEND(Dest, Buf, Len) \
{ \
    int __msg_id; \
    mpc_send(Buf, Len, Dest, 0, &__msg_id); \
    while (mpc_status(__msg_id) < 0) ; \
}

#define MP_RECV(Source, Buf, Len) \
{ \
    int __source = DONTCARE; \
    int __type = DONTCARE; \
    int __n_bytes; \
    int __msg_id; \
    mpc_recv(Buf, Len, &__source, &__type, &__msg_id); \
    mpc_wait(&__msg_id, &__n_bytes); \
}
*/

#endif /* PP_ARCH_MPL */


/******************************************************************
 *		INX Support
 ******************************************************************/
#ifdef PP_ARCH_INX
#include <nx.h>

extern char *getenv(char *);

#define PPTYPE "ppinx"
#define DEFAULT_FILENAME "ppinx.utp"

#define MP_INIT_NODE_INFO(My_node, N_nodes, Ping_node, Pong_node) \
{ \
    (My_node) = mynode(); \
    (N_nodes) = numnodes(); \
    (Ping_node) = 0; \
    (Pong_node) = 1; \
}

#define MP_SHUTDOWN()

#define MP_SEND(Dest, Buf, Len) \
{ \
    csend(0, Buf, Len, Dest, 0); \
}

#define MP_RECV(Source, Buf, Len) \
{ \
    crecv(-1, Buf, Len); \
}

#endif /* PP_ARCH_INX */


/******************************************************************
 *		MPI Support
 ******************************************************************/
#ifdef PP_ARCH_MPI
#include "mpi.h"

#ifndef NULL
#define NULL (void *) 0
#endif

#ifdef MPITIME
#define PPTYPE "mpitime"
#define DEFAULT_FILENAME "mpitime.utp"
#else
#define PPTYPE "ppmpi"
#define DEFAULT_FILENAME "ppmpi.utp"
#endif

#define MP_INIT_NODE_INFO(My_node, N_nodes, Ping_node, Pong_node) \
{ \
    if (   (MPI_Init(&argc,&argv) != MPI_SUCCESS) \
	|| (MPI_Comm_size(MPI_COMM_WORLD,&(N_nodes)) != MPI_SUCCESS) \
	|| (MPI_Comm_rank(MPI_COMM_WORLD,&(My_node)) != MPI_SUCCESS) ) \
    { \
	printf("ERROR: Failed initialization\n"); \
	exit(1); \
    } \
    if (argc == 1) \
    { \
	(Ping_node) = 0; \
	(Pong_node) = 1; \
    } \
    else if (argc == 3) \
    { \
	(Ping_node) = atoi(argv[1]); \
	(Pong_node) = atoi(argv[2]); \
	if (((Ping_node) >= (N_nodes)) || ((Pong_node) >= (N_nodes))) \
	{ \
	    printf("ERROR: Invalid node number\n"); \
	    MPI_Finalize(); \
	    exit(1); \
	} \
    } \
    else \
    { \
	printf("Usage: ppmpi [<ping_node> <pong_node>]\n"); \
	MPI_Finalize(); \
	exit(1); \
    } \
}

#define MP_SHUTDOWN() \
{ \
    MPI_Finalize(); \
}

#define MP_SEND(Dest, Buf, Len) \
{ \
    if (MPI_Send((void *) (Buf), (Len), MPI_BYTE, (Dest), 42, MPI_COMM_WORLD) != MPI_SUCCESS) \
    { \
	printf("ERROR: MPI_Send() failed\n"); \
	exit(1); \
    } \
}

#define MP_RECV(Source, Buf, Len) \
{ \
    MPI_Status __status; \
    if (MPI_Recv((void *) (Buf), (Len), MPI_BYTE, (Source), 42, MPI_COMM_WORLD, &__status) != MPI_SUCCESS) \
    { \
	printf("ERROR: MPI_Recv() failed\n"); \
	exit(1); \
    } \
}

#endif /* PP_ARCH_MPI */


#ifndef NULL
#define NULL (void *) 0
#endif


/******************************************************************
 *		Ping-pong code
 ******************************************************************/

static void ping_main(int pong_node);
static void pong_main(int ping_node);

/*
 * main()
 */
int main(int argc, char **argv)
{
    int n_nodes, my_node;
    int ping_node, pong_node;

    MP_INIT_NODE_INFO(my_node, n_nodes, ping_node, pong_node);

    /* See if we have enough nodes */
    if (n_nodes < 2)
    {
	printf("Error: Must run with 2 or more nodes.\n");
	goto shutdown;
    }

    if (my_node == 0)
    {
        printf("Running between nodes %d and %d\n", ping_node, pong_node);
    }

    if (my_node == ping_node)
    {
	ping_main(pong_node);
    }
    else if (my_node == pong_node)
    {
	pong_main(ping_node);
    }

 shutdown:
    MP_SHUTDOWN();
    return(0);
} /* main() */


/*
 * ping_main()
 */
static void ping_main(int pong_node)
{
    char *msgbuf;
    int msg_size;
    int n_iters;
    int starting_n_iters;
    int i, j;
    char *data_file;
    int n_msg_sizes;
    int max_msg_size;
    double time;
    int precision;
    int ntest;
    int timer_index;
    
    /* Count the number of tests */
    for (n_msg_sizes = 0, max_msg_size = 1;
	 msg_size_array[n_msg_sizes] >= 0;
	 n_msg_sizes++)
    {
	if (msg_size_array[n_msg_sizes] > max_msg_size)
	    max_msg_size = msg_size_array[n_msg_sizes];
    }

    /* Get a filename */
    if ((data_file = getenv("PP_FILE")) == (char *) NULL)
    {
	data_file = DEFAULT_FILENAME;
    }

    /* Allocate room for the message buffer */
    msgbuf = (char *) malloc(max_msg_size);

    /* Initialize timers */
    globus_module_activate(GLOBUS_UTP_MODULE);
    globus_utp_init((n_msg_sizes * N_TESTS), GLOBUS_UTP_MODE_SHARED);
    globus_utp_set_attribute("rcsid%s", "", "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/performance/ppmp.c,v 1.2 1998/06/12 21:58:07 tuecke Exp $");
    globus_utp_set_attribute("type%s", "", PPTYPE);
    globus_utp_set_attribute("program_rcsid%s", "", rcsid+1);
    globus_utp_set_attribute("n_tests%s", "", "%d", N_TESTS);

    /*
     * Send some messages around to get things warmed up.
     * Use the results of this to figure out a multipler
     * which will automatically scale all of the test lengths.
     */
    msg_size = msg_size_array[0];
    n_iters = CALIBRATE_ITERS;
    printf("Calibrating...\n");
    globus_utp_start_timer(0);
    for (j = 0; j < n_iters; j++)
    {
	MP_SEND(pong_node, msgbuf, msg_size);
	MP_RECV(pong_node, msgbuf, msg_size);
    }
    globus_utp_stop_timer(0);
    globus_utp_get_accum_time(0, &time, &precision);
    globus_utp_reset_timer(0);

    /* Adjust n_iters and send it to pong_main() */
    starting_n_iters = PP_MAX(((int) ((TEST_RUN_TIME / time) * n_iters)), 1);
    MP_SEND(pong_node, &starting_n_iters, sizeof(starting_n_iters));
    
    /* Run timings */
    for (ntest = 0; ntest < N_TESTS; ntest++)
    {
	n_iters = starting_n_iters;
	
	for (i = 0; i < n_msg_sizes; i++)
	{
	    timer_index = (ntest * n_msg_sizes) + i;
	    msg_size = msg_size_array[i];
	    
	    printf("Timing size=%d, iterations=%d, ntest=%d\n",
		   msg_size, n_iters, ntest);

	    globus_utp_name_timer(timer_index, "msg_len:%d count:%d testno:%d",
			   msg_size, n_iters, ntest);
	    globus_utp_start_timer(timer_index);
	    
	    for (j = 0; j < n_iters; j++)
	    {
		MP_SEND(pong_node, msgbuf, msg_size);
		MP_RECV(pong_node, msgbuf, msg_size);
	    }
	    
	    globus_utp_stop_timer(timer_index);
	    
	    /* Update and send the iters_multiplier to pong_main() */
	    globus_utp_get_accum_time(timer_index, &time, &precision);
#ifdef DISPLAY_TIMES
	    printf("Results:\n");
	    printf("                     roundtrip count: %d\n", n_iters);
	    printf("              message length (bytes): %d\n", msg_size);
	    printf("                   elapsed time (us): %f\n",
                   time*1000000);
	    printf("         average roundtrip time (us): %f\n",
		   time*1000000/n_iters);
	    printf("    latency (average oneway time,us): %f\n",
		   (time*1000000/n_iters)/2);
	    printf("            bandwidth (bytes/second): %f\n",
		   (n_iters*msg_size*2)/time);
	    printf("\n");
#endif /* DISPLAY_TIMES */
	    n_iters = PP_MAX(((int) ((TEST_RUN_TIME / time) * n_iters)), 1);
	    MP_SEND(pong_node, &n_iters, sizeof(n_iters));
	}
    }

    printf("Done.\n");
    
    /* Write out the timer file */
    globus_utp_write_file(data_file);
    
    globus_module_deactivate(GLOBUS_UTP_MODULE);

    free(msgbuf);
} /* ping_main() */


/*
 * pong_main()
 */
static void pong_main(int ping_node)
{
    char *msgbuf;
    int msg_size;
    int n_iters;
    int starting_n_iters;
    int i, j;
    int n_msg_sizes;
    int max_msg_size;
    int ntest;

    /* Count the number of tests */
    for (n_msg_sizes = 0, max_msg_size = 1;
	 msg_size_array[n_msg_sizes] >= 0;
	 n_msg_sizes++)
    {
	if (msg_size_array[n_msg_sizes] > max_msg_size)
	    max_msg_size = msg_size_array[n_msg_sizes];
    }

    /* Allocate room for the message buffer */
    msgbuf = (char *) malloc(max_msg_size);

    /* Send some primer messages around */
    msg_size = msg_size_array[0];
    n_iters = CALIBRATE_ITERS;
    for (j = 0; j < n_iters; j++)
    {
	MP_RECV(ping_node, msgbuf, msg_size);
	MP_SEND(ping_node, msgbuf, msg_size);
    }

    /* Get n_iters from ping_main() */
    MP_RECV(ping_node, &starting_n_iters, sizeof(starting_n_iters));

    /* Run timings */
    for (ntest = 0; ntest < N_TESTS; ntest++)
    {
	n_iters = starting_n_iters;
	
	for (i = 0; i < n_msg_sizes; i++)
	{
	    msg_size = msg_size_array[i];
	    
	    for (j = 0; j < n_iters; j++)
	    {
		MP_RECV(ping_node, msgbuf, msg_size);
		MP_SEND(ping_node, msgbuf, msg_size);
	    }
	    
	    /* Get an n_iters from ping_main() */
	    MP_RECV(ping_node, &n_iters, sizeof(n_iters));
	}
    }
    
    free(msgbuf);
    
} /* pong_main() */
