/*
 * pphandle.c
 *
 * Ping pong between two non-threaded handlers.
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/performance/pphandle.c,v 1.5 1998/06/11 00:42:38 tuecke Exp $";

#include <stdio.h>

#define USE_MACROS
#include "globus_nexus.h"
#include "globus_utp.h"

/* This is a hack */
#undef globus_libc_malloc
#define globus_libc_malloc malloc

#ifdef PP_THREADED_HANDLERS
#define DEFAULT_UTP_FILE "pphandle-t.utp"
#else /* PP_THREADED_HANDLERS */
#define DEFAULT_UTP_FILE "pphandle.utp"
#endif /* PP_THREADED_HANDLERS */


#undef GLOBUS_NEXUS_CRITICAL_PATH_TIMER
/*
*/
#ifndef DISPLAY_TIMES
#define DISPLAY_TIMES
#endif
/*
*/
/*
#define PP_PRINTF_ON
*/

#undef GLOBUS_NEXUS_USE_ACK_PROTOCOL
#undef GLOBUS_NEXUS_SPOOF_COPY

#include "pputil.h"

#define PPTYPE "pphandle"
#define DEFAULT_MSG_SIZE_FILE "pp_msg_sizes"
#ifndef CALIBRATE_ITERS
#define CALIBRATE_ITERS 1000
#endif
#define MIN_ITERS 10

#ifdef GLOBUS_NEXUS_CRITICAL_PATH_TIMER
#define GLOBUS_NEXUS_SPLIT_CRITICAL_PATH_TIMER
/*
*/
extern bool_t _nx_time_critical_path;
extern bool_t _nx_critical_path_timer_started;
extern int _nx_critical_path_start_timer;
extern int _nx_critical_path_stop_timer;
#ifdef GLOBUS_NEXUS_SPLIT_CRITICAL_PATH_TIMER
#define EXTRA_CRITICAL_PATH_TIMERS_MULT 3
#else
#define EXTRA_CRITICAL_PATH_TIMERS_MULT 2
#endif
#else
#define EXTRA_CRITICAL_PATH_TIMERS_MULT 1
#endif

#ifdef USE_THREADED_HANDLERS
#define PP_HANDLER_TYPE GLOBUS_NEXUS_HANDLER_TYPE_THREADED
#else
#define PP_HANDLER_TYPE GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED
#endif

#define PP_MAX(V1,V2) (((V1) > (V2)) ? (V1) : (V2))
#define PP_MIN(V1,V2) (((V1) < (V2)) ? (V1) : (V2))

#ifdef PP_PRINTF_ON
#define pp_printf(message) globus_libc_printf message
#else
#define pp_printf(message)
#endif


static int
attach_to_server(char *server_host,
		 unsigned short server_port);
static void
start_client(char *client_host,
	     char *executable,
	     globus_nexus_startpoint_t *client_sp,
	     globus_bool_t nostart);

static void
pp_driver(globus_nexus_startpoint_t *server_sp,
	  char *msg_size_file);


/* Shutdown prototypes */
static void
barrier_init();

static void
barrier_destroy();

static void
barrier_wait();

static void
barrier_signal();

static void
shutdown_client(globus_nexus_startpoint_t *client_sp);

/* Handlers */
static void
shutdown_handler(globus_nexus_endpoint_t *endpoint,
		 globus_nexus_buffer_t *buffer,
		 globus_bool_t is_nonthreaded_handler);
#define SHUTDOWN_HANDLER_ID 0

static void
shutdown_reply_handler(globus_nexus_endpoint_t *endpoint,
		       globus_nexus_buffer_t *buffer,
		       globus_bool_t is_nonthreaded_handler);
#define SHUTDOWN_REPLY_HANDLER_ID 1

static void
pong_starter_handler(globus_nexus_endpoint_t *endpoint,
		     globus_nexus_buffer_t *buffer,
		     globus_bool_t is_nonthreaded_handler);
#define PONG_STARTER_HANDLER_ID 2

static void
ping_starter_handler(globus_nexus_endpoint_t *endpoint,
		     globus_nexus_buffer_t *buffer,
		     globus_bool_t is_nonthreaded_handler);
#define PING_STARTER_HANDLER_ID 3

static void
pong_handler(globus_nexus_endpoint_t *endpoint,
	     globus_nexus_buffer_t *buffer,
	     globus_bool_t is_nonthreaded_handler);
#define PONG_HANDLER_ID 4

static void
ping_handler(globus_nexus_endpoint_t *endpoint,
	     globus_nexus_buffer_t *buffer,
	     globus_bool_t is_nonthreaded_handler);
#define PING_HANDLER_ID 5

static void
client_checkin_handler(globus_nexus_endpoint_t *endpoint,
		       globus_nexus_buffer_t *buffer,
		       globus_bool_t is_nonthreaded_handler);
#define CLIENT_CHECKIN_HANDLER_ID 6

static globus_nexus_handler_t handler_table[] =
{
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,
     shutdown_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,
     shutdown_reply_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,
     pong_starter_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,
     ping_starter_handler},
    {PP_HANDLER_TYPE,
     pong_handler},
    {PP_HANDLER_TYPE,
     ping_handler},
    {GLOBUS_NEXUS_HANDLER_TYPE_NON_THREADED,
     client_checkin_handler},
};
#define HANDLER_TABLE_SIZE 7

static globus_nexus_endpointattr_t	global_epattr;
static globus_nexus_endpoint_t		global_endpoint;

int timernum;


#ifdef CHECKMESSAGE
static char *OriginalMessage;
static char Error;
#endif /* CHECKMESSAGE */

/*
 * print_usage()
 */
void
print_usage(int argc, char **argv)
{
    printf("Usage: %s [-f <message_size_file>]\n", argv[0]);
    printf("          [-c <server_host> <server_port>]\n");
    printf("          [-s <startup_host>]\n");
    printf("          [-nostart]\n");
} /* print_usage() */

/*****************************************************************
 *		MAIN
 *****************************************************************/

/*
 * main()
 */
int
main(int argc, char **argv)
{
    int i;
    char *msg_size_file;
    char *server_host;
    unsigned short server_port;
    char *client_host;
    globus_bool_t nostart;

    /* Get the command line arguments */
    msg_size_file = DEFAULT_MSG_SIZE_FILE;
    server_host = NULL;
    server_port = 0;
    client_host = NULL;
    nostart = GLOBUS_FALSE;

    for (i = 1; i < argc; i++)
    {
	if ((strcmp(argv[i],"-f") == 0) && ((i + 1) < argc))
	{
	    msg_size_file = argv[++i];
	}
	else if ((strcmp(argv[i],"-c") == 0) && ((i + 2) < argc))
	{
	    server_host = argv[++i];
	    server_port = atoi(argv[++i]);
	}
	else if ((strcmp(argv[i],"-s") == 0) && ((i + 1) < argc))
	{
	    client_host = argv[++i];
	}
	else if (strcmp(argv[i],"-h") == 0)
	{
	    print_usage(argc, argv);
	    return(0);
	}
	else if (strcmp(argv[i],"-nostart") == 0)
	{
	    nostart = GLOBUS_TRUE;
	}
	else
	{
	    printf("Invalid argument: %s\n", argv[i]);
	    print_usage(argc, argv);
	    return(1);
	}
    }

    if (globus_module_activate(GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS)
    {
	printf("ERROR: Failed to activate nexus module\n");
	exit(1);
    }

    /* Create the endpoint that will be used for everything */
    globus_nexus_endpointattr_init(&global_epattr);
    globus_nexus_endpointattr_set_handler_table(&global_epattr,
						handler_table,
						HANDLER_TABLE_SIZE);
    globus_nexus_endpoint_init(&global_endpoint, &global_epattr);

    /* Initialize the shutdown barrier */
    barrier_init();
    
    if (server_host)
    {
	/* Client: Connect to server, and wait for tests to run */
	int rc;
	rc = attach_to_server(server_host, server_port);
	if (rc)
	{
	    printf("ERROR: globus_nexus_attach() failed with rc=%i\n", rc);
	}
	else
	{
	    barrier_wait();
	}
    }
    else if (client_host)
    {
	/* Server: Start client and run test */
	globus_nexus_startpoint_t client_sp;
	start_client(client_host, argv[0], &client_sp, nostart);
	pp_driver(&client_sp, msg_size_file);
	shutdown_client(&client_sp);
    }
    else
    {
	print_usage(argc, argv);
    }

    /* Cleanup the shutdown barrier */
    barrier_destroy();

    globus_module_deactivate(GLOBUS_NEXUS_MODULE);

    return(0);
} /* main() */


/*****************************************************************
 *		ATTACHMENT
 *****************************************************************/

/*
 * attach_approval_func()
 */
static int
attach_approval_func(void *user_arg,
		     char *url,
		     globus_nexus_startpoint_t *sp)
{
    globus_nexus_startpoint_bind(sp, &global_endpoint);
    return(0);
} /* attach_approval_func() */

static globus_nexus_startpoint_t *checkin_client_sp;

/*
 * start_client()
 */
static void
start_client(char *client_host,
	     char *executable,
	     globus_nexus_startpoint_t *client_sp,
	     globus_bool_t nostart)
{
    unsigned short port = 0;
    char *host;
    char command[1024];
    char directory[1024];

    checkin_client_sp = client_sp;
    
    globus_nexus_allow_attach(&port,
		       &host,
		       attach_approval_func,
		       NULL); /* user_arg */

    getcwd(directory, 1024);
    strip_tmp_mnt_from_path(directory);
    
    globus_libc_lock();
    sprintf(command, "%s -c %s %hu\n", executable, host, port);
    globus_libc_unlock();

    if (nostart)
    {
	globus_libc_lock();
	printf("Would run: \"cd %s; %s\"\n", directory, command);
	globus_libc_unlock();
    }
    else
    {
	start_remote_node(client_host,
			  command,
			  directory);
    }
    
    barrier_wait();
    
} /* start_client() */


/*
 * client_checkin_handler()
 */
static void
client_checkin_handler(globus_nexus_endpoint_t *endpoint,
		       globus_nexus_buffer_t *buffer,
		       globus_bool_t called_from_non_threaded_handler)
{
    globus_nexus_get_startpoint(buffer, checkin_client_sp, 1);
    barrier_signal();
} /* client_checkin_handler() */


/*
 * attach_to_server()
 */
static int
attach_to_server(char *server_host,
		 unsigned short server_port)
{
    globus_nexus_startpoint_t server_sp;
    int rc;
    char url[1024];

    globus_libc_lock();
    sprintf(url, "x-nexus://%s:%hu/", server_host, server_port);
    globus_libc_unlock();
    
    rc = globus_nexus_attach(url, &server_sp);

    if (rc == 0)
    {
	globus_nexus_buffer_t buffer;
	globus_nexus_startpoint_t client_sp;
	int buf_size;

	globus_nexus_startpoint_bind(&client_sp, &global_endpoint);
	buf_size = globus_nexus_sizeof_startpoint(&client_sp, 1);
	globus_nexus_buffer_init(&buffer, buf_size, 0);
	globus_nexus_put_startpoint_transfer(&buffer, (&client_sp), 1);
	if (globus_nexus_send_rsr(&buffer,
				  &server_sp,
				  CLIENT_CHECKIN_HANDLER_ID,
				  GLOBUS_TRUE,
				  GLOBUS_FALSE) != 0)
	{
	    globus_libc_printf("attach_to_server(): WARNING: send failed\n");
	}
	globus_nexus_startpoint_destroy(&server_sp);
    }
    
    return(rc);
} /* attach_to_server() */


/*****************************************************************
 *		SHUTDOWN
 *****************************************************************/

typedef struct barrier_struct
{
    int			count;
    globus_mutex_t	mutex;
    globus_cond_t	cond;
} barrier_t;

static barrier_t barrier;

/*
 * shutdown_handler()
 */
static void
shutdown_handler(globus_nexus_endpoint_t *endpoint,
		 globus_nexus_buffer_t *buffer,
		 globus_bool_t called_from_non_threaded_handler)
{
    globus_nexus_buffer_t reply_buffer;
    globus_nexus_startpoint_t reply_sp;

    globus_nexus_get_startpoint(buffer, &reply_sp, 1);
    
    globus_nexus_buffer_init(&reply_buffer, 0, 0);
    if (globus_nexus_send_rsr(&reply_buffer,
			      &reply_sp,
			      SHUTDOWN_REPLY_HANDLER_ID,
			      GLOBUS_TRUE,
			      called_from_non_threaded_handler) != 0)
    {
	globus_libc_printf("shutdown_handler(): WARNING: send failed\n");
    }

    globus_nexus_startpoint_destroy(&reply_sp);
    
    barrier_signal();
    
} /* shutdown_handler() */


/*
 * shutdown_reply_handler()
 */
static void
shutdown_reply_handler(globus_nexus_endpoint_t *endpoint,
		       globus_nexus_buffer_t *buffer,
		       globus_bool_t called_from_non_threaded_handler)
{
    barrier_signal();
} /* shutdown_reply_handler() */


/*
 * barrier_init()
 */
static void
barrier_init()
{
    barrier.count = 1;
    globus_mutex_init(&(barrier.mutex),
		      (globus_mutexattr_t *) NULL);
    globus_cond_init(&(barrier.cond),
		     (globus_condattr_t *) NULL);
} /* barrier_init() */


/*
 * barrier_destroy()
 */
static void
barrier_destroy()
{
    globus_mutex_destroy(&(barrier.mutex));
    globus_cond_destroy(&(barrier.cond));
} /* barrier_destroy() */


/*
 * barrier_wait()
 */
static void
barrier_wait()
{
    globus_mutex_lock(&(barrier.mutex));
    while (barrier.count > 0)
    {
	globus_cond_wait(&(barrier.cond),
			 &(barrier.mutex) );
    }
    barrier.count = 1;
    globus_mutex_unlock(&(barrier.mutex));
} /* barrier_wait() */


/*
 * barrier_signal()
 */
static void
barrier_signal()
{
    globus_mutex_lock(&(barrier.mutex));
    if (--barrier.count == 0)
    {
	globus_cond_signal(&(barrier.cond));
    }
    globus_mutex_unlock(&(barrier.mutex));
} /* barrier_signal() */


/*
 * shutdown_client()
 */
static void
shutdown_client(globus_nexus_startpoint_t *client_sp)
{
    globus_nexus_buffer_t buffer;
    globus_nexus_startpoint_t reply_sp;
    int buf_size;

    globus_nexus_startpoint_bind(&reply_sp, &global_endpoint);
    buf_size = globus_nexus_sizeof_startpoint(&reply_sp, 1);
    globus_nexus_buffer_init(&buffer, buf_size, 0);
    globus_nexus_put_startpoint_transfer(&buffer, (&reply_sp), 1);
    if (globus_nexus_send_rsr(&buffer,
			      client_sp,
			      SHUTDOWN_HANDLER_ID,
			      GLOBUS_TRUE,
			      GLOBUS_FALSE) != 0)
    {
	globus_libc_printf("shutdown_client(): WARNING: send failed\n");
    }

    barrier_wait();
    
} /* shutdown_client() */


/*****************************************************************
 *              ping pong
 *****************************************************************/

typedef struct
{
    globus_nexus_endpoint_t ep;
    globus_mutex_t mutex;
    globus_cond_t cond;
    globus_bool_t done;
    int count;
    int length;
    int iter;
    int timernum;
    globus_nexus_startpoint_t other_sp;
    char *message;
} message_monitor_t;


/*
 * pp_driver()
 */
static void
pp_driver(globus_nexus_startpoint_t *pong_sp,
	  char *msg_size_file)
{
    globus_nexus_buffer_t buffer;
    globus_nexus_startpoint_t my_sp;
    message_monitor_t *monitor;
    globus_nexus_endpointattr_t epattr;
    int i;
    int ntest;
    int n_tests;
    int n_msg_sizes;
    int *msg_size_array;
    int max_msg_size;
    int n_iters;
    int starting_n_iters;
    double time;
    int precision;
    int timer_index;
    int msg_size;
    double test_run_time;
#ifdef CHECKMESSAGE
    unsigned char *rcp, *cp;
    int r;
#endif /* CHECKMESSAGE */
	
    pp_printf(("pp_driver(): entering\n"));

    if (!read_msg_sizes(msg_size_file,
			globus_libc_malloc,
			GLOBUS_TRUE, /* allow_zero_length_message */
			&n_tests,
			&test_run_time,
			&n_msg_sizes,
			&msg_size_array,
			&max_msg_size))
    {
	globus_nexus_fatal("Couldn't read sizes file: %s\n", msg_size_file);
    }

    globus_module_activate(GLOBUS_UTP_MODULE);
    
    globus_utp_init((n_msg_sizes * n_tests) * EXTRA_CRITICAL_PATH_TIMERS_MULT,
		    GLOBUS_UTP_MODE_SHARED);
    globus_utp_set_attribute("rcsid%s", "", "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/performance/pphandle.c,v 1.5 1998/06/11 00:42:38 tuecke Exp $");
    globus_utp_set_attribute("type%s", "", PPTYPE);
    globus_utp_set_attribute("program_rcsid%s", "", rcsid+1);
    globus_utp_set_attribute("n_tests%s", "", "%d", n_tests);
    globus_utp_set_attribute("test_run_time%s", "", "%f", test_run_time);
    globus_utp_set_attribute("sizes_file%s", ": ", "%s", msg_size_file);

    /* Initialize monitor */
    monitor = (message_monitor_t *)
	globus_libc_malloc(sizeof(message_monitor_t));
    globus_nexus_endpoint_init(&(monitor->ep), &global_epattr);
    globus_nexus_endpoint_set_user_pointer(&(monitor->ep), (void *) monitor);
    globus_mutex_init(&(monitor->mutex), (globus_mutexattr_t *) NULL);
    globus_cond_init(&(monitor->cond), (globus_condattr_t *) NULL);
    monitor->message = (char *) globus_libc_malloc(max_msg_size);

#ifdef CHECKMESSAGE
    OriginalMessage = (char *) globus_libc_malloc(max_msg_size);
    Error = 0;

    /* randomizing the message body */
    rcp = (unsigned char *) &r;
    rcp += 2;
    for (i = 0, cp = (unsigned char *) (monitor->message);
        i < max_msg_size; i ++, cp ++)
    {
        r = rand();
        *cp = *rcp;
    } /* endfor */
#endif /* CHECKMESSAGE */
    
    /*
     * Run calibration loop
     */
    globus_libc_lock();
    printf("Calibrating...\n");
    globus_libc_unlock();

    msg_size = msg_size_array[0];
    n_iters = CALIBRATE_ITERS;
    
    monitor->done = GLOBUS_FALSE;
    monitor->count = n_iters;
    monitor->length = msg_size;
    monitor->iter = 0;
    monitor->timernum = 0;

#ifdef GLOBUS_NEXUS_CRITICAL_PATH_TIMER
    _nx_critical_path_start_timer = 1;
    _nx_critical_path_stop_timer = 1;
#endif    

    /* Send message to pong_starter to initialize its monitor. */
    globus_nexus_startpoint_bind(&my_sp, &(monitor->ep));
    globus_nexus_buffer_init(&buffer,
			     (globus_nexus_sizeof_startpoint(&my_sp, 1)
			      + (globus_nexus_sizeof_int(1) * 2)),
			     0);
    globus_nexus_put_startpoint_transfer(&buffer, (&my_sp), 1);
    globus_nexus_put_int(&buffer, &monitor->count, 1);
    globus_nexus_put_int(&buffer, &monitor->length, 1);
    globus_nexus_send_rsr(&buffer,
			  pong_sp,
			  PONG_STARTER_HANDLER_ID,
			  GLOBUS_TRUE,
			  GLOBUS_FALSE);
	   
    /* Wait for the ping pong to complete */
    globus_mutex_lock(&(monitor->mutex));
    while (!monitor->done)
    {
	globus_cond_wait(&(monitor->cond), &(monitor->mutex));
    }
    globus_mutex_unlock(&(monitor->mutex));
    
    globus_utp_get_accum_time(0, &time, &precision);
    globus_utp_reset_timer(0);
#ifdef GLOBUS_NEXUS_CRITICAL_PATH_TIMER
    globus_utp_reset_timer(1);
#endif    

    /* Adjust n_iters */
    starting_n_iters = PP_MAX(((int) ((test_run_time / time) * n_iters)),
			      MIN_ITERS);
    
    /*
     * Run timed loops.
     */
    for (ntest = 0; ntest < n_tests; ntest++)
    {
	n_iters = starting_n_iters;
	
	for (i = 0; i < n_msg_sizes; i++)
	{
	    timer_index = (ntest * n_msg_sizes) + i;
	    msg_size = msg_size_array[i];

	    globus_libc_lock();
	    printf("Timing size=%d, iterations=%d, ntest=%d\n",
		   msg_size, n_iters, ntest);
	    globus_libc_unlock();

	    globus_utp_name_timer(timer_index,
				  "msg_len:%d count:%d testno:%d",
				  msg_size, n_iters, ntest);

#ifdef GLOBUS_NEXUS_CRITICAL_PATH_TIMER
            _nx_critical_path_start_timer
                = (n_msg_sizes * n_tests) + timer_index;
#ifdef GLOBUS_NEXUS_SPLIT_CRITICAL_PATH_TIMER
            _nx_critical_path_stop_timer
                = (2 * (n_msg_sizes * n_tests)) + timer_index;
            globus_utp_name_timer(_nx_critical_path_start_timer,
                           "critical_path_timer_receive msg_len:%d testno:%d",
                           msg_size, ntest);
            globus_utp_name_timer(_nx_critical_path_stop_timer,
                           "critical_path_timer_send msg_len:%d testno:%d",
                           msg_size, ntest);
#else
            _nx_critical_path_stop_timer = _nx_critical_path_start_timer;
            globus_utp_name_timer(_nx_critical_path_start_timer,
                           "critical_path_timer msg_len:%d testno:%d",
                           msg_size, ntest);
#endif
#endif    

	    /* Initialize the monitor */
	    monitor->done = GLOBUS_FALSE;
	    monitor->count = n_iters;
	    monitor->length = msg_size;
	    monitor->iter = 0;
	    monitor->timernum = timer_index;

	    /* Send message to pong_starter to initialize its monitor. */
	    globus_nexus_startpoint_bind(&my_sp, &(monitor->ep));
	    globus_nexus_buffer_init(&buffer,
				     (globus_nexus_sizeof_startpoint(&my_sp, 1)
				      + (globus_nexus_sizeof_int(1) * 2)),
				     0);
	    globus_nexus_put_startpoint_transfer(&buffer, (&my_sp), 1);
	    globus_nexus_put_int(&buffer, &monitor->count, 1);
	    globus_nexus_put_int(&buffer, &monitor->length, 1);
	    globus_nexus_send_rsr(&buffer,
				  pong_sp,
				  PONG_STARTER_HANDLER_ID,
				  GLOBUS_TRUE,
				  GLOBUS_FALSE);
	    
	    /* Wait for the ping pong to complete */
	    globus_mutex_lock(&(monitor->mutex));
	    while (!monitor->done)
	    {
		globus_cond_wait(&(monitor->cond), &(monitor->mutex));
	    }
	    globus_mutex_unlock(&(monitor->mutex));
	    
	    /* Update n_iters */
	    globus_utp_get_accum_time(monitor->timernum, &time, &precision);
#ifdef DISPLAY_TIMES
	    globus_libc_lock();
	    printf("Results:\n");
	    printf("                     roundtrip count: %d\n", n_iters);
	    printf("              message length (bytes): %d\n", msg_size);
	    printf("                   elapsed time (us): %f\n",
                   time*1000000);
	    printf("         average roundtrip time (us): %f\n",
		   time*1000000/n_iters);
	    printf("    latency (average oneway time,us): %f\n",
		   (time*1000000/n_iters)/2);
	    printf("            bandwidth (bytes/second): %f\n",
		   (n_iters*msg_size*2)/time);
	    printf("\n");
	    globus_libc_unlock();
#endif /* DISPLAY_TIMES */
	    n_iters = PP_MAX(((int) ((test_run_time / time) * n_iters)),
			     MIN_ITERS);
	}
    }

    globus_libc_lock();
#ifdef CHECKMESSAGE
    if (Error)
    {
        printf("Done.  ERROR: some messages as a result of communication.\n");
    }
    else
    {
        printf("Done.  All messages received in tact.\n");
    }
#else
    printf("Done.\n");
#endif /* CHECKMESSAGE */
    globus_libc_unlock();

    /* Free resources */
    globus_nexus_endpoint_destroy(&(monitor->ep));
    globus_mutex_destroy(&(monitor->mutex));
    globus_cond_destroy(&(monitor->cond));
    globus_libc_free(monitor->message);
    globus_libc_free(monitor);
#ifdef CHECKMESSAGE
    globus_libc_free(OriginalMessage);
#endif /* CHECKMESSAGE */

    globus_utp_write_file(DEFAULT_UTP_FILE);
    
    globus_module_deactivate(GLOBUS_UTP_MODULE);
    
} /* pp_driver() */


/*
 * pong_starter_handler()
 */
static void pong_starter_handler(globus_nexus_endpoint_t *endpoint,
				 globus_nexus_buffer_t *buffer,
				 globus_bool_t is_nonthreaded_handler)
{
    globus_nexus_startpoint_t my_sp;
    globus_nexus_buffer_t send_buffer;
    message_monitor_t *monitor;

    pp_printf(("pong_starter_handler(): entering\n"));

    /* Initialize monitor */
    monitor = (message_monitor_t *)
	globus_libc_malloc(sizeof(message_monitor_t));
    globus_nexus_endpoint_init(&(monitor->ep), &global_epattr);
    globus_nexus_endpoint_set_user_pointer(&(monitor->ep), (void *) monitor);
    globus_mutex_init(&(monitor->mutex), (globus_mutexattr_t *) NULL);
    globus_cond_init(&(monitor->cond), (globus_condattr_t *) NULL);
    monitor->done = GLOBUS_FALSE;
    monitor->iter = 0;
    globus_nexus_get_startpoint(buffer, &(monitor->other_sp), 1);
    globus_nexus_get_int(buffer, &(monitor->count), 1);
    globus_nexus_get_int(buffer, &(monitor->length), 1);
    if (monitor->length > 0)
    {
	monitor->message = (char *) globus_libc_malloc(monitor->length);
	/* Should probably randomize the message body */
    }

    /* Give the ping side a moment to settle down */
    globus_libc_usleep(5000);
    
    /* Send to ping handler, to start off the ping pong */
    globus_nexus_startpoint_bind(&my_sp, &(monitor->ep));
    globus_nexus_buffer_init(&send_buffer,
			     globus_nexus_sizeof_startpoint(&my_sp, 1),
			     0);
    globus_nexus_put_startpoint_transfer(&send_buffer, (&my_sp), 1);
    globus_nexus_send_rsr(&send_buffer,
			  &(monitor->other_sp),
			  PING_STARTER_HANDLER_ID,
			  GLOBUS_TRUE,
			  is_nonthreaded_handler);

    ACK_PROTOCOL_ON();

    pp_printf(("pong_starter_handler(): exiting\n"));

} /* pong_starter_handler() */


/*
 * ping_starter_handler()
 */
static void ping_starter_handler(globus_nexus_endpoint_t *endpoint,
				 globus_nexus_buffer_t *buffer,
				 globus_bool_t is_nonthreaded_handler)
{
    message_monitor_t *monitor;
    globus_nexus_buffer_t send_buffer;

    pp_printf(("ping_starter_handler(): entering\n"));

    monitor = (message_monitor_t *)
	globus_nexus_endpoint_get_user_pointer(endpoint);
    
    /* Get the global pointer to the pong monitor */
    globus_nexus_get_startpoint(buffer, &(monitor->other_sp), 1);

    /* Start the timer */
#ifdef GLOBUS_NEXUS_CRITICAL_PATH_TIMER    
    _nx_time_critical_path = GLOBUS_TRUE;
#endif
    ACK_PROTOCOL_ON();
    globus_utp_start_timer(monitor->timernum);
    
    /* Send the first message to the pong_handler */
    if (monitor->length > 0)
    {
	globus_nexus_buffer_init(&send_buffer,
				 globus_nexus_sizeof_char(monitor->length),
				 0);
#ifdef CHECKMESSAGE
        /* making a copy of original randomized message to check against  */
        /* what is received later (after each round trip) in ping_handler */
        memcpy(OriginalMessage, monitor->message, monitor->length);
#endif /* CHECKMESSAGE */

	SPOOF_COPY_ON();
	globus_nexus_put_char(&send_buffer, monitor->message, monitor->length);
	SPOOF_COPY_OFF();
    }
    else
    {
	globus_nexus_buffer_init(&send_buffer, 0, 0);
    }
    globus_nexus_send_rsr(&send_buffer,
			  &(monitor->other_sp),
			  PONG_HANDLER_ID,
			  GLOBUS_TRUE,
			  is_nonthreaded_handler);

    pp_printf(("ping_starter_handler(): exiting\n"));

} /* ping_starter_handler() */


/*
 * pong_handler()
 */
static void pong_handler(globus_nexus_endpoint_t *endpoint,
			 globus_nexus_buffer_t *buffer,
			 globus_bool_t is_nonthreaded_handler)
{
    message_monitor_t *monitor;
    globus_nexus_buffer_t send_buffer;

    pp_printf(("pong_handler(): entering\n"));

    monitor = (message_monitor_t *)
	globus_nexus_endpoint_get_user_pointer(endpoint);
    
    /* Copy the message to my buffer */
    if (monitor->length > 0)
    {
	SPOOF_COPY_ON();
	globus_nexus_get_char(buffer, monitor->message, monitor->length);
	SPOOF_COPY_OFF();
    }
       
    /* Bounce message back to ping_handler */
    if (monitor->length > 0)
    {
	globus_nexus_buffer_init(&send_buffer,
				 globus_nexus_sizeof_char(monitor->length),
				 0);
	SPOOF_COPY_ON();
	globus_nexus_put_char(&send_buffer, monitor->message, monitor->length);
	SPOOF_COPY_OFF();
    }
    else
    {
	globus_nexus_buffer_init(&send_buffer, 0, 0);
    }
    globus_nexus_send_rsr(&send_buffer,
			  &(monitor->other_sp),
			  PING_HANDLER_ID,
			  GLOBUS_TRUE,
			  is_nonthreaded_handler);

    /* If end of test, free up pong monitor */
    if (++monitor->iter >= monitor->count)
    {
        ACK_PROTOCOL_OFF();
	globus_nexus_endpoint_destroy(&(monitor->ep));
	globus_mutex_destroy(&(monitor->mutex));
	globus_cond_destroy(&(monitor->cond));
	globus_nexus_startpoint_destroy(&(monitor->other_sp));
	if (monitor->length > 0)
	{
	    globus_libc_free(monitor->message);
	}
	globus_libc_free(monitor);
    }

    globus_nexus_buffer_destroy(buffer);

} /* pong_handler() */


/*
 * ping_handler()
 */
static void ping_handler(globus_nexus_endpoint_t *endpoint,
			 globus_nexus_buffer_t *buffer,
			 globus_bool_t is_nonthreaded_handler)
{
    message_monitor_t *monitor;
    globus_nexus_buffer_t send_buffer;
#ifdef CHECKMESSAGE
    int i;
    char *cp;
#endif /* CHECKMESSAGE */

    pp_printf(("ping_handler(): entering, iter=%d, count=%d\n",
	       monitor->iter, monitor->count));

    monitor = (message_monitor_t *)
	globus_nexus_endpoint_get_user_pointer(endpoint);
    
    /* Copy the message to my buffer */
    if (monitor->length > 0)
    {
	SPOOF_COPY_ON();
	globus_nexus_get_char(buffer, monitor->message, monitor->length);
	SPOOF_COPY_OFF();
#ifdef CHECKMESSAGE
        if (memcmp(monitor->message, OriginalMessage, monitor->length))
        {
            /*
	     * ERROR: received message different than
	     * originally sent message
	     */

            Error = 1;

            /* printing the discrepencies in the smaller buffers only */
            if (monitor->length < 100)
            {
                globus_libc_lock();
                printf("ping_handler(): Original Message >");
                for (i = 0, cp = OriginalMessage;
                    i < monitor->length; cp ++, i ++)
                        printf("%02x", *cp);
                printf("<\n");
                printf("ping_handler():  Received Buffer >");
                for (i = 0, cp = monitor->message;
                    i < monitor->length; cp ++, i ++)
                        printf("%02x", *cp);
                printf("<\n");
                globus_libc_unlock();
            } /* endif */
globus_nexus_fatal("ping_handler(): msg len %d: rcvd message does not match OriginalMessage\n", monitor->length);
        } /* endif */
#endif /* CHECKMESSAGE */

    }
       
#ifdef GLOBUS_NEXUS_SPLIT_CRITICAL_PATH_TIMER
    globus_utp_stop_timer(_nx_critical_path_start_timer);
    globus_utp_start_timer(_nx_critical_path_stop_timer);
#endif

    if (++monitor->iter < monitor->count)
    {
	/* Bounce message to other node */
	if (monitor->length > 0)
	{
	    globus_nexus_buffer_init(&send_buffer,
				     globus_nexus_sizeof_char(monitor->length),
				     0);
	    SPOOF_COPY_ON();
	    globus_nexus_put_char(&send_buffer, monitor->message, monitor->length);
	    SPOOF_COPY_OFF();
	}
	else
	{
	    globus_nexus_buffer_init(&send_buffer, 0, 0);
	}
	globus_nexus_send_rsr(&send_buffer,
			      &(monitor->other_sp),
			      PONG_HANDLER_ID,
			      GLOBUS_TRUE,
			      is_nonthreaded_handler);
    }
    else
    {
#ifdef GLOBUS_NEXUS_CRITICAL_PATH_TIMER
	if (_nx_critical_path_timer_started)
	{
	    globus_utp_stop_timer(_nx_critical_path_stop_timer);
	    _nx_critical_path_timer_started = GLOBUS_FALSE;
	}
	_nx_time_critical_path = GLOBUS_FALSE;
#endif
    
	/* End of test */
	globus_utp_stop_timer(monitor->timernum);

	pp_printf(("ping_handler(): end of test\n"));
	
        ACK_PROTOCOL_OFF();

	/* Wake up the initial thread */
	globus_mutex_lock(&(monitor->mutex));
	monitor->done = GLOBUS_TRUE;
	globus_cond_signal(&(monitor->cond));
	globus_mutex_unlock(&(monitor->mutex));
    }

    globus_nexus_buffer_destroy(buffer);

} /* ping_handler() */
