/**************************************************************************
*
* Transform        COMPRESSION 
*                  ZLIB - algorithm: a variation of LZ77 (Lempel-Ziv 1977)
*
* Module:          transform_gzip.c
*
* Written at:      High-Performance Computing Laboratory
*                  Department of Computer Science
*                  Northern Illinois University
*                  DeKalb, IL
*
* Original Author: Doug Sale   
*
* Description:
*
**************************************************************************/

#include <nexus.h>
#include "internal.h"
#include <zlib.h>

/********************************/
/* Externally Visible Functions */
/********************************/

void *_nx_transform_gzip_info(void);

/*******************/
/* Local Functions */
/*******************/
static int unzip(nexus_byte_t *destination_start,
		unsigned long *destination_size,
		nexus_byte_t *data_start,
		unsigned long data_size);

/******************************/
/* Transform Table Functions */
/*****************************/

static int transform_gzip_transform_id();
static void transform_gzip_init(nexus_bool_t   *modifies_data,
				unsigned long  *header_size,
				unsigned long  *trailer_size);
static void transform_gzip_shutdown(); 
static void transform_gzip_transformattr_init(void *info,
						nexus_transformattr_t **attr);
static void transform_gzip_transformattr_destroy(nexus_transformattr_t *attr);
static void transform_gzip_transformattr_get_info(nexus_transformattr_t *attr,
							void **info);
static void transform_gzip_init_endpoint_state(nexus_transformattr_t *attr, 
						nexus_transformstate_t **ep_state);
static void transform_gzip_destroy_endpoint_state(
						nexus_transformstate_t *ep_state);
static void transform_gzip_update_endpoint_state(
						nexus_transformstate_t *ep_state,
						nexus_transformstate_t *sp_state);
static void transform_gzip_init_startpoint_state(
						nexus_transformstate_t *ep_state,
						nexus_transformstate_t **sp_state,
						nexus_bool_t  *copy_sp_locally,
						nexus_bool_t  *destroy_sp_locally);
static void transform_gzip_destroy_startpoint_state(
						nexus_transformstate_t *sp_state);
static void transform_gzip_copy_startpoint_state(
						nexus_transformstate_t *sp_state,
						nexus_transformstate_t **sp_state_copy);
static int transform_gzip_sizeof_startpoint_state(nexus_transformstate_t *s);
static void transform_gzip_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state);
static void transform_gzip_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state);
static int transform_gzip_transform(nexus_transformstate_t *startpoint_state,
					nexus_byte_t *storage_start,
					unsigned long storage_size,
					nexus_byte_t *data_start,
					unsigned long *data_size,
					nexus_bool_t must_alloc_new_buffer,
					nexus_byte_t *transform_info_start,
					nexus_byte_t **out_storage_start,
					unsigned long *out_storage_size,
					nexus_byte_t **out_data_start,
					unsigned long *out_data_size);

static int transform_gzip_untransform(nexus_transformstate_t *endpoint_state,
					nexus_byte_t *data_start,
					unsigned long *data_size,
					nexus_byte_t *transform_info_start,
					int format,
					nexus_byte_t **destination_start,
					unsigned long *destination_size);
 
/*******************/
/* Data Structures */
/*******************/

typedef struct _endpoint_transform_state_t
{
	/* compression level */
	nexus_byte_t level;
} endpoint_transform_state_t; 

typedef struct _startpoint_transform_state_t
{
	/* compression level */
	nexus_byte_t level;
} startpoint_transform_state_t;

/**************************/
/* Local Global Variables */
/**************************/

static nexus_transform_funcs_t transform_gzip_funcs =
{
	transform_gzip_transform_id,
	transform_gzip_init,
	transform_gzip_shutdown,
	transform_gzip_transformattr_init,
	transform_gzip_transformattr_destroy,
	transform_gzip_transformattr_get_info,
	transform_gzip_init_endpoint_state,
	transform_gzip_destroy_endpoint_state,
	transform_gzip_update_endpoint_state,
	transform_gzip_init_startpoint_state,
	NULL, /* transform_gzip_copy_startpoint_state, */
	      /* do not provide copy_sp routine when copy_locally = NEXUS_FALSE */
	transform_gzip_destroy_startpoint_state,
	transform_gzip_sizeof_startpoint_state,
	transform_gzip_put_startpoint_state,
	transform_gzip_get_startpoint_state,
	transform_gzip_transform,
	transform_gzip_untransform
};

static unsigned char *work_buf;
static unsigned long work_size = 0;

/********************************/
/* Externally Visible Functions */
/********************************/

/*
 * _nx_transform_gzip_info()
 *
 * Return the function table for this module.
 */
void *_nx_transform_gzip_info(void)
{
	return ((void *) (&transform_gzip_funcs));
}

/*******************/
/* Local Functions */
/*******************/

/*
 * transform_gzip_transform_id()
 */
static int transform_gzip_transform_id()
{
	return(NEXUS_TRANSFORM_GZIP_ID);
} /* transform_gzip_transform_id() */

/*
 * transform_gzip_init()
 */
static void transform_gzip_init(nexus_bool_t   *modifies_data,
				unsigned long  *header_size,   
				unsigned long  *trailer_size)
{
	NexusAssert2((modifies_data), 
		("transform_gzip_init(): rcvd NULL modifies_data\n"));
	NexusAssert2((header_size), 
		("transform_gzip_init(): rcvd NULL header_size\n"));
	NexusAssert2((trailer_size), 
		("transform_gzip_init(): rcvd NULL trailer_size\n"));
	NexusAssert2((sizeof(unsigned long) <= LARGEST_POSSIBLE_SIZEOF_ULONG), 
		("transform_gzip_init(): detected sizeof(unsigned long) > LARGEST_POSSIBLE_SIZEOF_ULONG\n"));
	
	*modifies_data = NEXUS_TRUE;
	
	/* transform info size: uncompressed length and compression level */
	*header_size = sizeof(unsigned long) + sizeof(nexus_byte_t);
	
	/* compression requires separate work buffer and resulting data 
	   should be of smaller or the same size */
	*trailer_size = 0; 

} /* transform_gzip_init() */

/*
 * transform_gzip_shutdown()
 */
static void transform_gzip_shutdown()
{
	/* here is where the work buffer gets freed */
	nexus_free((void *)work_buf);
} /* transform_gzip_shutdown() */

/*
 * transform_gzip_transformattr_init()
 */
static void transform_gzip_transformattr_init(void *info,
						nexus_transformattr_t **attr)
{
	NexusAssert2((info), 
		("transform_gzip_transformattr_init(): rcvd NULL info\n"));
	
	/* allocate mem for attr and assign it the value of info */
	*attr = (nexus_transformattr_t *)nexus_malloc(sizeof(nexus_byte_t));
	memcpy(*attr, info, sizeof(nexus_byte_t));

} /* transform_gzip_transformattr_init() */

/*
 * transform_gzip_transformattr_destroy()
 */
static void transform_gzip_transformattr_destroy(nexus_transformattr_t *attr)
{
	NexusAssert2((attr), 
		("transform_gzip_transformattr_destroy(): rcvd NULL transform attr\n"));
	
	/* free mem allocated for attr */
	nexus_free(attr); 

} /* transform_gzip_transformattr_destroy() */

/*
 * transform_gzip_transformattr_get_info()
 */
static void transform_gzip_transformattr_get_info(nexus_transformattr_t *attr,
							void **info)
{
	NexusAssert2((attr), 
		("transform_gzip_transformattr_get_info(): rcvd NULL transform attr\n"));
	
	/* allocate mem for info and assign it the value of attr */
	*info = (void *)nexus_malloc(sizeof(nexus_byte_t));
	memcpy(*info, attr, sizeof(nexus_byte_t));

} /* transform_gzip_transformattr_get_info() */

/*
 * transform_gzip_init_endpoint_state()
 */
static void transform_gzip_init_endpoint_state(nexus_transformattr_t *attr,
						nexus_transformstate_t **ep_state)
{
	/*nexus_printf("transform_gzip_init_endpoint_state\n");*/
	
	/* allocate mem for ep transform state and assign the transform attr to level */
	*ep_state = (endpoint_transform_state_t *)
	nexus_malloc(sizeof(endpoint_transform_state_t));
	((endpoint_transform_state_t *) (*ep_state))->level = *((nexus_byte_t *)attr);

	/*nexus_printf("transform_gzip_init_endpoint_state(): compression level = %d\n", 
		(int)(((endpoint_transform_state_t *) (*ep_state))->level));*/
	
} /* transform_gzip_init_endpoint_state() */

/*
 * transform_gzip_destroy_endpoint_state()
 */
static void transform_gzip_destroy_endpoint_state(nexus_transformstate_t *ep_state)
{
	/*nexus_printf("transform_gzip_destroy_endpoint_state\n");*/
	
	NexusAssert2((ep_state), 
		("transform_gzip_destroy_endpoint_state(): rcvd NULL ep_state\n"));
	
	nexus_free(ep_state);
	
} /* transform_gzip_destroy_endpoint_state() */

/*
 * transform_gzip_update_endpoint_state()
 */
static void transform_gzip_update_endpoint_state(nexus_transformstate_t *ep_state,
						nexus_transformstate_t *sp_state)
{
	/* nexus_printf("transform_gzip_update_endpoint_state\n");*/
	
	NexusAssert2((ep_state), 
		("transform_gzip_update_endpoint_state(): rcvd NULL ep_state\n"));
	NexusAssert2((sp_state), 
		("transform_gzip_update_endpoint_state(): rcvd NULL sp_state\n"));
	
	/*nexus_printf("***********TODO write transform_gzip_update_endpoint_state()\n");*/
	
} /* end transform_gzip_update_endpoint_state() */

/*
 * transform_gzip_init_startpoint_state()
 */
static void transform_gzip_init_startpoint_state(nexus_transformstate_t *ep_state,
						nexus_transformstate_t **sp_state,
						nexus_bool_t  *copy_sp_locally,
						nexus_bool_t  *destroy_sp_locally)
{
	/*nexus_printf("transform_gzip_init_startpoint_state\n");*/
	
	NexusAssert2((ep_state), 
		("transform_gzip_init_startpoint_state(): rcvd NULL ep_state\n"));
	NexusAssert2((sp_state), 
		("transform_gzip_init_startpoint_state(): rcvd NULL sp_state\n"));
	NexusAssert2((copy_sp_locally), 
		("transform_gzip_init_startpoint_state(): rcvd NULL copy_sp_locally\n"));
	NexusAssert2((destroy_sp_locally), 
		("transform_gzip_init_startpoint_state(): rcvd NULL destroy_sp_locally\n"));
	
		/* allocate mem for sp state */
	*sp_state = (startpoint_transform_state_t *) 
			nexus_malloc(sizeof(startpoint_transform_state_t));
	
	/* initializing sp state's compression level */
	((startpoint_transform_state_t *)(*sp_state))->level = 
		((endpoint_transform_state_t *) ep_state)->level;
		
	*copy_sp_locally = NEXUS_FALSE;
	*destroy_sp_locally = NEXUS_FALSE;

} /* transform_gzip_init_startpoint_state() */

/*
 * transform_gzip_destroy_startpoint_state()
 */
static void transform_gzip_destroy_startpoint_state(nexus_transformstate_t *sp_state)
{
	/*nexus_printf("transform_gzip_destroy_startpoint_state\n");*/
	NexusAssert2((sp_state), 
		("transform_gzip_destroy_startpoint_state(): rcvd NULL sp_state\n"));
	nexus_free(sp_state); 

} /* transform_gzip_destroy_startpoint_state() */

/*
 * transform_gzip_sizeof_startpoint_state()
 */
static int transform_gzip_sizeof_startpoint_state(nexus_transformstate_t *s)
{
	NexusAssert2((s), ("transform_gzip_sizeof_startpoint_state(): rcvd NULL s\n"));

	/* compression level */
	return(sizeof(nexus_byte_t));

} /* transform_gzip_sizeof_startpoint_state() */

/*
 * transform_gzip_put_startpoint_state()
 */
static void transform_gzip_put_startpoint_state(nexus_byte_t **buffer, 
						nexus_transformstate_t *state)
{
	/*nexus_printf("transform_gzip_put_startpoint_state\n");*/
	
	NexusAssert2((buffer), 
		("transform_gzip_put_startpoint_state(): rcvd NULL buffer\n"));
	NexusAssert2((state), 
		("transform_gzip_put_startpoint_state(): rcvd NULL state\n"));
	
	nexus_dc_put_u_char(buffer, &(((startpoint_transform_state_t *) state)->level), 1);

} /* transform_gzip_put_startpoint_state() */

/*
 * transform_gzip_get_startpoint_state()
 */
static void transform_gzip_get_startpoint_state(nexus_byte_t **buffer, 
						int format, 
						nexus_transformstate_t **state)
{
	/*nexus_printf("transform_gzip_get_startpoint_state\n");*/
	
	NexusAssert2((buffer), 
		("transform_gzip_get_startpoint_state(): rcvd NULL buffer\n"));
	NexusAssert2((state), 
		("transform_gzip_get_startpoint_state(): rcvd NULL state\n"));
	
	*state = (startpoint_transform_state_t *) 
		nexus_malloc(sizeof(startpoint_transform_state_t));
	
	nexus_dc_get_u_char(buffer, &(((startpoint_transform_state_t *)(*state))->level),
				1, format); 

} /* transform_gzip_get_startpoint_state() */
 
/*
 * transform_gzip_transform()
 */
static int transform_gzip_transform(nexus_transformstate_t *startpoint_state,
					nexus_byte_t *storage_start,
					unsigned long storage_size,
					nexus_byte_t *data_start,
					unsigned long *data_size,
					nexus_bool_t must_alloc_new_buffer,
					nexus_byte_t *transform_info_start,
					nexus_byte_t **out_storage_start,
					unsigned long *out_storage_size,
					nexus_byte_t **out_data_start,
					unsigned long *out_data_size)
{
	int rc;
	z_stream stream;
	nexus_byte_t  level;
	unsigned long compressed_size,
	              uncompressed_size;
	
	NexusAssert2((startpoint_state), ("transform_gzip_transform(): rcvd NULL state\n"));
	NexusAssert2((storage_start), ("transform_gzip_transform(): rcvd NULL storage_start\n"));
	NexusAssert2((data_start), ("transform_gzip_transform(): rcvd NULL data_start\n"));
	NexusAssert2((data_size), ("transform_gzip_transform(): rcvd NULL data_size\n"));
	NexusAssert2((transform_info_start), ("transform_gzip_transform(): rcvd NULL transform_info_start\n"));
	NexusAssert2((out_storage_start), ("transform_gzip_transform(): rcvd NULL out_storage_start\n"));
	NexusAssert2((out_storage_size), ("transform_gzip_transform(): rcvd NULL out_storage_size\n"));
	NexusAssert2((out_data_start), ("transform_gzip_transform(): rcvd NULL out_data_start\n"));
	NexusAssert2((out_data_size), ("transform_gzip_transform(): rcvd NULL out_data_size\n"));
	
	/*nexus_printf("transform_gzip_transform: data_size = %d\n", *data_size);*/
	
	/* calc space needed to compress in */
	compressed_size = *data_size + 12 + ((*data_size < 1000)? 1 : *data_size/1000);
	
	/* save uncompressed size - used to uncompress on receiving end */
	uncompressed_size = *data_size;
	
	/* allocate work space if not enough already available */
	if (compressed_size > work_size)
	{
		if (work_size > 0)
			nexus_free(work_buf);

		work_buf = (unsigned char *)nexus_malloc(compressed_size);
	}
	
	/* get compression level from startpoint state */
	level = ((startpoint_transform_state_t *)(startpoint_state))->level; 
/*nexus_printf("transform_gzip_transform - using compression level %d\n", level);*/

	/* initialize gzip data structure */
	stream.next_in = (Bytef*)data_start/*source*/;
	stream.avail_in = (uInt)(*data_size)/*sourceLen*/;

	/* default return code of 'ok' */
        rc = Z_OK;

#ifdef MAXSEG_64K
	/* Check for *data_size > 64K on 16-bit machine: */
	if ((uLong)stream.avail_in != *data_size)/*sourceLen*/) rc = Z_BUF_ERROR;
#endif

	if (rc == Z_OK)
	{	
		/* more gzip data struct initialization */
		stream.next_out = work_buf/*dest*/;
		stream.avail_out = (uInt)compressed_size/**destLen*/;

		/* check for casting errors */
		if ((uLong)stream.avail_out != compressed_size/**destLen*/)
		{
			rc = Z_BUF_ERROR;
		}
		else
		{
			/* set fxn ptrs to null (use default gzib fxns) */
			stream.zalloc = (alloc_func)0;
			stream.zfree = (free_func)0;
			stream.opaque = (voidpf)0;

			rc = deflateInit(&stream, level/*comprLevel*/);
			if (rc == Z_OK)
			{
				/* compression */
				rc = deflate(&stream, Z_FINISH);
				if (rc != Z_STREAM_END) 
				{
					deflateEnd(&stream);
					rc = (rc == Z_OK)? Z_BUF_ERROR : rc;
				}
			}
		}
	}

	compressed_size/**destLen*/ = stream.total_out;
	rc = deflateEnd(&stream);

	/*nexus_printf("transform_gzip_transform.compress(): rc = %d\n", rc);*/

	if (must_alloc_new_buffer)
	{
	/*nexus_printf("transform_gzip_transform: must allocate new buffer\n");*/

		/* if compress useful, copy to data_start and set *data_size */
		if ((compressed_size < *data_size) && (rc == Z_OK))
		{
	/*nexus_printf("transform_gzip_transform: successful compress - %lu\n", compressed_size);*/

			*out_storage_size = data_start - storage_start + compressed_size;	
			*out_data_size = compressed_size;
			*out_storage_start = (nexus_byte_t *)nexus_malloc(*out_data_size);
			*out_data_start = (*out_storage_start) + (data_start - storage_start);
			memcpy((void *)*out_data_start, (void *)work_buf, compressed_size);
		}
		else /* compressed data larger, or compress() bombed */
		     /* copy data over, set compression level to no compression */
		{
		/*nexus_printf("transform_gzip_transform: unsuccessful compress - %lu\n", *data_size);*/
	
			*out_storage_size = data_start - storage_start + *data_size;	
			*out_data_size = *data_size;
			*out_storage_start = (nexus_byte_t *)nexus_malloc(*out_data_size);
			*out_data_start = (*out_storage_start) + (data_start - storage_start);
			memcpy((void *)*out_data_start, (void *)data_start, *data_size);
			level = Z_NO_COMPRESSION;
		}
	}
	else
	{
		/* if compress useful, copy to data_start and set *data_size */
		if ((compressed_size < *data_size) && (rc == Z_OK))
		{
		/*nexus_printf("transform_gzip_transform(): successful compress - %lu\n", compressed_size);*/
			memcpy((void *)data_start, (void *)work_buf, compressed_size);
			*data_size = compressed_size;
		}
		/* else leave buffer as it was on entry */
		/* set transform level to no compression in *transform_info */	
		else
		{
		/*nexus_printf("transform_gzip_transform(): unsuccessful compress - %lu\n", *data_size);*/
			compressed_size = *data_size;
			level = Z_NO_COMPRESSION;
		} 
	}
	
	/* put uncompressed size and compression level in transform info */
	nexus_dc_put_u_long(&transform_info_start, &uncompressed_size, 1);
	nexus_dc_put_u_char(&transform_info_start, &level, 1);
	
	/*nexus_printf("exit transform_gzip_transform()\n");*/
	return(NEXUS_SUCCESS);

} /* transform_gzip_transform() */

/*
 * transform_gzip_untransform()
 */
static int transform_gzip_untransform(nexus_transformstate_t *endpoint_state,
					nexus_byte_t *data_start,
					unsigned long *data_size,
					nexus_byte_t *transform_info_start,
					int format,
					nexus_byte_t **destination_start,
					unsigned long *destination_size)
{
	int rc;
	nexus_byte_t  level;
	unsigned long uncompressed_size;

	NexusAssert2((endpoint_state),
		("transform_ecb_untransform(): rcvd NULL endpoint_state\n"));
	NexusAssert2((data_start),
		("transform_ecb_untransform(): rcvd NULL data_start\n"));
	NexusAssert2((data_size),
		("transform_ecb_untransform(): rcvd NULL data_size\n"));
	NexusAssert2((transform_info_start),
		("transform_ecb_untransform(): rcvd NULL transform_info_start\n"));
	NexusAssert2((destination_start),
		("transform_ecb_untransform(): rcvd NULL destination_start\n"));
	NexusAssert2((destination_size),
		("transform_ecb_untransform(): rcvd NULL destination_size\n"));
	
	/*nexus_printf("enter transform_gzip_untransform()\n");*/
	
	/* get the transform info out - uncompressed buffer size and compression level */
	nexus_dc_get_u_long(&transform_info_start, &uncompressed_size, 1, format);
	nexus_dc_get_u_char(&transform_info_start, &level, 1, format);
	
	if(*destination_start == NULL)
	{	
		if (level == Z_NO_COMPRESSION)
		{
			/* if data wasn't compressed, leave in place */
			rc = Z_OK;
	
		/*nexus_printf("transform_gzip_untransform(): not uncompressing\n");*/
		}
		else /* create destination (also workspace) and uncompress */
		{
			/* malloc space for destination */
			*destination_start = (nexus_byte_t *)nexus_malloc(uncompressed_size);
			*destination_size = uncompressed_size;
	
			/* uncompress */
			rc = unzip(*destination_start, destination_size,
	                	data_start, *data_size);
		/*nexus_printf("transform_gzip_untransform(): uncompressing - %lu\n", *destination_size);*/
		}
	}
	else
	{
		if (*destination_size == uncompressed_size)
		{
			if (level == Z_NO_COMPRESSION)
			{
				/* if not compressed, just copy data into destination */
				memcpy((void *)*destination_start,
					       (void *)data_start,
					       uncompressed_size);
				rc = Z_OK;
		/*nexus_printf("transform_gzip_untransform(): not uncompressing\n");*/
			}
			else
			{
				/* uncompress into destination */
				rc = unzip(*destination_start, destination_size,
				                data_start, *data_size);
		/*nexus_printf("transform_gzip_untransform(): uncompressing - %lu\n", *destination_size);*/
			}
		}
		/* else can't untransform */
		else 
		{
			rc = -1;
			/*nexus_printf("transform_gzip_untransform(): cannot untransform\n");*/
		}
	}
	
	/*nexus_printf("exit transform_gzip_untransform()\n");*/
	if (rc == Z_OK) return(NEXUS_SUCCESS);
	else            return(NEXUS_FAILURE);
	
} /* transform_gzip_untransform */

/*
 * unzip()
 */
static int unzip(nexus_byte_t *destination_start,
		unsigned long *destination_size,
		nexus_byte_t *data_start,
		unsigned long data_size)
{
    z_stream stream;
    int err;

    stream.next_in = (Bytef*)data_start;
    stream.avail_in = (uInt)data_size;

    /* Check for data_size > 64K on 16-bit machine: */
    if ((uLong)stream.avail_in != data_size) return Z_BUF_ERROR;

    stream.next_out = destination_start;
    stream.avail_out = (uInt)*destination_size;
    if ((uLong)stream.avail_out != *destination_size) return Z_BUF_ERROR;

    stream.zalloc = (alloc_func)0;
    stream.zfree = (free_func)0;

    err = inflateInit(&stream);
    if (err != Z_OK) return err;

    err = inflate(&stream, Z_FINISH);
    if (err != Z_STREAM_END) {
        inflateEnd(&stream);
        return err;
    }
    *destination_size = stream.total_out;

    err = inflateEnd(&stream);
    return err;
} /* unzip() */
