/* -*- c -*-
 * Nexus
 *
 * pr_xtp.C		- XTP Source-Ordered Reliable Multicast 
 *                        Message Delivery protocol module
 */

static char *rcsid =(char *) "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/libraries/nexus/pr_xtp.C,v 1.5 1999/09/01 20:35:21 bresnaha Exp $";

#include <stdlib.h>
#include "internal.h"

#if defined(HAVE_XTP_PROTO)

#include <sys/uio.h>
#include <sys/socket.h>
#include <netinet/in.h>
/* Need strings.h for bcopy...how can we do this in a mach-indep way? */
#include <strings.h>
#include <netdb.h>


#include "xtp/xtpif.h"
/*
 * GLOBUS_L_NEXUS_XTP_PROTOCOL_VERSION
 *
 * The version of this protocol module's wire protocol.  If you change
 * this module wire protocol, bump this version number.
 */
#define GLOBUS_L_NEXUS_XTP_PROTOCOL_VERSION (0+GLOBUS_I_NEXUS_BUFFER_VERSION)

/*
 * GLOBUS_L_NEXUS_XTP_MI_PROTO_VERSION
 *
 * The version of this protocol module's mi_proto.  If you change
 * the contents of this module's mi_proto, bump this version number.
 *
 * Currently not used, since XTP mi_proto structures aren't transmitted
 * between machines.
 */
#define GLOBUS_L_NEXUS_XTP_MI_PROTO_VERSION 0


/*
 * NEXUS_XTP_DIRECT_CUSTOM_MAX_SIZE
 *
 * Direct custom size is to determine at what message size nexus should switch
 * to fragmented buffers instead of inlining the direct components into the
 * base message.  (This is not implemented yet, use MAX_U_LONG to turn off)
 */
#if !defined(NEXUS_XTP_DIRECT_CUSTOM_MIN_SIZE)
#    define NEXUS_XTP_DIRECT_CUSTOM_MIN_SIZE NEXUS_DC_MAX_U_LONG
#endif

/*
 * Only one thread is allowed to be in the xtp code (and thus
 * mucking with data structures) at a time.
 */

#ifdef BUILD_LITE
#   define xtp_enter()
#   define xtp_exit()
#else  /* BUILD_LITE */
    static globus_mutex_t		xtp_mutex;
#   define xtp_enter() globus_mutex_lock(&xtp_mutex);
#   define xtp_exit()	globus_mutex_unlock(&xtp_mutex);
#endif /* BUILD_LITE */

static globus_bool_t		xtp_done;
static globus_list_t		*xtp_outgoing_list;
				/* list of all outgoings */
static globus_list_t		*xtp_incoming_list;
				/* list of all incomings */

static globus_list_t		*xtp_outgoing_select_list;
				/* list of outgoings trying to write */
static globus_list_t		*xtp_incoming_select_list;
				/* list of incomings trying to read */

static xtpif **			xtp_outgoing_select_cache_array;
static xtpif **			xtp_incoming_select_cache_array;
static globus_bool_t		xtp_outgoing_select_cache_count;
static globus_bool_t		xtp_incoming_select_cache_count;
static globus_bool_t 		xtp_outgoing_select_cache_valid;
static globus_bool_t 		xtp_incoming_select_cache_valid;

static globus_hashtable_t	xtp_incoming_hash_table;
				/* map nexus_endpoint_t to xtp_incoming_t */

#define xtp_fatal xtp_exit(); nexus_fatal

/*
 * Some useful queue macros--these implement a SLL with explicit
 * head and tail pointers.
 */
#define Enqueue(Qhead, Qtail, Item) \
{ \
    if (Qhead) \
    { \
	(Qtail)->next = (Item); \
	(Qtail) = (Item); \
    } \
    else \
    { \
	(Qhead) = (Qtail) = (Item); \
    } \
}

#define Dequeue(Qhead, Qtail, Item) \
{ \
    (Item) = (Qhead); \
    (Qhead) = (Qhead)->next; \
}

#define QueueNotEmpty(Qhead)	(Qhead)


/*
 * Define some xtp specific variables and constants
 */

/* 
 * Header size = 0...one-to-one correspondence between instance of an 
 * xtpif and a incoming or an outgoing, so no need for any other "stuff"
 * on the front end of an XTP packet for NEXUS's sake.
 */
#define XTP_HDR_SIZE         0

/*
 * Some forward references...
 */
typedef struct xtp_outgoing_s xtp_outgoing_t;
typedef struct xtp_incoming_s xtp_incoming_t;

/*
 * xtp_incoming_t
 *
 * XTP-specific information about an endpoint.
 */
struct xtp_incoming_s
{

    nexus_endpoint_t *ep;

    /*
     * additional Xtp specific fields
     */
    xtpif *xi;

    /*
     * state variables
     */
#define INCOMING_STATE_CLOSED 0
#define INCOMING_STATE_VERSION 1
#define INCOMING_STATE_MSG_FORMAT 2
#define INCOMING_STATE_MSG_SIZE 3
#define INCOMING_STATE_BODY 4
    int state;
    int remaining;
    globus_byte_t *storage;	/* start of buffer */
    globus_byte_t *current;	/* current point in buffer */

    globus_byte_t version;	/* version number of this message */
    globus_byte_t format;	/* format of message (one byte) */

#define XTP_MAX_ENCAP_LEN 16
    globus_byte_t encapfield[XTP_MAX_ENCAP_LEN]; /* buffer for encapsulation fields */
    unsigned long msglen;	/* length received from incoming message */

    struct globus_nexus_buffer_s *dispatch_q_head,
	*dispatch_q_tail;	/* queue of inbound message */

    globus_bool_t dispatch_in_progress;

};

/*
 * xtp_outgoing_t
 *
 * xtp_outgoing_t is an overload of nexus_proto_t.  Every Nexus
 * startpoint has a pointer to a nexus_proto_t and therefore each startpoint
 * using the xtp protocol has a pointer to a xtp_outgoing_t.
 *
 */
struct xtp_outgoing_s
{
    /*
     * required nexus_proto_t fields
     */
    nexus_proto_type_t			type;
    nexus_proto_funcs_t *		funcs;
    int					version;
    unsigned long			direct_custom_min_size;
    unsigned long			direct_custom_max_size;
    unsigned long			direct_pointer_min_size;
    unsigned long			direct_pointer_max_size;
    globus_bool_t			can_use_iovec;
    unsigned long			reserved_header_size;

    /* send queue */
    globus_bool_t		write_in_progress;
    struct globus_nexus_buffer_s *	write_q_head;
    struct globus_nexus_buffer_s *	write_q_tail;

    /*
     * additional Xtp specific fields
     */
    xtpif *xi;			/* XTP interface object */

    size_t offset;		/* offset into currently-sending segment */

    int diameter;		/* multicast diameter */
    struct in_addr addr;	/* multicast group address */
    short port;			/* port number */
    short16 edge_freq;		/* edge triggering frequency*/
    short16 edge_counter;	/* edge counter */

    globus_bool_t error;
    globus_bool_t connected;
    globus_bool_t sent_sreq;	/* sent sreq to try to free up buffers? */

    short16 config_options;	/* XTP options for config */
    short16 encap_options;	/* XTP options for encapsulation header */
    short16 payload_options;	/* XTP options for payload */

};

/*
 * nexus_proto_info_xtp_t
 *
 * Contains protocol-specific information passed to xtp_get_my_mi_proto,
 * as the proto_info parameter (if that parameter was a class instead
 * of void, this structure would be a descendant of that class).
 */
#if 0
typedef struct {

    globus_bool_t sender;	/* sender flag */
    int diameter;		/* multicast diameter */
    struct in_addr addr;	/* multicast group address */
    short port;     		/* port numbers */
    int sreq_freq;		/* how often to send SREQ flag */

} nexus_proto_info_xtp_t;
#endif

static globus_callback_handle_t        globus_l_nexus_xtp_callback_handle;
/*
 * Various forward declarations of procedures
 */
static void
xtp_init(
    globus_bool_t *			add_to_my_mi_proto);

static void
xtp_shutdown();

static int
xtp_send_rsr(
    struct globus_nexus_buffer_s *		buffer);

static globus_bool_t
xtp_send_rsr_outstanding(globus_nexus_proto_t *nproto);

static void
xtp_increment_reference_count(
    nexus_proto_t *			nproto);

static globus_bool_t
xtp_decrement_reference_count(
    nexus_proto_t *			nproto);

static int
xtp_get_my_mi_proto(
    globus_byte_t **			array,
    int *				size,
    void *				proto_info,
    nexus_endpoint_t *			endpoint);

static int
xtp_destroy_my_mi_proto(
    nexus_endpoint_t *			endpoint,
    globus_byte_t *			proto_array,
    int					size);

static globus_bool_t
xtp_construct_from_mi_proto(
    nexus_proto_t **			proto,
    nexus_mi_proto_t *			mi_proto,
    globus_byte_t *			proto_array,
    int					size);

static int
xtp_test_proto(
    nexus_proto_t *			proto);

static int
xtp_direct_info_size(void);

static void
outgoing_write_callback(
    void *				arg,
    int					fd,
    char *				buf,
    size_t				nbytes);

static void
outgoing_write_error_callback(
    void *				arg,
    int					fd,
    char *				buf,
    size_t				n_bytes,
    int					error);

static void
incoming_read_callback(
    void *				arg,
    int					fd,
    char *				buf,
    size_t				nbytes,
    char **				new_buf,
    size_t *				new_max_nbytes,
    size_t *				new_wait_for_nbytes);

static void
incoming_read_error_callback(
    void *				arg,
    int					fd,
    char *				buf,
    size_t				nbytes,
    int					error);

static globus_bool_t
xtp_poll(globus_time_t      time_can_block,
	 void *             user_args);

void xtp_register_for_write(xtp_outgoing_t *outgoing);
oid xtp_register_for_read(xtp_incoming_t *incoming);
void xtp_unregister_for_write(xtp_outgoing_t *outgoing);
void xtp_unregister_for_read(xtp_incoming_t *incoming);
void xtp_get_registered_read(void);
void xtp_get_registered_write(void);

void xtp_process_outgoing(xtpif *xi);
void xtp_process_incoming(xtpif *xi);

#endif /* HAVE_XTP_PROTO */

#define GLOBUS_L_XTP_PROTO_COUNT 1

extern "C"
{
static nexus_proto_type_t
xtp_proto_type(void);

static globus_bool_t    xtp_startpoint_proto_match(
                                          globus_nexus_mi_proto_t *   mi_proto0,
                                          int                         offset0,
                                          nexus_byte_t *              subarray0,
                                          int                         sub_length0,
                                          globus_nexus_mi_proto_t *   mi_proto1,
                                          int                         offset1,
                                          nexus_byte_t *              subarray1,
                                          int                         sub_length1);

static int              xtp_proto_count(void);
}

#if  defined(HAVE_XTP_PROTO)

static nexus_proto_funcs_t xtp_proto_funcs =
{
    xtp_proto_type,
    xtp_init,
    xtp_shutdown,
    xtp_increment_reference_count,
    xtp_decrement_reference_count,
    xtp_get_my_mi_proto,
    xtp_construct_from_mi_proto,
    xtp_destroy_my_mi_proto,
    xtp_test_proto,
    xtp_send_rsr,
    xtp_send_rsr_outstanding,
    xtp_direct_info_size,
    NULL /* xtp_direct_get */,
    xtp_startpoint_proto_match,
    xtp_proto_count,
/* XXX #ifdef BUILD_RESOURCE XXX */
};

#else

static nexus_proto_funcs_t xtp_proto_funcs =
{
    xtp_proto_type,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    GLOBUS_NULL,
    xtp_startpoint_proto_match,
    xtp_proto_count,
/* XXX #ifdef BUILD_RESOURCE XXX */
};

#endif /* HAVE_XTP_PROTO */

static globus_bool_t    xtp_startpoint_proto_match(
                                          globus_nexus_mi_proto_t *   mi_proto0,
                                          int                         offset0,
                                          nexus_byte_t *              subarray0,
                                          int                         sub_length0,
                                          globus_nexus_mi_proto_t *   mi_proto1,
                                          int                         offset1,
                                          nexus_byte_t *              subarray1,
                                          int                         sub_length1)
{
    return GLOBUS_TRUE;
}


static int              xtp_proto_count(void)
{
   return GLOBUS_L_XTP_PROTO_COUNT;
}

/*
 * Need to turn off C++ name mangling for this function only, as it will
 * get called by the C code inside the rest of Nexus.
 */
extern "C" {
    void *_nx_pr_xtp_info(void);
};

/*
 * _nx_pr_xtp_info()
 *
 * Return the nexus_proto_funcs_t function table for this protocol module.
 *
 * This procedure is used for bootstrapping the protocol module.
 * The higher level Nexus code needs to call this routine to
 * retrieve the functions it needs to use this protocol module.
 */
void *_nx_pr_xtp_info(void)
{
    return((void *) (&xtp_proto_funcs));
} /* _nx_pr_xtp_info() */


/*
 * xtp_proto_type()
 *
 * Return the nexus_proto_type_t for this protocol module.
 */
static nexus_proto_type_t xtp_proto_type(void)
{
    return (NEXUS_PROTO_TYPE_XTP);
} /* xtp_proto_type() */


#if  defined(HAVE_XTP_PROTO)

/*
 * xtp_init()
 *
 * Initialize the XTP protocol.
 */
static void xtp_init(globus_bool_t * add_to_my_mi_proto)
{

    int					i;
    int					rc;
    globus_reltime_t                    delay_time;

    nexus_debug_printf(
	2,
	("xtp_init(): entry\n"));

#   if !defined(BUILD_LITE)
    {
	globus_mutex_init(&xtp_mutex, (globus_mutexattr_t *) NULL);
    }
#   endif
   
    GlobusTimeReltimeSet(delay_time, 0, 0); 
    globus_callback_register_periodic(&globus_l_nexus_xtp_callback_handle,
				      &delay_time,
				      &delay_time,
				      xtp_poll,
				      GLOBUS_POLL,
				      GLOBUS_POLL,
				      GLOBUS_POLL);

    xtp_done = NEXUS_FALSE;

    /*
     * If we have some options that can be specified by a
     * configuration option on the command line, we can process those
     * here.
     */

    /*
     * Initialize other data structures.  Might need to tune some voodoo
     * constants in here.
     */
    globus_hashtable_init(&xtp_incoming_hash_table, 15, 
			  globus_hashtable_voidp_hash, 
			  globus_hashtable_voidp_keyeq);

    xtp_outgoing_list = NULL;
    xtp_incoming_list = NULL;
    xtp_outgoing_select_list = NULL;
    xtp_incoming_select_list = NULL;
    xtp_outgoing_select_cache_array = NULL;
    xtp_incoming_select_cache_array = NULL;
    xtp_outgoing_select_cache_valid = NEXUS_FALSE;
    xtp_incoming_select_cache_valid = NEXUS_FALSE;

    /*
     * Do not put xtp in default mi_proto list, and get_my_mi_proto() should
     * not be called during configuration
     */
    *add_to_my_mi_proto = NEXUS_FALSE;
}
/* xtp_init() */


/*
 * xtp_shutdown()
 *
 * This routine is called during normal shutdown of a process.
 *
 */
static void xtp_shutdown()
{
    int					i;
    int					fd;
    int					rc;

    xtp_enter();
    {
	globus_list_t *			comm_entry;
	xtp_done = NEXUS_TRUE;

	/*
	 * Walk a list of incoming_t and outgoing_t structures
	 * and close and throw them away.
	 *
	 * For incoming_t, release and deallocate and remove from hash.
	 *
	 * For outgoing_t, release but do not deallocate.
	 */
	globus_list_t *current;
	for (current = xtp_incoming_list; 
	     !globus_list_empty(current); 
	     current = globus_list_rest(current)) {

	    xtp_incoming_t *incoming = (xtp_incoming_t *) current->datum;

	    if (incoming->xi) {
		incoming->xi->release();
		delete(incoming->xi);
	    }
	    globus_hashtable_remove(&xtp_incoming_hash_table, 
				    (void *) incoming->ep);
	    NexusFree(incoming);

	}
	globus_list_free(xtp_incoming_list);

	for (current = xtp_outgoing_list;
	     !globus_list_empty(current);
	     current = globus_list_rest(current)) {

	    xtp_outgoing_t *outgoing = (xtp_outgoing_t *) current->datum;

	    if (outgoing->xi) {
		outgoing->xi->release();
		delete(outgoing->xi);
	    }
	    
	}
	globus_list_free(xtp_outgoing_list);


	globus_callback_unregister(globus_l_nexus_xtp_callback_handle);

    }
    xtp_exit();

}
/* xtp_shutdown() */

/*
 * xtp_increment_reference_count()
 *
 * Increase the reference count on the associated process_group.  This routine
 * is called when nexus_startpoint_copy() is called to create a duplicate
 * startpoint within the current context.
 */
static void
xtp_increment_reference_count(
    nexus_proto_t *			nproto)
{
    NexusAssert2(NEXUS_FALSE, 
		 ("nexus_startpoint_copy not supported by pr_xtp\n"));	
    /* Don't ever want a startpoint copy to happen, so make this blow up */

    nexus_printf( "xtp_increment_reference_count(): called\n");
}
/* xtp_increment_reference_count() */


/*
 * xtp_decrement_reference_count()
 *
 * Decrement the reference count for the outgoing_t associated
 * with this proto_t.  If the reference count goes to 0 then 
 * we release resource used by the associated xtp_process_group_t.
 *
 * This is called by nexus_startpoint_destroy()
 *
 * Return NEXUS_TRUE if this function frees the proto.
 */
static globus_bool_t
xtp_decrement_reference_count(
    nexus_proto_t *			nproto)
{
    xtp_outgoing_t *		outgoing;;

    outgoing = (xtp_outgoing_t *) nproto;

    nexus_printf( "xtp_decrement_reference_count(): called\n");

    xtp_enter();
    {
	short16 options_sendcntl;

	/*
	 * If by chance we were trying to write something to this
	 * connection, stop.  If we weren't, no harm done.
	 */
	xtp_unregister_for_write(outgoing);

	/*
	 * Close the connection as gracefully as possible.
	 */
	options_sendcntl = WCLOSE | RCLOSE | MULTI | DREQ | END;
	outgoing->xi->sendcntl(0, &options_sendcntl);

	/*
	 * Do a release on the sending XTP connection and blow away the 
	 * interface object.  Note that there's only going to be a refcount
	 * of one (at most) so anytime this function gets called, we're
	 * closing the connection.
	 */
	outgoing->xi->release();
	delete outgoing->xi;

	NexusFree(outgoing);
    }
    xtp_exit();

    return(NEXUS_TRUE);
} /* xtp_decrement_reference_count() */


/*
 * xtp_get_my_mi_proto()
 *
 * This is called by nexus_endpoint_init()
 *
 * Return in 'array' and 'size' a byte array containing
 * enough information to enable another process to connect
 * to this one.  This byte array will be assembled with those
 * of the other protocol modules and placed in a nexus_mi_proto_t.
 */
static int
xtp_get_my_mi_proto(
    globus_byte_t **			array,
    int *				size,
    void *				proto_info,
    nexus_endpoint_t *			endpoint)
{
    int					i;
    nexus_proto_info_xtp_t *		xtp_info;
    long				pid;
    xtp_config 				xcf;
    int					rc;
    int					length;
    address_segment 			addr;

    xtp_outgoing_t *		outgoing;
    xtp_outgoing_t **		outgoing_ptr;

    xtp_incoming_t *		incoming;
    xtpif *			xi;

    nexus_debug_printf(
	2,
	("xtp_get_my_mi_proto(): entry\n"));

    xtp_info = (nexus_proto_info_xtp_t *) proto_info;

    xtp_enter();
    {

	/*
	 * Need to handle this differently if we're sender or receiver.
	 */
	if (xtp_info->sender) {

	    nexus_debug_printf(
		2,
		("xtp_get_my_mi_proto(): sender\n"));

	    /*
	     * Create a structure to hold all the parameters.
	     * An xtp_outgoing_t will contain parms from
	     * the nexus_proto_info_xtp_t, plus pointer to 
	     * xtpif plus...???...
	     */
	    NexusMalloc(xtp_get_my_mi_proto(),
			outgoing,
			xtp_outgoing_t *,
			sizeof(xtp_outgoing_t));
	    globus_list_insert(&xtp_outgoing_list, outgoing);
	    
	    /* Fill in the structure from xtp_info */
	    outgoing->version = GLOBUS_L_NEXUS_XTP_PROTOCOL_VERSION;
	    outgoing->reserved_header_size = XTP_HDR_SIZE;
	    outgoing->diameter = xtp_info->diameter;
	    outgoing->addr = xtp_info->addr;
	    outgoing->port = xtp_info->port;
	    outgoing->config_options = RCLOSE | MULTI;
	    outgoing->encap_options = RCLOSE | MULTI;
	    outgoing->payload_options = RCLOSE | MULTI | EOM; 

	    /*
	     * SREQ frequency can be specified by user, defaults to 2.
	     * This simulates the edge bit, but these SREQs are explicitly
	     * requested, so they are protected.
	     */
	    if (xtp_info->sreq_freq > 0) {
		outgoing->edge_freq = xtp_info->sreq_freq;
	    }
	    else {
		outgoing->edge_freq = 2;
	    }
	    outgoing->edge_counter = outgoing->edge_freq - 1;

	    outgoing->type = NEXUS_PROTO_TYPE_XTP;
	    outgoing->funcs = &xtp_proto_funcs;
	    outgoing->direct_custom_min_size = (
		NEXUS_XTP_DIRECT_CUSTOM_MIN_SIZE);
	    outgoing->direct_custom_max_size = NEXUS_DC_MAX_U_LONG;
	    outgoing->direct_pointer_min_size = NEXUS_DC_MAX_U_LONG;
	    outgoing->direct_pointer_max_size = NEXUS_DC_MAX_U_LONG;
	    outgoing->can_use_iovec = NEXUS_FALSE;

	    outgoing->write_q_head = NULL;
	    outgoing->write_q_tail = NULL;
	    outgoing->write_in_progress = NEXUS_FALSE;
	    outgoing->error = NEXUS_FALSE;
	    outgoing->connected = NEXUS_FALSE;
	    outgoing->offset = 0;
	    outgoing->sent_sreq = NEXUS_FALSE;

	    /*
	     * Return the machine independent protocol array
	     */
	    NexusMalloc(xtp_get_my_mi_proto(),
			outgoing_ptr,
			xtp_outgoing_t **,
			sizeof(xtp_outgoing_t *));

	    *outgoing_ptr = outgoing;
	    *array = (globus_byte_t *) outgoing_ptr;
	    *size = sizeof(xtp_outgoing_t *);

	    nexus_debug_printf(
		3,
		("xtp_get_my_mi_proto(): outgoing_ptr=%x and *outgoing_ptr=%x\n",
		 outgoing_ptr, *outgoing_ptr));

	}
	else {

	    short16 options;	/* XTP options */

	    nexus_debug_printf(
		2,
		("xtp_get_my_mi_proto(): non-sender\n"));

	    /*
	     * We're the receiver, so create an incoming, and add it
	     * to the hash table.  Note that unlike the sender case,
	     * we actually create our end of the XTP association here.
	     */
	    NexusMalloc(xtp_get_my_mi_proto(),
			incoming,
			xtp_incoming_t *,
			sizeof(xtp_incoming_t));

	    incoming->ep = endpoint;
	    globus_hashtable_insert(&xtp_incoming_hash_table, endpoint, incoming);
	    globus_list_insert(&xtp_incoming_list, incoming);

	    xi = new xtpif();
	    incoming->xi = xi;
	    xi->user_ptr = incoming;

	    /*
	     * Configure the receiver side of the association.  It's much
	     * like the sender, except we don't set the RCLOSE option and
	     * we don't need to set the multicast diameter.
	     */
	    xcf.options = (MULTI);
	    xcf.extra_modes = (GRACEFULCLOSE | SNDCNTLONEDGE);
	    
	    nexus_debug_printf(
		3,
		("xtp_get_my_mi_proto(): ready to xtpif::config\n"));

	    if ((xi->config(XCFMODES, &xcf)) != EXOK) {
		xi->perror(rc, "config: ");
		delete xi;
		goto mutex_abort;
	    }

	    /*
	     * Having configured the connection, register it with the 
	     * daemon.
	     */
	    nexus_debug_printf(
		3,
		("xtp_get_my_mi_proto(): ready to xtpif::reg\n"));

	    if ((rc = xi->reg()) != EXOK) {
		xi->perror(rc, "reg: ");
		delete xi;
		goto mutex_abort;
	    }

	    /* 
	     * Bind the local endpoint.
	     */
	    memset((char *) &addr, (char) 0, sizeof(address_segment));
	    addr.IPaddr.srchost = (word32) INADDR_ANY;
	    addr.IPaddr.srcport = xtp_info->port;
	    addr.IPaddr.dsthost = (word32) xtp_info->addr.s_addr;
	    addr.IPaddr.dstport = 0;
	    addr.alength = (short16)(4 + sizeof(IPaddress));
	    addr.adomain = (byte8) 0;
	    addr.aformat = (byte8) 0x01; /* XTP/IP */

	    nexus_debug_printf(
		3,
		("xtp_get_my_mi_proto(): ready to xtpif::bind\n"));

	    if ((rc = xi->bind(&addr)) != EXOK) {
		xi->perror(rc, "bind: ");
		delete xi;
		goto mutex_abort;
	    }

	    /*
	     * Do a non-blocking listen to get a FIRST packet.
	     * This xtpif object will show ready-to-read when something
	     * comes in.
	     */
	    nexus_debug_printf(
		3,
		("xtp_get_my_mi_proto(): ready to xtpif::listen\n"));

	    options = WCLOSE | MULTI | SREQ;

	    if ((rc = xi->listen(0, &options)) != EXOK) {
		xi->perror(rc, "listen: ");
		delete xi;
		goto mutex_abort;
	    };

	    /*
	     * Need to register xi as a xtpif that we want to
	     * check to be ready for reading, by some future xtpif::select().
	     */
	    xtp_register_for_read(incoming);

	    incoming->state = INCOMING_STATE_VERSION;

	    /*
	     * Not sender, so don't return any mi_proto information.
	     */
	    *array = NULL;
	    *size = 0;

	}
      mutex_abort: ;
    }
    xtp_exit();

    nexus_debug_printf(
	3,
	("xtp_get_my_mi_proto(): return with array=%x and size=%d\n",
	 array, *size));

    /* XXX how do we indicate an error condition?  How to clean up? XXX */
    return(0);
}


/*
 * xtp_destroy_my_mi_proto()
 *
 * called by nexus_endpoint_destroy()
 */
static int
xtp_destroy_my_mi_proto(
    nexus_endpoint_t *			endpoint,
    globus_byte_t *			proto_array,
    int					size)
{
    xtp_incoming_t			*incoming;

    nexus_debug_printf(
	2,
	("xtp_destroy_my_mi_proto(): entry\n"));

    xtp_enter();
    {

	/*
	 * Find the associated xtp_incoming_t, given the endpoint.
	 */
	incoming = (xtp_incoming_t *) 
	    globus_hashtable_lookup(&xtp_incoming_hash_table, endpoint);

	if (incoming != NULL) {

	    /*
	     * Get rid of the XTP connection
	     */
	    if (incoming->xi) {
		incoming->xi->release();
		delete incoming->xi;
	    }

	    /*
	     * Delete the incoming from the hash table, and then
	     * blow it away.  Note:  If we couldn't find the incoming,
	     * then this endpoint is on the sender, and we only cared
	     * about the startpoint.  Ergo, do nothing.
	     */
	    globus_hashtable_remove(&xtp_incoming_hash_table, endpoint);
	    xtp_unregister_for_read(incoming);
	    NexusFree(incoming);

	}

    }
    xtp_exit();

    return(0);
}
/* xtp_destroy_my_mi_proto() */


/*
 * xtp_construct_from_mi_proto()
 *
 * This is called by nexus_startpoint_bind() or nexus_get_startpoint()
 *
 * From the passed machine independent protocol list ('mi_proto'), plus
 * the xtp specific entry from that list ('proto_array' and 'size'),
 * see if I can use the information to create a nexus_proto_t object
 * that can be used to connect to the context:
 *	- If I cannot use this protocol to attach to the context, then
 *		return NEXUS_FALSE.  (This option is useful if two contexts
 *		both speak a particular protocol, but they cannot
 *		talk to each other via that protocol.  For example,
 *		on two MPP's, the contexts within a single MPP can
 *		talk to each other via the native messaging protocol,
 *		but cannot talk to the contexts on the other MPP
 *		using that native protocol.)
 *	- If this xtp protocol points to myself, and thus the local
 *		protocol module should be used, then set
 *		*proto=NULL, and return NEXUS_TRUE.
 *	- Otherwise, construct a xtp protocol object for this mi_proto
 *		and put it in *proto.  Then return NEXUS_TRUE.
 *
 * The 'proto_array' should contain:
 *	format ID (1 byte)
 *	process group id (4 bytes)
 */
static globus_bool_t
xtp_construct_from_mi_proto(
    nexus_proto_t **			proto,
    nexus_mi_proto_t *			mi_proto,
    globus_byte_t *			proto_array,
    int					size)
{
    globus_bool_t			return_value;
    int					format;
    long				pid;
    xtp_outgoing_t *			outgoing;
    xtp_outgoing_t **			outgoing_ptr;
    xtp_config 				xcf;
    int					rc;
    int					length;
    address_segment 			addr;

    /*
     * Get the pointer to the outgoing.  This is kind of a hack, we need
     * to get the pointer to something (ergo, we need proto_array longword-
     * aligned).  XXX Shouldn't proto_array be properly aligned anyways?
     */
    outgoing = NULL;		/* compiler, shut up! */
//    bcopy((void *) proto_array, (void *) &outgoing, sizeof(xtp_outgoing_t *));
    memcpy((void *) &outgoing, (void *) proto_array, sizeof(xtp_outgoing_t *));

    nexus_debug_printf(
	3,
	("xtp_construct_from_mi_proto(): entry with proto_array=%x, *proto_array=%x, and size=%d\n", proto_array, outgoing, size));

    return_value = NEXUS_TRUE;

    xtp_enter();
    {
	/* 
         * Create and configure the sender's side of a connection
	 */
	outgoing->xi = new xtpif;

	outgoing->xi->user_ptr = outgoing;

	nexus_debug_printf(
	    3,
	    ("xtp_construct_from_mi_proto(): ready to xtpif::config\n"));

	/*
	 * Set configuration of this connection (er, association in
	 * XTP parlance).  Some of this information is going to come from
	 * the outgoing, originally from the user's passed-in parameters.
	 */
	xcf.options = outgoing->config_options;
	xcf.mcast_diameter = outgoing->diameter;
/*	xcf.edge_freq = outgoing->edge_freq; */
	xcf.extra_modes = (GRACEFULCLOSE | SOLCNTLONEDGE );

	if ((rc = outgoing->xi->config(XCFMODES | XCFMCDIAM /* | XCFEDGEFREQ */, &xcf)) != EXOK) {
	    outgoing->xi->perror(rc, "config: ");
	    delete outgoing->xi;
	    outgoing->xi = NULL;
	    return_value = NEXUS_FALSE;
	    goto mutex_abort;
	};

	nexus_debug_printf(
	    3,
	    ("xtp_construct_from_mi_proto(): ready to xtpif::reg\n"));

	/*
	 * Register connection.
	 */
	if ((rc = outgoing->xi->reg()) != EXOK) {
	    outgoing->xi->perror(rc, "reg: ");
	    delete outgoing->xi;
	    outgoing->xi = NULL;
	    return_value = NEXUS_FALSE;
	    goto mutex_abort;
	};

	/*
	 * Bind local endpoint.
	 */
	memset((char *) &addr, (char) 0, sizeof(address_segment));
	addr.IPaddr.srchost = (word32) INADDR_ANY;
	addr.IPaddr.srcport = 0;
	addr.IPaddr.dsthost = (word32) outgoing->addr.s_addr;
	addr.IPaddr.dstport = outgoing->port;
	addr.alength = (short16)(4 + sizeof(IPaddress));
	addr.adomain = (byte8) 0;
	addr.aformat = (byte8) 0x01; /* XTP/IP */

	if ((rc = outgoing->xi->bind(&addr)) != EXOK) {
	    outgoing->xi->perror(rc, "bind: ");
	    delete outgoing->xi;
	    outgoing->xi = NULL;
	    return_value = NEXUS_FALSE;
	    goto mutex_abort;
	};

	outgoing->connected = NEXUS_TRUE;

mutex_abort: ;
    }
    xtp_exit();

    /*
     * Return process group
     */
    *proto = (nexus_proto_t *) outgoing;

    nexus_debug_printf(
	2,
	("xtp_construct_from_mi_proto(): return(%d)\n", return_value));

    return (return_value);
} /* xtp_construct_from_mi_proto() */


/*
 * xtp_test_proto()
 * 
 * See if there was an error on the start point.  We can tell this because
 * the outgoing_t has its error flag set to NEXUS_TRUE.
 */
static int
xtp_test_proto(
    nexus_proto_t *			proto)
{
    xtp_outgoing_t *		outgoing;

    outgoing = (xtp_outgoing_t *) proto;

    if (outgoing->connected == NEXUS_FALSE)
    {
	return(NEXUS_ERROR_PROCESS_DIED);
    }
    else {
	return(NEXUS_SUCCESS);
    }
}
/* xtp_test_proto() */


/*
 * xtp_direct_info_size()
 */
static int
xtp_direct_info_size(void)
{
    /* TODO: This needs to be filled in */
    return(0);
}
/* xtp_direct_info_size() */


/*
 * xtp_send_rsr()
 */
static int
xtp_send_rsr(
    struct globus_nexus_buffer_s *	buffer)
{
    int					rc;
    xtp_outgoing_t *			outgoing;
    size_t				total_size;

    rc = NEXUS_SUCCESS;

    nexus_debug_printf(
	2,
	("xtp_send_rsr(): invoked with buffer: %x\n",buffer));

    /*
     * If there are any direct components to the message, then fail
     */
    NexusAssert(buffer->n_direct == 0);

    /*
     * Get the outgoing process group information for this message
     */
    outgoing = (xtp_outgoing_t *) buffer->proto;

    xtp_enter();
    {

	/*
	 * Make sure we are connected still.
	 */
	if (outgoing->connected == NEXUS_FALSE)
	{
	    rc = NEXUS_ERROR_PROCESS_DIED;
	    buffer->using_barrier = NEXUS_FALSE;
	    goto mutex_abort;
	}

//	NexusAssert2((process_group->type == NEXUS_PROTO_TYPE_XTP),
//		     ("xtp_send_rsr(): Internal error: "
//		      "proto_type is not NEXUS_PROTO_TYPE_XTP\n"));
    
	/*
	 * Enqueue this message on the communication channel's outgoing queue
	 */
	Enqueue(outgoing->write_q_head,
		outgoing->write_q_tail,
		buffer);
    
	/*
	 * Register that we want to write on this xtpif.  The write will
	 * eventually happen as a consequence of our poll routing getting
	 * called.
	 */
	xtp_register_for_write(outgoing);

      mutex_abort: ;
    }
    xtp_exit();

    return (rc);
} /* xtp_send_rsr() */


/*
 * xtp_send_rsr_outstanding()
 *
 * Return true if there are any sends outstanding for this proto,
 * otherwise false.
 */
static globus_bool_t
xtp_send_rsr_outstanding(globus_nexus_proto_t *nproto)
{
    globus_bool_t rc = GLOBUS_FALSE;
    xtp_outgoing_t *outgoing = (xtp_outgoing_t *) nproto;
    xtp_enter();
    if (outgoing->write_q_head)
    {
	rc = GLOBUS_TRUE;
    }
    xtp_exit();
    return(rc);
} /* xtp_send_rsr_outstanding() */


/*
 * xtp_poll
 */
static globus_bool_t xtp_poll(
     globus_time_t                        time_can_block,
     void *                               user_args)
{

    int ready, readyRead, readyWrite;
    xtpif **xiRead, **xiWrite;
    int i;
    globus_bool_t message_handled = NEXUS_FALSE;

    nexus_debug_printf(
	4,
	("xtp_poll(): entry\n"));

    xtp_enter();
    {

	/*
	 * Build up the arrays of things to select on, which includes
	 * all xtpifs associated with incoming_t's and a subset of those
	 * associated with outgoing_t's.
	 */
	xtp_get_registered_read();
	xtp_get_registered_write();
	
	/*
	 * XXX How do we deal with this?  We need to allocate an array 
	 * of unknown size (unknown until this time, that is), which is a
	 * copy of each of the xtpif arrays for reading and writing.  But we
	 * want to avoid doing this on every poll.  Can we cache these
	 * allocations also?  The way we're doing this now is evil.
	 */
	NexusMalloc(xtp_poll(),
		    xiWrite,
		    xtpif **,
		    sizeof(xtpif *) * xtp_outgoing_select_cache_count);
//	bcopy(xtp_outgoing_select_cache_array,
//	      xiWrite,
//	      sizeof(xtpif *) * xtp_outgoing_select_cache_count);
	memcpy(xiWrite,
	       xtp_outgoing_select_cache_array,
	       sizeof(xtpif *) * xtp_outgoing_select_cache_count);

	NexusMalloc(xtp_poll(),
		    xiRead,
		    xtpif **,
		    sizeof(xtpif *) * xtp_incoming_select_cache_count);
//	bcopy(xtp_incoming_select_cache_array,
//	      xiRead,
//	      sizeof(xtpif *) * xtp_incoming_select_cache_count);
	memcpy(xiRead,
	       xtp_incoming_select_cache_array,
	       sizeof(xtpif *) * xtp_incoming_select_cache_count);

	/*
	 * Execute a non-blocking xtpif::select(), with the sets of xtpifs
	 * constructed above.  Note that we only do the select if we
	 * actually have file descriptors we're interestd in, otherwise
	 * we skip it.
	 */
	readyRead = xtp_incoming_select_cache_count;
	readyWrite = xtp_outgoing_select_cache_count;

	ready = 0;
	if ((readyRead > 0) || (readyWrite > 0)) {

	    nexus_debug_printf(
		4,
		("xtp_poll(): ready for xtpif::select(readyRead=%d (%x), xiRead=%x, readyWrite=%d (%x), xiWrite=%x, blockflag=%d)\n", readyRead, &readyRead, xiRead, readyWrite, &readyWrite, xiWrite, 0));
	    
	    ready = xtpif::select(readyRead, xiRead,
				  readyWrite, xiWrite,
				  0);

	    nexus_debug_printf(
		4,
		("xtp_poll(): xtpif::select() returns %d, readyRead = %d, readyWrite = %d\n", ready, readyRead, readyWrite));
	    
	}
	else {
	    nexus_debug_printf(
		4,
		("xtp_poll(): skipping xtpif::select()\n"));
	}

	if (ready != 0) {

	    /*
	     * Handle incoming data, if any.
	     */
	    for (i = 0; i < readyRead; i++) {

		/* Assumption we got an array of xtpifs back ready to read. */
		xtp_process_incoming(xiRead[i]);
		message_handled = NEXUS_TRUE;
		/* XXX is setting message_handled here right? XXX */
		
	    }

	    /*
	     * Handle outgoing data, if any.
	     */
	    for (i = 0; i < readyWrite; i++) {

		/* Assumption we got an array of xtpifs back ready to write. */
		xtp_process_outgoing(xiWrite[i]);
		message_handled = NEXUS_TRUE;
		/* XXX is setting message_handled here right? XXX */

	    }

	}

	NexusFree(xiRead);
	NexusFree(xiWrite);
    }
    xtp_exit();

    return(message_handled);
}

void xtp_process_incoming(xtpif *xi)
{
    
    /* simplified state machine model after tcp_incoming_t */

    xtp_incoming_t *incoming;
    int length;			/* length of data to read */
    int rc;			/* return code */
    globus_bool_t done;		/* done with reading data? */
    globus_bool_t message_enqueued; /* at least one message queued up */
    struct globus_nexus_buffer_s 		*buffer;

    short16 options;		/* XTP options word */

    done = NEXUS_FALSE;
    message_enqueued = NEXUS_FALSE;
    incoming = (xtp_incoming_t *) xi->user_ptr;

    while (!done) {

	options = 0;

	switch (incoming->state) {

	    /*
	     * Verify that the internal structure of the message is one of
	     * the formats that we understand
	     */
	case (INCOMING_STATE_VERSION):
	{

	    nexus_debug_printf(
		2,
		("xtp_process_incoming(): INCOMING_STATE_VERSION\n"));

	    length = sizeof(globus_byte_t);
	    rc = xi->receive((void *) &(incoming->version), length, (word32) 0, &options);

	    if (rc < EXOK) {
		xi->perror(rc, "receive FORMAT: ");
		if ((rc == EXKEY) || (rc == EXINVA)) {
		    xtp_unregister_for_read(incoming);
		    done = NEXUS_TRUE;
		    incoming->state = INCOMING_STATE_CLOSED;
		}
	    }

	    nexus_debug_printf(
		1,
		("xtp_process_incoming(): options = 0x%x\n", options));
	    if (options & END) {
		xtp_unregister_for_read(incoming);
		done = NEXUS_TRUE;
		incoming->state = INCOMING_STATE_CLOSED;
		    
		nexus_debug_printf(
		    1,
		    ("xtp_process_incoming(): got END\n"));

	    }

	    nexus_debug_printf(
		3,
		("xtp_process_incoming(): read %d bytes\n", 
		 rc));

	    if (rc == sizeof(globus_byte_t)) {

		nexus_debug_printf(
		    3,
		    ("xtp_process_incoming(): message format %d\n", 
		     incoming->format));

		incoming->remaining = sizeof(globus_byte_t);
		incoming->current = (globus_byte_t *) incoming->encapfield;
		incoming->state = INCOMING_STATE_MSG_FORMAT;
	    }
	    else {
		/* 0 bytes read */
		done = NEXUS_TRUE;
	    }
	}
	break;

	    /* 
	     * Read format byte.  This particular case is slightly
	     * simplified compared to the others, since there's only one
	     * byte to read and we've either got it or we haven't (i.e.
	     * no partial reads).
	     */
	case (INCOMING_STATE_MSG_FORMAT):
	{

	    nexus_debug_printf(
		2,
		("xtp_process_incoming(): INCOMING_STATE_MSG_FORMAT\n"));

	    length = sizeof(globus_byte_t);
	    rc = xi->receive((void *) &(incoming->format), length, (word32) 0, &options);

	    if (rc < EXOK) {
		xi->perror(rc, "receive FORMAT: ");
		if ((rc == EXKEY) || (rc == EXINVA)) {
		    xtp_unregister_for_read(incoming);
		    done = NEXUS_TRUE;
		    incoming->state = INCOMING_STATE_CLOSED;
		}
	    }

	    nexus_debug_printf(
		1,
		("xtp_process_incoming(): options = 0x%x\n", options));
	    if (options & END) {
		xtp_unregister_for_read(incoming);
		done = NEXUS_TRUE;
		incoming->state = INCOMING_STATE_CLOSED;
		    
		nexus_debug_printf(
		    1,
		    ("xtp_process_incoming(): got END\n"));

	    }

	    nexus_debug_printf(
		3,
		("xtp_process_incoming(): read %d bytes\n", 
		 rc));

	    if (rc == sizeof(globus_byte_t)) {

		nexus_debug_printf(
		    3,
		    ("xtp_process_incoming(): message format %d\n", 
		     incoming->format));

		incoming->remaining = 
		    nexus_dc_sizeof_remote_u_long(1, incoming->format);
		incoming->current = (globus_byte_t *) incoming->encapfield;
		incoming->state = INCOMING_STATE_MSG_SIZE;
	    }
	    else {
		/* 0 bytes read */
		done = NEXUS_TRUE;
	    }
	}
	break;
    
	/*
	 * Read length word.  This word could be any one of a number of lengths,
	 * depending on the architecture of the sender.  Also, it's possible for us
	 * to get partial reads on this (though this isn't expected to
	 * happen in regular operation).
	 */
	case (INCOMING_STATE_MSG_SIZE):
	{

	    nexus_debug_printf(
		2,
		("xtp_process_incoming(): INCOMING_STATE_MSG_SIZE\n"));

	    length = incoming->remaining;
	    rc = xi->receive((void *) incoming->current, length, (word32) 0, &options);

	    if (rc < EXOK) {

		xi->perror(rc, "receive LENGTH: ");
		if ((rc == EXKEY) || (rc == EXINVA)) {
		    xtp_unregister_for_read(incoming);
		    done = NEXUS_TRUE;
		    incoming->state = INCOMING_STATE_CLOSED;
		}
	    }

	    nexus_debug_printf(
		1,
		("xtp_process_incoming(): options = 0x%x\n", options));
	    if (options & END) {
		xtp_unregister_for_read(incoming);
		done = NEXUS_TRUE;
		incoming->state = INCOMING_STATE_CLOSED;

		nexus_debug_printf(
		    1,
		    ("xtp_process_incoming(): got END\n"));

	    }
	    nexus_debug_printf(
		3,
		("xtp_process_incoming(): read %d bytes\n", 
		 rc));

	    /*
	     * Did we get all that was left?
	     */
	    if (rc == incoming->remaining) {

		globus_byte_t *encap;
		encap = (globus_byte_t *) incoming->encapfield;
				/* need to make a copy of encapfield
				 * pointer because nexus_dc_get_u_long
				 * will mutate it. */

		/*
		 * Got the entire length word, so do format conversion.
		 */
		nexus_dc_get_u_long(&(encap),
				    &(incoming->msglen),
				    1,
				    incoming->format);

		nexus_debug_printf(
		    3,
		    ("xtp_process_incoming(): message size %d\n", 
		     incoming->msglen));

		/*
		 * XXX Sanity-check on incoming->msglen XXX
		 */
		if (incoming->msglen < 1) {

		    /* XXX should we emit some warning here? XXX */
		    xtp_unregister_for_read(incoming);
		    done = NEXUS_TRUE;
		    incoming->state = INCOMING_STATE_CLOSED;
		}
		else {

		    /* 
		     * Get space for the message and do some
		     * bookkeeping.  Note that if we didn't need to
		     * pre-allocate the buffer, we could read the
		     * entire message in one gulp, eliminating the
		     * need for the state machine.  
		     */
		    NexusMalloc(xtp_process_incoming,
				incoming->storage,
				globus_byte_t *,
				sizeof(globus_byte_t) * incoming->msglen);
		    incoming->current = incoming->storage;
		    incoming->remaining = incoming->msglen;
		    incoming->state = INCOMING_STATE_BODY;

		    // XXX Copy incoming->version, incoming->format,
		    // and incoming->msglen to newly-allocated
		    // buffer.  Frob the incoming->current and
		    // incoming->remaining vars accordingly.

		    *(incoming->current++) = incoming->version;
		    *(incoming->current++) = incoming->format;
//		    bcopy((void *) incoming->encapfield,
//			  (void *) incoming->current,
//			  nexus_dc_sizeof_remote_u_long(1, incoming->format));
		    memcpy((void *) incoming->current,
			   (void *) incoming->encapfield,
			   nexus_dc_sizeof_remote_u_long(1, incoming->format));
		    incoming->current +=
			nexus_dc_sizeof_remote_u_long(1, incoming->format);
		    incoming->remaining -= 
			2 + 
			nexus_dc_sizeof_remote_u_long(1, incoming->format);

		}

	    }
	    else {

		/*
		 * Didn't get all of the length, so increment pointers
		 * to get the rest later.
		 */
		incoming->remaining -= rc;
		incoming->current += rc;
		done = NEXUS_TRUE;
	    }

	}
	break;

	/*
	 * Read message body.  At last, we get to do the *real* work.
	 * After we get all of the message body (noting that it might take
	 * multiple reads to get everything into the buffer), queue the
	 * message for later dispatching.
	 */
	case (INCOMING_STATE_BODY):
	{

	    nexus_debug_printf(
		2,
		("xtp_process_incoming(): INCOMING_STATE_MSG_BODY\n"));

	    length = incoming->remaining;
	    rc = xi->receive((void *) incoming->current, length, (word32) 0, &options);

	    if (rc < EXOK) {

		xi->perror(rc, "receive BODY: ");
		if ((rc == EXKEY) || (rc == EXINVA)) {
		    xtp_unregister_for_read(incoming);
		}
	    }

	    nexus_debug_printf(
		1,
		("xtp_process_incoming(): options = 0x%x\n", options));
	    if (options & END) {
		xtp_unregister_for_read(incoming);
		done = NEXUS_TRUE;
		incoming->state = INCOMING_STATE_CLOSED;

		nexus_debug_printf(
		    1,
		    ("xtp_process_incoming(): got END\n"));

	    }

	    if (rc == incoming->remaining) {
		incoming->state = INCOMING_STATE_VERSION;

		_nx_buffer_create_from_raw(
		    incoming->storage, /* where's payload */
		    incoming->msglen, /* how much payload */
		    0, /* offset to nexus header */
		    incoming->msglen,
		    incoming-> ep, /* endpoint */
		    &buffer); /* where to put buffer ptr */
		Enqueue(incoming->dispatch_q_head,
			incoming->dispatch_q_tail,
			buffer);
		message_enqueued = NEXUS_TRUE;

	    }
	    else {
		incoming->remaining -= rc;
		incoming->current += rc;
		done = NEXUS_TRUE;
	    }

	}
	break;

	default:
	{
	    nexus_debug_printf(
		0,
		("xtp_process_incoming(): bad state\n"));
	}
	break;

	}
    }
    
    nexus_debug_printf(
	3,
	("xtp_process_incoming(): Ready to dispatch\n"));

    /*
     * Dispatch accumulated messages, if we are not currently in the middle
     * of a dispatch.
     */
    if (message_enqueued && !incoming->dispatch_in_progress) {

	incoming->dispatch_in_progress = NEXUS_TRUE;
	while (QueueNotEmpty(incoming->dispatch_q_head))
	{
	    Dequeue(incoming->dispatch_q_head,
		    incoming->dispatch_q_tail,
		    buffer);
	    xtp_exit();
	    nexus_debug_printf(2, ("dispatching buffer\n"));
	    _nx_buffer_dispatch(buffer);
	    xtp_enter();
	}
	incoming->dispatch_in_progress = NEXUS_FALSE;

    }

}

void xtp_process_outgoing(xtpif *xi)
{

    xtp_outgoing_t *outgoing;
    globus_bool_t done;

    nexus_buffer_t buffer;
    globus_byte_t *payload;	/* pointer to payload of rsr */
    size_t length;		/* length of payload */

    static globus_byte_t *header; /* encapsulation header */
    globus_byte_t *headerptr;	/* pointer to somewhere in encap. header */
    static size_t headerlen;	/* length of encapsulation header */

    globus_byte_t format;	/* format byte */

    short16 options;		/* XTP options for send */
    int rc;			/* return code */

    nexus_debug_printf(
	2,
	("xtp_process_outgoing(): entry\n"));

    /* Initialization... */
    outgoing = (xtp_outgoing_t *) xi->user_ptr;
    done = NEXUS_FALSE;

    while (!done) {

	/*
	 * See if there are any messages queued up for us to send.
	 * If not, we're done for now.
	 */
	if (!QueueNotEmpty(outgoing->write_q_head)) {
	    
	    xtp_unregister_for_write(outgoing);
	    done = NEXUS_TRUE;
	    break;

	}

	/*
	     * We're working on the message body.  To reiterate:  We
	     * are making the assumption that pr_totem does, that there's
	     * only going to be one base segment (and no direct segments).
	     * Perhaps we ought to check for this.  XXX
	     */
	buffer = outgoing->write_q_head;
	payload = buffer->base_segments->current;
	length = buffer->base_segments->size_used;

	nexus_debug_printf(
	    3,
	    ("xtp_process_outgoing(): length %d offset %d\n", length, outgoing->offset));

	/* Account for any part of segment that's already been written. */
	payload += outgoing->offset;
	length -= outgoing->offset;
	    
	    /* Send data.  */
	options = outgoing->payload_options;
	if (++(outgoing->edge_counter) == outgoing->edge_freq) {
	    options |= SREQ;
	    outgoing->edge_counter = 0;
	}

	if ((rc = xi->send((void *) payload,
			   length,
			   0,
			   &options)) < EXOK) {

	    /* 
		 * See if we ran out of buffer space.  If so, then send 
		 * SREQ to try to get the receiver(s) to send us a CNTL
		 * packet, so we can free up some buffer space. 
		 */
	    if (rc == EXSBUFF) {
		if (!(outgoing->sent_sreq)) {
		    short16 options_sendcntl = RCLOSE | MULTI | SREQ;
		    xi->sendcntl(0, &options_sendcntl);
		    outgoing->sent_sreq = NEXUS_TRUE;

		    nexus_debug_printf(
			3,
			("xtp_process_outgoing(): request sreq for payload space\n"));
		}
	    }
	    else {
		xi->perror(rc, "write: ");

		/*
		 * This is a fairly fatal protocol error.
		 */
		if (rc == EXSTATE) {
		    outgoing->connected = NEXUS_FALSE;
		    done = NEXUS_TRUE;
		}

	    }
	    rc = 0;
	}
	    
	nexus_debug_printf(
	    3,
	    ("xtp_process_outgoing(): sent %d bytes of %d requested\n", rc, length));

	/*
	     * If we managed to send all the (remaining?) data...
	     */
	if (rc == length + headerlen) { // XXX check this

	    outgoing->sent_sreq = NEXUS_FALSE;

	    /* Move on to the next buffer (if any) */
	    Dequeue(outgoing->write_q_head, 
		    outgoing->write_q_tail,
		    buffer);
	    nexus_buffer_destroy(&buffer);

	    outgoing->offset = 0;

	    if (!QueueNotEmpty(outgoing->write_q_head)) {

		/* Nothing left... */
		xtp_unregister_for_write(outgoing);
		done = NEXUS_TRUE;

		nexus_debug_printf(
		    3,
		    ("xtp_process_outgoing(): done with data\n"));

	    }
	    else {

		/* Still buffers left to send. */

	    }

	}
	else {

	    /* Update internal state so we know what to do next time. */

	    /* Note that if we had an error, we come down here, and
	       since we already twiddled rc to be 0, we'll just exit. 
	       Also note that since SandiaXTP doesn't do partial
	       sends, that's the *only* case in which we'll get here. */
	    outgoing->offset += rc;
	    done = NEXUS_TRUE;
		
	}
    }
}

/*
 * Procedures to manage our lists of incoming and outgoings.
 * We internally use the globus_list.c routines.  The SLLs are a little
 * difficult to deal when trying to xtpif::select(), because it
 * requires arrays of pointers to xtpifs.  We cache the results of 
 * building the xtpif arrays in hope of amortizing the time spent
 * building them.
 */
void xtp_register_for_write(xtp_outgoing_t *outgoing)
{
    if (globus_list_search(xtp_outgoing_select_list, (void *) (outgoing->xi)) == NULL) {
	globus_list_insert(&xtp_outgoing_select_list, (void *) (outgoing->xi));
	xtp_outgoing_select_cache_valid = NEXUS_FALSE;
    }

    nexus_debug_printf(3, ("xtp_register_for_write: registered %x\n", outgoing));

}

void xtp_register_for_read(xtp_incoming_t *incoming)
{
    if (globus_list_search(xtp_incoming_select_list, (void *) (incoming->xi)) == NULL) {
	globus_list_insert(&xtp_incoming_select_list, (void *) (incoming->xi));
	xtp_incoming_select_cache_valid = NEXUS_FALSE;
    }

    nexus_debug_printf(3, ("xtp_register_for_read: registered %x\n", incoming));

}

void xtp_unregister_for_write(xtp_outgoing_t *outgoing)
{

    globus_list_t *target;

    target = globus_list_search(xtp_outgoing_select_list,
				(void *) (outgoing->xi));

    if (target) {
	globus_list_remove(&xtp_outgoing_select_list, target);
	xtp_outgoing_select_cache_valid = NEXUS_FALSE;
    }
}

void xtp_unregister_for_read(xtp_incoming_t *incoming)
{
    globus_list_t *target;

    target = globus_list_search(xtp_incoming_select_list,
				(void *) (incoming->xi));

    if (target) {
	globus_list_remove(&xtp_incoming_select_list, target);
	xtp_incoming_select_cache_valid = NEXUS_FALSE;
    }
}

/*
 * globus_list_make_array would ideally go into the globus_list routines,
 * and it is named as such.  If we keep it in pr_xtp, we ought to give it
 * a new name so as not to get confused with the "real" globus_list
 * procedures and functions.
 */
void globus_list_make_array(globus_list_t *head, void ***array, int *count)
{
    *count = globus_list_size(head);

    if (*array != NULL) {
	NexusFree(*array);
    }

    if (*count > 0) {
	globus_list_t *current;
	int i;

	NexusMalloc(globus_list_make_array(),
		    *array,
		    void **,
		    sizeof(xtp_outgoing_t *));

	for (i = 0, current = head; 
	     current != NULL; 
	     i++, current = current->next) {
	     (*array)[i] = current->datum;
	     nexus_debug_printf(3, ("globus_list_make_array: (*array)[%d] = %x\n\n", i, (*array)[i]));
	}
    }
    else {
	*array = NULL;
	nexus_debug_printf(3, ("globus_list_make_array: NULL array\n"));
    }

}

/*
 * These routines deal with creating the various arrays of xtpifs we need.
 * If the cached version of the array of xtpifs is consistent,
 * then we'll use the cache.
 */
void xtp_get_registered_write() 
{
    if (xtp_outgoing_select_cache_valid == NEXUS_FALSE) {
	globus_list_make_array(xtp_outgoing_select_list,
			       (void ***) &xtp_outgoing_select_cache_array,
			       &xtp_outgoing_select_cache_count);
	
	xtp_outgoing_select_cache_valid = NEXUS_TRUE;
    }
}

void xtp_get_registered_read() 
{
    if (xtp_incoming_select_cache_valid == NEXUS_FALSE) {
	globus_list_make_array(xtp_incoming_select_list,
			       (void ***) &xtp_incoming_select_cache_array,
			       &xtp_incoming_select_cache_count);
	
	xtp_incoming_select_cache_valid = NEXUS_TRUE;
    }
}

#endif  /* HAVE_XTP_PROTO */
