/*
 * pr_iface.c
 *
 * Protocol interface routines
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/libraries/nexus/pr_iface.c,v 1.112 1999/09/10 19:52:06 bresnaha Exp $";

#include "internal.h"

#include <netinet/in.h>
#ifdef TARGET_ARCH_AXP
#define NX_NTOHL(m) m
#define NX_HTONL(m) m
#else
#define NX_NTOHL(m) ntohl(m)
#define NX_HTONL(m) htonl(m)
#endif

#ifdef BUILD_DEBUG
/* These are used by nexus_global_pointer_string() */
#define GP_N_STRINGS 10
static nexus_mutex_t gp_string_mutex;
#endif /* BUILD_DEBUG */

/*
 * mi_proto table stuff.
 *
 * The mi_proto table contains linked lists of nexus_mi_proto_t
 * structures.
 *
 * This table is used to avoid creating multiple nexus_mi_proto_t
 * objects that are the same.  Multiple global pointers which have
 * the same mi_proto information share a single nexus_mi_proto_t.
 */
#define MI_PROTO_TABLE_SIZE 1021
static nexus_mi_proto_t *	mi_proto_table[MI_PROTO_TABLE_SIZE];
static nexus_mutex_t		mi_proto_table_mutex;
static void			mi_proto_table_init(void);
static int			mi_proto_array_hash(nexus_byte_t *array,
						    int size);
/*
 * Other typedefs
 */
int				_nx_mi_proto_instantiate(
						nexus_mi_proto_t *mi_proto);

/*
 * Proto module list
 */
typedef struct _proto_module_list_t
{
    nexus_proto_funcs_t *		funcs;
    char *				name;
    nexus_proto_type_t			type;
    int					mi_proto_size;
    nexus_byte_t *			mi_proto_array;
    nexus_bool_t			add_to_my_mi_proto;
    struct _proto_module_list_t *	next;
} proto_module_list_t;
static proto_module_list_t *	proto_module_list_head;
static proto_module_list_t *	proto_module_list_tail;

#define AddProtoModuleToList(Caller, Name, Funcs, Type) \
{ \
    proto_module_list_t *__p; \
    NexusMalloc(Caller, __p, proto_module_list_t *, \
		sizeof(proto_module_list_t)); \
    __p->name = _nx_copy_string(Name); \
    __p->funcs = (Funcs); \
    __p->type = (Type); \
    __p->mi_proto_size = 0; \
    __p->mi_proto_array = (nexus_byte_t *) NULL; \
    __p->next = (proto_module_list_t *) NULL; \
    if (proto_module_list_head) \
    { \
	proto_module_list_tail->next = __p; \
	proto_module_list_tail = __p; \
    } \
    else \
    { \
	proto_module_list_head = proto_module_list_tail = __p; \
    } \
}


/*
 * saved proto arguments
 */
static char saved_no_proto_arg[256];

/*
 * _nx_proto_new_process_params()
 *
 * Call the new_process_params() function for each protocol
 *
 * Each of those functions may add stuff to 'buf', returning the number
 * of characters that they added.
 *
 * Return: The total number of characters added to 'buf'.
 */
int _nx_proto_new_process_params(char *buf, int size)
{
    char tmp_buf[256];
    int n_left = size;
    char *b = buf;
    int n_added;

    n_added = 0;
    tmp_buf[0] = '\0';

    nexus_stdio_lock();

    if (strlen(saved_no_proto_arg) > 0)
    {
	char tmp_buf2[256];

	sprintf(tmp_buf2, "-no_pr %s ", saved_no_proto_arg);
	strcat(tmp_buf, tmp_buf2);
    }

    n_added = strlen(tmp_buf);
    if (n_added > n_left)
    {
	nexus_stdio_unlock();
        globus_fatal("_nx_proto_new_process_params(): Internal error: "
		    "Not enough room in buffer for arguments\n");
    }
    strcpy(b, tmp_buf);
    b += n_added;
    n_left -= n_added;

    nexus_stdio_unlock();

    return (size - n_left);
} /* _nx_proto_new_process_params() */


/*
 * _nx_proto_init()
 *
 * Initialize the protocol modules.
 *
 * TODO: This needs to be made more dynamic
 */
void _nx_proto_init(nexus_module_list_t module_list[])
{
    int i;
    int rc;
    nexus_proto_funcs_t *proto_funcs;
    nexus_proto_type_t proto_type;
    proto_module_list_t *proto_module;
    nexus_byte_t *array;
    int size;
    int hostname_length;
    int arg_num;
    char *arg;
    char *next=GLOBUS_NULL, *no_proto=GLOBUS_NULL;
    nexus_bool_t add_proto;
    int pid;
    nexus_mi_proto_t *tmp_mi_proto;

    /*
     * Parse any arguments
     */
    if ((arg = globus_nexus_option_find("no_pr")) != GLOBUS_NULL)
    {
	strcpy(saved_no_proto_arg, arg);
    }
    
#ifdef BUILD_DEBUG
    nexus_mutex_init(&gp_string_mutex, (nexus_mutexattr_t *) NULL);
#endif

    /*
     * Scan the module_list looking for protocol modules.
     * For each of these, get the function table,
     * and add that module to the proto_module_list.
     * Put the local protocol module at the front of the list.
     */
    proto_module_list_head = proto_module_list_tail
	= (proto_module_list_t *) NULL;
    proto_funcs = _nx_pr_local_info();
    proto_type = (*proto_funcs->proto_type)();
    AddProtoModuleToList(_nx_proto_init(),
			 "local",
			 proto_funcs,
			 proto_type);

    for (i = 0; module_list[i].family_name != (char *) NULL; i++)
    {
	if ((strcmp(module_list[i].family_name, "protocols") == 0))
	{
	    add_proto = NEXUS_TRUE;
	    next = strdup(saved_no_proto_arg);
	    /* 
	     * This can be changed to while(next && add_proto) if the
	     * number of no_pr options increase.  With only tcp, atm,
	     * and inx, this is not an optimization.
	     *
	     */
	    while (next)
	    {
		char *prev = next;
		_nx_get_next_value(next, ':', &next, &no_proto);

		NexusFree(prev);
		if (strcmp(module_list[i].module_name, no_proto) == 0)
		{
		    add_proto = NEXUS_FALSE;
		}
		NexusFree(no_proto);
	    }

	    /*
	    rc = _nx_module_load(module_list[i].family_name,
				 module_list[i].module_name,
				 (void *) &proto_funcs);
	    */
	    rc = 0;
	    if (rc == 0 && add_proto)
	    {
	        proto_funcs
		    = (nexus_proto_funcs_t *) (*module_list[i].info_func)();
		proto_type = (*proto_funcs->proto_type)();

		AddProtoModuleToList(_nx_proto_init(),
				     module_list[i].module_name,
				     proto_funcs,
				     proto_type);
            }	    

	}
    }

    /*
     * Initialize each protocol module
     */
    for (proto_module = proto_module_list_head;
	 proto_module;
	 proto_module = proto_module->next)
    {
	if(
           proto_module->funcs->init)
	{
	    (*proto_module->funcs->init)(&proto_module->add_to_my_mi_proto);
	}
	else
	{
            proto_module->add_to_my_mi_proto = GLOBUS_FALSE;
	}
    }

    /*
     * Get the nexus_proto_t for the local protocol module
     * and fill it into _nx_my_mi_proto.
     */
    (proto_module_list_head->funcs->construct_from_mi_proto)(&_nx_local_proto,
						(nexus_mi_proto_t *) NULL,
						(nexus_byte_t *) NULL,
						0);

    /*
     * Construct my nexus_mi_proto_t.
     * First ask each protocol module what size its component will be,
     * then malloc an array big enough, and then ask
     * then ask each protocol module to fill their component into
     * the array.
     */
    mi_proto_table_init();
    hostname_length = strlen(_nx_my_hostname);
    size = SizeofMIProtoHeader(hostname_length);
    for (proto_module = proto_module_list_head;
	 proto_module;
	 proto_module = proto_module->next)
    {
	if (proto_module->add_to_my_mi_proto &&
	    proto_module->funcs->get_my_mi_proto)
	{
	    if (
		(proto_module->funcs->get_my_mi_proto)(
		    &(proto_module->mi_proto_array),
		    &(proto_module->mi_proto_size),
		    NULL,
		    NULL) == 0)
	    {
		size += SizeofMIProtoEntry(proto_module->mi_proto_size);
	    }
	    else
	    {
		proto_module->mi_proto_size = 0;
	    }
	}
    }
    _nx_my_mi_proto = _nx_mi_proto_create(size, (nexus_byte_t *) NULL, 
					  _nx_local_proto);
    array = _nx_my_mi_proto->array;
    i = 0;
    pid = globus_libc_getpid();
    PackMIProtoHeader(array, i, pid, _nx_my_hostname, hostname_length,
		      GLOBUS_I_NEXUS_MI_PROTO_HEADER_VERSION);
    for (proto_module = proto_module_list_head;
	 proto_module;
	 proto_module = proto_module->next)
    {
	if (proto_module->mi_proto_size > 0)
	{
	    PackMIProtoEntry(array, i,
			     proto_module->type,
			     proto_module->mi_proto_size,
			     proto_module->mi_proto_array);
	}
    }
    _nx_mi_proto_table_insert(_nx_my_mi_proto, &tmp_mi_proto);

} /* _nx_proto_init() */


/*
 * _nx_proto_shutdown()
 *
 * If 'shutdown_others' is NEXUS_TRUE,
 * then try to shutdown all other contexts.
 */
void _nx_proto_shutdown(void)
{
    proto_module_list_t *proto_module;
    
    for (proto_module = proto_module_list_head;
	 proto_module;
	 proto_module = proto_module->next)
    {
	if (proto_module->funcs->shutdown)
	{
	    (proto_module->funcs->shutdown)();
	}
    }
} /* _nx_proto_shutdown() */


/*
 * nexus_startpoint_test()
 *
 * Return 0 if the passed startpoint is (probably) still working,
 * else return non-0.
 *
 * If the protocol module for this sp does not implement the test_proto
 * function, then return 0.
 *
 * TODO: This really should be implemented as a round-trip
 */
int nexus_startpoint_test(nexus_startpoint_t *sp)
{
    int rc;
    if (   (sp)
	&& (sp->mi_proto)
	&& (sp->mi_proto->proto->funcs->test_proto))
    {
	rc = (sp->mi_proto->proto->funcs->test_proto)(sp->mi_proto->proto);
    }
    else
    {
	rc = 0;
    }
    return(rc);
} /* nexus_startpoint_test() */


/*
 * globus_nexus_startpoint_flush
 *
 * Wait until there are no more outstanding sends for this startpoint.
 */
void
globus_nexus_startpoint_flush(globus_nexus_startpoint_t *startpoint)
{
    globus_bool_t		done;
    globus_nexus_proto_t *	proto;
    globus_bool_t               first = GLOBUS_TRUE;

    proto = startpoint->mi_proto->proto;

    done = GLOBUS_FALSE;
    while (!done)
    {
	if (!(proto->funcs->send_rsr_outstanding)(proto))
	{
	    done = GLOBUS_TRUE;
	}
	else
	{
            if(first)
	    {
		first = GLOBUS_FALSE;
	        globus_thread_blocking_will_block();
            }
	    globus_poll_blocking();
	}
    }
} /* globus_nexus_startpoint_flush() */


#define STARTPOINT_ARGUMENT_STRING "nf_nx_sp"

/*
 * _nx_proto_get_creator_sp_params()
 *
 * Fill in 'buf', up to 'buf_size' bytes, with a command line
 * argument encoding * of the passed global pointer, 'gp', so
 * that a new process can connect to this one.
 */
void _nx_proto_get_creator_sp_params(char *buf,
				     int buf_size,
				     nexus_startpoint_t *sp)
{
    nexus_startpoint_t sp_copy;
    int sp_size;
    nexus_byte_t *sp_bytes;
    nexus_byte_t *b;
    char *c;
    int arg_len = strlen(STARTPOINT_ARGUMENT_STRING);

    /*
     * We need to make a copy of the startpoint, since
     * nexus_user_put_startpoint_transfer() destroys the startpoint.
     */
    nexus_startpoint_copy(&sp_copy, sp);
    
    sp_size = nexus_sizeof_startpoint(&sp_copy, 1);
    if (buf_size < ((2 * (sp_size + 1))
		    + arg_len
		    + 4)) /* "-" + 2 spaces + null termination */
    {
	globus_fatal("_nx_proto_get_creator_sp_params(): Internal error: Buffer not big enough for arguments\n");
    }

    NexusMalloc(_nx_proto_get_creator_sp_params(),
		sp_bytes,
		nexus_byte_t *,
		(sp_size + 1));
    b = sp_bytes;
    *b++ = GLOBUS_DC_FORMAT_LOCAL;
    nexus_user_put_startpoint_transfer(&b, &sp_copy, 1);

    c = buf;
    nexus_stdio_lock();
    sprintf(c, "-%s ", STARTPOINT_ARGUMENT_STRING);
    nexus_stdio_unlock();
    c += arg_len + 2;
    _nx_hex_encode_byte_array(sp_bytes,
			      sp_size + 1,
			      c);
    c += (2 * (sp_size + 1));
    *c++ = ' ';
    *c++ = '\0';
    NexusFree(sp_bytes);
} /* _nx_proto_get_creator_sp_params() */


/*
 * _nx_proto_construct_creator_sp()
 */
void _nx_proto_construct_creator_sp(nexus_startpoint_t *creator_sp)
{
    char *arg_string;
    nexus_byte_t *sp_bytes;
    nexus_byte_t *b;
    int len;
    int format;

    if ((arg_string = globus_nexus_option_find(
			       STARTPOINT_ARGUMENT_STRING)) != GLOBUS_NULL)
    {
	len = strlen(arg_string) / 2;
	NexusMalloc(_nx_proto_construct_creator_sp(),
		    sp_bytes,
		    nexus_byte_t *,
		    len + 1);
	_nx_hex_decode_byte_array(arg_string,
				  len,
				  sp_bytes);
	b = sp_bytes;
	format = (int) *b++;
	nexus_user_get_startpoint(&b, creator_sp, 1, format);
	NexusFree(sp_bytes);
    }
    else
    {
	globus_fatal("_nx_proto_construct_creator_sp(): Internal error: Expected a -nf_nx_sp argument.\n");
    }
} /* _nx_proto_construct_creator_sp() */


/*
 * _nx_mi_proto_create()
 *
 * Return an mi_proto for the passed 'size, 'array', and 'proto'.
 * If array==NULL, then do not try to copy it.  It will be filled in later.
 */
nexus_mi_proto_t *_nx_mi_proto_create(int size,
				      nexus_byte_t *array,
				      nexus_proto_t *proto)
{
    nexus_mi_proto_t *new_mi_proto;

    NexusMalloc(_nx_mi_proto_create(),
		new_mi_proto,
		nexus_mi_proto_t *,
		(sizeof(nexus_mi_proto_t) + size));
    new_mi_proto->proto = proto;
    new_mi_proto->next = (nexus_mi_proto_t *) NULL;
    new_mi_proto->size = size;
    new_mi_proto->array = (nexus_byte_t *) (((char *) new_mi_proto)
					    + sizeof(nexus_mi_proto_t));
    if (array)
    {
	memcpy(new_mi_proto->array, array, size);
    }
    
    return (new_mi_proto);
    
} /* _nx_mi_proto_create() */


/*
 * _nx_mi_proto_destroy()
 */
void _nx_mi_proto_destroy(nexus_mi_proto_t *mi_proto)
{
    if (mi_proto->proto)
    {
	if (mi_proto->proto->funcs->decrement_reference_count)
	{
	    if (
		(mi_proto->proto->funcs->decrement_reference_count)(
		    mi_proto->proto))
	    {
		/* The proto has been freed */
		mi_proto->proto = (nexus_proto_t *) NULL;
	    }
	}

	/* TODO: remove mi_proto from the proto table and free its memory */
    }
} /* _nx_mi_proto_destroy() */


/*
 * _nx_proto_check_type()
 */
nexus_bool_t _nx_proto_check_type(nexus_proto_type_t type)
{
    proto_module_list_t *proto_module;

    for (proto_module = proto_module_list_head;
	 proto_module;
	 proto_module = proto_module->next)
    {
	if (proto_module->type == type)
	{
	    return(NEXUS_TRUE);
	}
    }
    return(NEXUS_FALSE);
} /* _nx_proto_check_type() */


/*
 * _nx_mi_proto_instantiate()
 *
 * Go through the entries in the mi_proto's array, looking for ones
 * that we can use.  When one is found, instantiate it into
 * a nexus_proto_t object.
 */
int
_nx_mi_proto_instantiate(nexus_mi_proto_t *mi_proto)
{
    proto_module_list_t *proto_module;
    nexus_proto_t *proto;
    nexus_byte_t *a;
    nexus_byte_t *proto_array;
    int i;
    nexus_proto_type_t type;
    int size;
    nexus_bool_t done;
    int context_differentiator;
    char *hostname;
    int version;
    int return_code = GLOBUS_SUCCESS;

    a = mi_proto->array;
    i = 0;

    NexusAssert2(((i + SizeofMIProtoHeader(0)) <= mi_proto->size),
		 ("_nx_mi_proto_instantiate(): Invalid mi_proto array.  Not enough room for header.\n"));
    
    UnpackMIProtoHeader(a, i,
			context_differentiator,
			hostname,
			version);

    if (version != GLOBUS_I_NEXUS_MI_PROTO_HEADER_VERSION)
    {
	return_code = GLOBUS_NEXUS_ERROR_VERSION_MISMATCH;
	goto fnexit;
    }
    
    done = NEXUS_FALSE;
    while (!done && (i < mi_proto->size))
    {
	NexusAssert2(((i + SizeofMIProtoEntry(0)) <= mi_proto->size),
		     ("_nx_mi_proto_instantiate(): Invalid mi_proto array.  Not enough room for protocol entry header.\n"));

	UnpackMIProtoEntry(a, i, type, size, proto_array);

	NexusAssert2((i <= mi_proto->size),
		     ("_nx_mi_proto_instantiate(): Invalid mi_proto array.  Not enough room for protocol entry.\n"));

	/* See if this protocol is in my proto_module_list */
	for (proto_module = proto_module_list_head;
	     !done && proto_module;
	     proto_module = proto_module->next)
	{

	    if (proto_module->type == type)
	    {
		/* Found a type match.  So call the proto to instantiate it */
		if (proto_module->funcs->construct_from_mi_proto != GLOBUS_NULL &&
		    (proto_module->funcs->construct_from_mi_proto)(
			&proto,
			mi_proto,
			proto_array,
			size))
		{
		    /* Successful instantiation */
		    if (!proto)
		    {
			/* Use local protocol module */
			proto = _nx_local_proto;
		    }
		    mi_proto->proto = proto;
		    done = NEXUS_TRUE;
		}
	    }
	}
    }

    if (mi_proto->proto == (nexus_proto_t *) NULL)
    {
	return_code = GLOBUS_NEXUS_ERROR_NO_COMPATIBLE_PROTOCOLS;
    }
    
  fnexit:
    return(return_code);

} /* _nx_mi_proto_instantiate() */


/*
 * mi_proto_table_init()
 *
 * Initialize the mi_proto hash table.
 */
static void mi_proto_table_init(void)
{
    int i;
    for (i = 0; i < MI_PROTO_TABLE_SIZE; i++)
    {
	mi_proto_table[i] = (nexus_mi_proto_t *) NULL;
    }
    nexus_mutex_init(&mi_proto_table_mutex, (nexus_mutexattr_t *) NULL);
} /* mi_proto_table_init() */


/*
 * mi_proto_array_hash()
 *
 * Hash the passed mi_proto 'array'.
 */
static int mi_proto_array_hash(nexus_byte_t *array, int size)
{
    unsigned long hval = 0;
    int i;

    for (i = 0; i < size; i++)
    {
	hval <<= 1;
	hval += (unsigned long) array[i];
    }
    return ((int) (hval % MI_PROTO_TABLE_SIZE));
} /* mi_proto_array_hash() */


/*
 * _nx_mi_proto_table_insert()
 *
 * If 'new_mi_proto' already exists in the mi_proto_table,
 * then free it and return the existing copy.
 * Otherwise, add new_mi_proto to the table and return it.
 */
int
_nx_mi_proto_table_insert(nexus_mi_proto_t *new_mi_proto,
			  nexus_mi_proto_t **return_mi_proto)
{
    int bucket;
    nexus_mi_proto_t *mi_proto;
    int return_code = GLOBUS_SUCCESS;

    bucket = mi_proto_array_hash(new_mi_proto->array, new_mi_proto->size);

    /* Search the bucket for this mi_proto */
    for (mi_proto = mi_proto_table[bucket];
	 mi_proto != (nexus_mi_proto_t *) NULL;
	 mi_proto = mi_proto->next)
    {
	if (   (new_mi_proto->size == mi_proto->size)
	    && (memcmp(new_mi_proto->array, mi_proto->array,
		       new_mi_proto->size) == 0) )
	{
	    /*
	     * Found mi_proto with this array in the table.
	     * So use the existing one, and free the new one.
	     * Bump the reference count of the proto for the existing one.
	     */
	    NexusFree(new_mi_proto);
	    if (mi_proto->proto)
	    {
		if (mi_proto->proto->funcs->increment_reference_count)
		{
		    (mi_proto->proto->funcs->increment_reference_count)(
							mi_proto->proto);
		}
	    }
	    else
	    {
		/*
		 * This mi_proto is in the table, but its proto has
		 * be freed.  So re-instantiate it.
		 */
		return_code = _nx_mi_proto_instantiate(mi_proto);
	    }
	    *return_mi_proto = mi_proto;
	    return(return_code);
	}
    }
    
    /*
     * We did not find new_mi_proto in the table.
     * So add this new one to this hash bucket.
     */
    new_mi_proto->next = mi_proto_table[bucket];
    mi_proto_table[bucket] = new_mi_proto;

    if (new_mi_proto->proto == (nexus_proto_t *) NULL)
    {
	return_code = _nx_mi_proto_instantiate(new_mi_proto);
    }

    *return_mi_proto = new_mi_proto;
    return (return_code);
    
} /* _nx_mi_proto_table_insert() */


/*
 * _nx_mi_proto_create_for_proto_type()
 */
int
_nx_mi_proto_create_for_proto_type(nexus_proto_type_t proto_type,
				   void *proto_info,
				   nexus_endpoint_t *endpoint,
				   nexus_mi_proto_t **return_mi_proto)
{
    proto_module_list_t *	proto_module;
    int				mi_proto_size;
    nexus_byte_t *		mi_proto_array;
    nexus_mi_proto_t *		mi_proto;
    nexus_byte_t *		array;
    int				size;
    int				hostname_length;
    int				i;
    int				pid;
    int				return_code = GLOBUS_SUCCESS;

    mi_proto_size = 0;

    hostname_length = strlen(_nx_my_hostname);
    size = SizeofMIProtoHeader(hostname_length);
    for (proto_module = proto_module_list_head;
	 proto_module;
	 proto_module = proto_module->next)
    {
	if (proto_module->type == proto_type)
	{
	    if (proto_module->funcs->get_my_mi_proto)
	    {
		/*
		 * this code assumes only one protocol module will match
		 * the specified protocol type; if this is not the case
		 * then we must do something else to detect failure
		 */
		if (
		    (proto_module->funcs->get_my_mi_proto)(
			&mi_proto_array,
			&mi_proto_size,
			proto_info,
			endpoint) == 0)
		{
		    size += SizeofMIProtoEntry(mi_proto_size);
		    break;
		}
		else
		{
		    /*
		     * get_my_mi_proto FAILED; report error
		     */
		    *return_mi_proto = (nexus_mi_proto_t *) NULL;
		    return (GLOBUS_NEXUS_ERROR_NO_COMPATIBLE_PROTOCOLS);
		}
	    }
	}
    }
    mi_proto = _nx_mi_proto_create(size,
				   (nexus_byte_t *) NULL, 
				   (nexus_proto_t *) NULL);
    array = mi_proto->array;
    i = 0;
    pid = globus_libc_getpid();
    PackMIProtoHeader(array, i, pid, _nx_my_hostname, hostname_length,
		      GLOBUS_I_NEXUS_MI_PROTO_HEADER_VERSION);
    if (mi_proto_size > 0)
    {
	PackMIProtoEntry(array, i,
			 proto_type,
			 mi_proto_size,
			 mi_proto_array);

	free(mi_proto_array);
    }

    return_code = _nx_mi_proto_instantiate(mi_proto);

    *return_mi_proto = mi_proto;
    return(return_code);
    
} /* _nx_mi_proto_create_for_proto_type() */


/*
 * _nx_mi_proto_destroy_for_proto_type()
 */
int _nx_mi_proto_destroy_for_proto_type(nexus_endpoint_t * endpoint,
					nexus_mi_proto_t *mi_proto)
{
    proto_module_list_t *proto_module;
    nexus_byte_t *a;
    int i;
    nexus_byte_t *proto_array;
    nexus_proto_type_t type;
    int size;
    nexus_bool_t done;
    int context_differentiator;
    char *hostname;
    int rc;
    int version;

    rc = 0;

    a = mi_proto->array;
    i = 0;

    NexusAssert2(((i + SizeofMIProtoHeader(0)) <= mi_proto->size),
		 ("_nx_mi_proto_destroy_for_proto_type(): Invalid mi_proto array.  Not enough room for header.\n"));
    
    UnpackMIProtoHeader(a, i,
			context_differentiator,
			hostname,
			version);
    if (version != GLOBUS_I_NEXUS_MI_PROTO_HEADER_VERSION)
    {
	return(GLOBUS_NEXUS_ERROR_VERSION_MISMATCH);
    }

    done = NEXUS_FALSE;
    while (!done && (i < mi_proto->size))
    {
	NexusAssert2(((i + SizeofMIProtoEntry(0)) <= mi_proto->size),
		     ("_nx_mi_proto_destroy_for_proto_type(): Invalid mi_proto array.  Not enough room for protocol entry header.\n"));

	UnpackMIProtoEntry(a, i, type, size, proto_array);

	NexusAssert2((i <= mi_proto->size),
		     ("_nx_mi_proto_destroy_for_proto_type(): Invalid mi_proto array.  Not enough room for protocol entry.\n"));

	/* See if this protocol is in my proto_module_list */
	for (proto_module = proto_module_list_head;
	     proto_module;
	     proto_module = proto_module->next)
	{
	    if (proto_module->type == type)
	    {
		if (proto_module->funcs->destroy_my_mi_proto)
		{
		    if (
			(proto_module->funcs->destroy_my_mi_proto)(
			    endpoint,
			    proto_array,
			    size) != 0)
		    {
			rc = GLOBUS_NEXUS_ERROR_UNKNOWN;
		    }

		    break;
		}
	    }
	}
    }

    return(rc);
} /* _nx_mi_proto_destroy_for_proto_type() */


#ifdef BUILD_RESOURCE

/*
 * _nx_mi_proto_name_for_type(nexus_mi_proto_t *mi_proto)
 */
nexus_resource_name_t * 
_nx_mi_proto_name_for_proto_type( nexus_proto_type_t proto_type,
				  nexus_endpoint_t *endpoint )  {
    proto_module_list_t *	proto_module;
    nexus_resource_name_t * 	result = NULL;

    for (proto_module = proto_module_list_head;
	 proto_module;
	 proto_module = proto_module->next) {

	if (proto_module->type == proto_type) 	{
	    if (proto_module->funcs->get_resource_name_sp)   {
	      result = proto_module->funcs->get_resource_name_ep(endpoint);
	      break;
	    }

	}
    } 

    return (result);

}

#endif /* BUILD_RESOURCE */

nexus_proto_funcs_t *
globus_i_find_proto_module(nexus_proto_type_t type)
{
    proto_module_list_t * i;

    for(i = proto_module_list_head;
        i != NULL;
	i = i->next)
    {
        if(type == i->type)
        {
  	    return i->funcs;
	}
    }

    return NULL;
}

/*
 *  globus_nexus_startpoint_num_protos()
 *  ------------------------------------
 *  This function counts the number of protocols supported by a given
 *  startpoint and returns the result to the user.
 *
 *  added by bresnaha
 */
globus_bool_t
globus_nexus_startpoint_num_protos(globus_nexus_startpoint_t * sp)
{
   globus_byte_t *             mi_proto_array;
   int                         mi_proto_size;
   int                         i = 0;
   int                         version;
   nexus_proto_type_t          type;
   char *                      hostname;
   int                         count = 0;
   int                         size;
   nexus_byte_t *              subarray;
   int                         tmp_int;
   nexus_proto_funcs_t *       funcs;

   mi_proto_array = sp->mi_proto->array;
   mi_proto_size = sp->mi_proto->size;

   UnpackMIProtoHeader(mi_proto_array,
                       i,
                       tmp_int,
                       hostname,
                       version);

   while(i < mi_proto_size)
   {
       UnpackMIProtoEntry(mi_proto_array,
                          i,
                          type,
                          size, subarray);

       funcs = globus_i_find_proto_module(type);

       if(funcs != NULL)
       {
           count = count + funcs->proto_count();
       }
       else
       {
/* print error, break globus ? */
       }
   }

   return count;
}





/*
 *  globus_nexus_startpoint_proto_match()
 *  -------------------------------------
 *  This function checks to see if two startpoints have
 *  protocols at the given index that can communicate with
 *  each other.
 *
 *  RETURNS:
 *    GLOBUS_TRUE   when true
 *    GLOBUS_FALSE  when false.
 *
 *  added by bresnaha
 */
int
globus_nexus_startpoint_proto_match(globus_nexus_startpoint_t * sp0,
                                    int                         proto_index_0,
                                    globus_nexus_startpoint_t * sp1,
                                    int                         proto_index_1)
{
   globus_byte_t *             mi_proto_array0;
   globus_byte_t *             mi_proto_array1;
   int                         mi_proto_size0;
   int                         mi_proto_size1;
   int                         tmp_int;
   int                         i0 = 0;
   int                         i1 = 0;
   int                         ctr;
   int                         size0;
   int                         size1;
   nexus_byte_t *              subarray0;
   nexus_byte_t *              subarray1;
   char *                      hostname0;
   char *                      hostname1;
   int                         version0;
   int                         version1;
   nexus_proto_type_t          type0;
   nexus_proto_type_t          type1;
   nexus_proto_funcs_t *       proto_funcs;
   proto_module_list_t *       i;
   int                         found;
   int                         proto_offset0 = 0;   
   int                         proto_offset1 = 0;


   /*  get sp0 info  */
   mi_proto_array0 = sp0->mi_proto->array;
   mi_proto_size0 = sp0->mi_proto->size;

   UnpackMIProtoHeader(mi_proto_array0,
                       i0,
                       tmp_int,
                       hostname0,
                       version0);

   /*  get sp1 info  */
   mi_proto_array1 = sp1->mi_proto->array;
   mi_proto_size1 = sp1->mi_proto->size;

   UnpackMIProtoHeader(mi_proto_array1,
                       i1,
                       tmp_int,
                       hostname1,
                       version1);

   /* test for compatible nexus versions */
   if(version0 != version1)
   {
       return GLOBUS_FALSE;
   }


   /* find specific proto for sp0 */
   ctr = 0;
   found = GLOBUS_FALSE;
   while(!found)
   {
       if(i0 >= mi_proto_size0)
       {
           return GLOBUS_FALSE;
       }


       UnpackMIProtoEntry(mi_proto_array0,
                          i0,
                          type0,
                          size0,
                          subarray0);
       proto_funcs = globus_i_find_proto_module(type0);
       tmp_int = proto_funcs->proto_count();

       if(proto_index_0 < ctr +  tmp_int
          && proto_index_0 >= ctr)
       {
           found = GLOBUS_TRUE;
           proto_offset0 = proto_index_0 - ctr;
       }
       ctr += tmp_int;
     }

   /* find specific proto for sp1 */
   ctr = 0;
   found = GLOBUS_FALSE;
   while(!found)
   {
       if(i1 >= mi_proto_size1)
       {
           return GLOBUS_FALSE;
       }

       UnpackMIProtoEntry(mi_proto_array1,
                          i1,
                          type1,
                          size1,
                          subarray1);
       proto_funcs = globus_i_find_proto_module(type1);
       tmp_int = proto_funcs->proto_count();

       if(proto_index_1 < tmp_int + ctr 
          && proto_index_1 >= ctr)
       {
           found = GLOBUS_TRUE;
           proto_offset1 = proto_index_1 - ctr;
       }
       ctr += tmp_int;
     }

   /* check for a protocol match */
   if(type0 != type1)
   {
       return GLOBUS_FALSE;
   }
   proto_funcs = globus_i_find_proto_module(type0);
   if(proto_funcs == NULL)
   {
       return GLOBUS_FALSE;
   }

   /* check to see if startpoint_proto_match() has been defined */
   if(proto_funcs->startpoint_proto_match == NULL)
   {
       return GLOBUS_FALSE;
   }

   return proto_funcs->startpoint_proto_match(sp0->mi_proto,
					proto_offset0,
                                        subarray0,
                                        size0,
                                        sp1->mi_proto,
					proto_offset1,
                                        subarray1,
                                        size1);
}






globus_bool_t
globus_i_is_local_area_network(char * hostname0,
			       char * hostname1)
{
    char * tmp_ptr0;
    char * tmp_ptr1;

    tmp_ptr0 = strchr(hostname0, '.');
    tmp_ptr1 = strchr(hostname1, '.');

    if(tmp_ptr0 != GLOBUS_NULL && tmp_ptr1 != GLOBUS_NULL)
    {
        if(strcmp(tmp_ptr0, tmp_ptr1) == 0)
        {
            return GLOBUS_TRUE;
	}
    }
    return GLOBUS_FALSE;
}
