/*
 * Nexus
 * Authors:     James Stepanek
 *              Aerospace
 *
 * overseer_qos.c	- Overseer QoS module
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/libraries/nexus/overseer_qos.c,v 1.11 1998/08/24 16:40:40 stepanek Exp $";

#include "internal.h"

#ifdef BUILD_RESOURCE

#include <fcntl.h>

#include <sys/socket.h>
#include <netinet/in.h>

#include "nexus_resource.h"

#include "qoslib.h"
 
/*
 * Only one thread is allowed to be in the QosLib code (and thus
 * mucking with data structures) at a time.  
 */

#ifdef BUILD_LITE

#define qos_enter()
#define qos_exit()

#else  /* BUILD_LITE */

static nexus_mutex_t	qos_mutex;
#define qos_enter()	nexus_debug_printf(1,("QoS_Enter\n"));nexus_mutex_lock(&qos_mutex);
#define qos_exit()	nexus_mutex_unlock(&qos_mutex);nexus_debug_printf(1,("QoS_Exit\n"));

#endif /* BUILD_LITE */

/*
 * null fd kludge for handling idle events
 */
#ifndef QOS_IDLE_FILE
#define QOS_IDLE_FILE "/dev/null"
#endif
static int		    qoslib_idle_fd;

/* 
 * Qoslib <-> Qosman
 */
static nexus_endpoint_t     qoslib_ep;
static nexus_endpointattr_t qoslib_ep_attr;
static nexus_bool_t	    qoslib_ep_initialized;
static unsigned long 	    qoslib_id;

/*
 * Attachment URL pulled from a file
 */

/* XXX url should come from mds? */

#ifndef QOSMAN_URL_FILE
#define QOSMAN_URL_FILE "/tmp/qosbert_url"
#endif

#ifndef QOSMAN_URL_MAX
#define QOSMAN_URL_MAX 256
#endif

/*
 * Qosman Attachment 
 */
nexus_startpoint_t   	qosman_sp;		/* qosman startpoint */
static struct {	
	nexus_mutex_t		mutex;
	nexus_cond_t		cond;
	enum {
		QOSMAN_DETACHING,
		QOSMAN_DETACHED,
		QOSMAN_ATTACHING,
		QOSMAN_ATTACHED,
	} state;
} qosman_monitor;		 		/* attachment monitor */

#ifdef BUILD_LITE 

#define qosman_enter()
#define qosman_exit()

#else /* nodef BUILD_LITE */

#define qosman_enter()	nexus_mutex_lock(&qosman_monitor.mutex);
#define qosman_exit()	nexus_mutex_unlock(&qosman_monitor.mutex);

#endif /* BUILD_LITE */

/*********************************/
/*                               */
/* Nexus Handler Functions Table */
/*                               */
/*********************************/

static void 
qos_attach_reply_handler( nexus_endpoint_t *, nexus_buffer_t *, nexus_bool_t );
static void 
qos_op_handler( nexus_endpoint_t *, nexus_buffer_t *, nexus_bool_t  );

static nexus_handler_t qoslib_handlers[] =  { 
#define QOS_ATTACH_REPLY_HANDLER_ID        0
    {NEXUS_HANDLER_TYPE_NON_THREADED, qos_attach_reply_handler},
#define QOS_OP_HANDLER_ID	           1
    {NEXUS_HANDLER_TYPE_THREADED, qos_op_handler},
};

/*
 * List of objects which may be associated with qos at some point
 * doubly-linked; null-terminated
 */
static enum {
	QOS_BOUND_RUSAGE	= 0x01,
	QOS_BOUND_REQUEST	= 0x02,
	QOS_BOUND_MASK		= 0x03
};

#define QOS_FULLY_BOUND(s)   \
	(((s) & QOS_BOUND_MASK) == (QOS_BOUND_RUSAGE | QOS_BOUND_REQUEST))

typedef struct _qos_cell {
	nexus_resource_name_t		key;
	struct _qos_cell		*next, *prev, *stack;
	int 				binding_state;
	nexus_rusage_info		rusage_info;
	qos_request_info		request_info;
} qos_cell;

static struct {
	struct _qos_cell		*head, *tail;
} qosres_list;

static qos_cell		*qosres_freelist = NULL;
static qos_cell		*qosres_pending = NULL;

#define QosCellAlloc(Caller, NewCell) \
	if (qosres_freelist) { \
		(NewCell) = qosres_freelist; \
		qosres_freelist = qosres_freelist->next; \
	} else { \
		NexusMalloc(Caller, (NewCell), qos_cell *, \
			sizeof(qos_cell)); \
	} \
	memset(NewCell, 0, sizeof(qos_cell)); 

#define QosCellFree(Cell) \
	(Cell)->next = qosres_freelist; \
	qosres_freelist = (Cell);

#define QosCellFind(Cell, Value) \
	for((Cell) = qosres_list.head; (Cell); (Cell) = (Cell)->next) \
		if ((nexus_resource_name_cmp(&((Cell)->key), (Value))) == 0) \
			break;

/*  
 * These macros work for either kind of freelist as long as the cells
 * have ->next and ->prev and the list head has ->head and ->tail
 */

#define ListPrepend(List, Cell) \
	(Cell)->next = (List)->head; \
	if ((List)->head) (List)->head->prev = (Cell); \
	if ((List)->tail == NULL) (List)->tail = (Cell); \
	(List)->head = (Cell); \
	(Cell)->prev = NULL; 

#define ListRemove(List, Cell) \
	if ((Cell)->prev) (Cell)->prev->next = (Cell)->next; \
	else (List)->head = (Cell)->next; \
	if ((Cell)->next) (Cell)->next->prev = (Cell)->prev; \
	else (List)->tail = (Cell)->prev; \
	(Cell)->next = (Cell)->prev = NULL; 


/*
 * Flags for qos module
 */
static nexus_bool_t	qos_done;
static nexus_bool_t     qos_aborting;
static nexus_bool_t	qos_enabled;

#define qos_fatal qos_exit(); globus_fatal

/********************************************/
/*                                          */
/* Forward declarations for Local Functions */
/*                                          */
/********************************************/

static void	qos_init(void);
static void	qos_shutdown(nexus_bool_t shutdown_others);
static int 	qos_rusage(nexus_rusage_obj*);

static void 	qos_post( void * , int  );
static nexus_bool_t qos_attach_assert();
static int 	qos_attach(char * attach_url);
static void 	qos_detach(void);
static char     *qos_get_url(void);

static void	qos_idle_register(void (*callback)(void*, int), 
				  void * callback_arg);

static int 	_rusage_socket(nexus_rusage_obj*, qos_cell*);
static int 	_rusage_process(nexus_rusage_obj*, qos_cell*);
static int 	_rusage_thread(nexus_rusage_obj*, qos_cell*);
static int 	_rusage_memory(nexus_rusage_obj*, qos_cell*);

static int 	rusage_dump(char * msg, nexus_rusage_obj * obj);

static int 	_request_socket(qos_request_obj *, qos_cell*);
static int 	_request_process(qos_request_obj *, qos_cell*);
static int 	_request_thread(qos_request_obj *, qos_cell*);
static int 	_request_memory(qos_request_obj *, qos_cell*);

/*
 * Rusage function table
 */
static int (*qos_rusage_funcs[NEXUS_RESOURCE_NTYPES])
  (nexus_rusage_obj*, qos_cell*) = {
    _rusage_socket,
    _rusage_process,
    _rusage_thread,
    _rusage_memory
  };

/*
 * Request function table
 */
static int (*qos_request_funcs[NEXUS_RESOURCE_NTYPES])
  (qos_request_obj*, qos_cell*) = {
    _request_socket,
    _request_process,
    _request_thread,
    _request_memory
  };


/*
 * Overseer interface for quality of service, a.k.a. rusage
 */
static nexus_overseer_funcs_t qos_funcs = {
  qos_init,
  qos_shutdown,
  qos_rusage,
};

/*
 * nexus_overseer_qos_info()
 *
 * Return the nexus_overseer_funcs_t function table for this overseer module.
 */
void *nexus_overseer_qos_info(void)    {

  return((void *) (&qos_funcs));

} /* nexus_overseer_qos_info() */

/*
 * qos_init()
 *
 * Initialize the QoS overseer
 */
static void qos_init(void)
{

  static char * idle_file = QOS_IDLE_FILE;

#ifndef BUILD_LITE
  nexus_mutex_init(&qos_mutex, (nexus_mutexattr_t *) NULL);
  nexus_mutex_init(&qosman_monitor.mutex, (nexus_mutexattr_t *) NULL);
#endif /* BUILD_LITE */

  /*
   * initialize fd for idle events
   */
  if (nexus_fd_open(idle_file, O_WRONLY, 0, &qoslib_idle_fd)) {
    globus_warning("Cannot grab idle fd %s.\topen(): %s\n", 
		    idle_file, globus_libc_system_error_string(errno));
  }

  /*
   * initialize attachment monitor
   */
  nexus_cond_init( &(qosman_monitor.cond), 
		   (nexus_condattr_t *)NULL );

  /* 
   * initial monitor state 
   */

  qosman_enter();
  qosman_monitor.state = QOSMAN_DETACHED;
  qosman_exit();

} /* qos_init() */


/*
 * qos_shutdown()
 *
 * This routine is called during normal shutdown of a process.
 */
static void qos_shutdown(nexus_bool_t shutdown_others)   {

  qos_enter();

  qos_detach();
  qos_done = NEXUS_TRUE;
  nexus_fd_close(qoslib_idle_fd);

  if (qoslib_ep_initialized) {
    nexus_endpoint_destroy( &qoslib_ep );
    nexus_endpointattr_destroy( &qoslib_ep_attr );
  }

  qos_exit();

} /* qos_shutdown() */


/*
 * qos_abort()
 *
 * This routine is called during the _abnormal_ shutdown of a process.
 *
 */
static void qos_abort()   {

  nexus_debug_printf(1, ("qos_abort(): entering\n"));
  qos_detach();
  nexus_fd_close(qoslib_idle_fd);

  qos_done     = NEXUS_TRUE;
  qos_aborting = NEXUS_TRUE;

  if (qoslib_ep_initialized) {
    nexus_endpoint_destroy( &qoslib_ep );
    nexus_endpointattr_destroy( &qoslib_ep_attr );
  }

} /* qos_abort() */

/*
 * qos_idle_register()
 * 
 * Execute the passed routine once the event loop starts up.
 *
 */
static void 
qos_idle_register(void (*callback)(void* arg, int unused), 
		  void * callback_arg) {
  nexus_fd_unregister(qoslib_idle_fd, NULL);
  nexus_fd_register_for_select(qoslib_idle_fd, 0, callback, 0,
			       (void*)callback_arg);
} /* qos_idle_register() */


/*
 * qos_detach()
 *
 * This detaches the current qoslib from qosman implying that qosman
 * releases all reservations made from this qoslib.
 *
 */
static void qos_detach( void ) {
  nexus_buffer_t buffer;
  int rsr_op = QOS_DETACH_OP;

  /* Prevent any further rusage/request calls by logically "detaching" */

  qosman_enter();
  if (qosman_monitor.state != QOSMAN_ATTACHED) {
    qosman_exit();
    return;
  }
  qosman_monitor.state = QOSMAN_DETACHING;
  qosman_exit();
  
  /* Detach for real */

  nexus_buffer_init( &buffer, nexus_sizeof_u_long(1)+nexus_sizeof_int(1), 0 );
  nexus_put_u_long( &buffer, &qoslib_id, 1 );
  nexus_put_int( &buffer, &rsr_op, 1 );
  nexus_send_rsr( &buffer, 
		  &qosman_sp,
		  QOS_OP_HANDLER_ID, 
		  NEXUS_TRUE, 
		  NEXUS_FALSE );

  /*  Should we put in a handshake for the detach???  */

  qosman_enter();
  qosman_monitor.state = QOSMAN_DETACHED;
  qosman_exit();

} /* qos_detach() */



/***********************************************************************
 * RUSAGE Interface
 **********************************************************************/


/*
 * qos_rusage()
 *
 * Notify the QoS overseer about resource usage
 *
 */
static int qos_rusage(nexus_rusage_obj* rusage_obj) {

  int rc = QOS_SUCCESS;
  qos_cell * cell;

  qos_enter();

  if (qos_done) {
    qos_exit();
    return(0);
  }

  QosCellFind(cell, rusage_obj->who);

  /*
   * If we have never seen this resource, add it to our list; otherwize
   * modify the existing list entry with the passed rusage object.  
   */

  if (cell == NULL) {
    QosCellAlloc(qos_rusage(), cell);
    memcpy(&(cell->key), rusage_obj->who, sizeof(cell->key));
    ListPrepend(&qosres_list, cell);
  }

  cell->binding_state |= QOS_BOUND_RUSAGE;

  if (qos_rusage_funcs[rusage_obj->who->type]) {

    rc = (qos_rusage_funcs[rusage_obj->who->type])(rusage_obj, cell);

  }

  /* protects the key from nexus */
  cell->rusage_info.object.who = &(cell->key);  

  if (QOS_FULLY_BOUND(cell->binding_state)) {
    cell->stack = qosres_pending;
    qosres_pending = cell;

    qos_idle_register(qos_post, (void*)NULL);
  } 

/*
 * On a destroy, remove the entry from the list if present.  
 * Only delete if it was not posted.
 */
  if (rusage_obj->op == RUSAGE_DESTROY) {
    ListRemove(&qosres_list, cell);
    if(!QOS_FULLY_BOUND(cell->binding_state)) {
      QosCellFree(cell);
    }
  }

  qos_exit();

  return (rc);

} /* qos_rusage() */


/*
 * _rusage_socket()
 *
 * Notify the QoS overseer about resource usage
 *
 */
static int _rusage_socket(nexus_rusage_obj* rusage_obj, qos_cell * cell) {

  memcpy(&cell->rusage_info.socket, rusage_obj, sizeof(nexus_rusage_socket));

  return (rusage_dump("_rusage_socket()!", rusage_obj));
} /* qos_rusage_socket */


/*
 * _rusage_process()
 *
 * Notify the QoS overseer about resource usage
 *
 */
static int _rusage_process(nexus_rusage_obj* rusage_obj, qos_cell * cell)
{
  nexus_buffer_t buffer;

  memcpy(&cell->rusage_info.process, rusage_obj, sizeof(nexus_rusage_process));

  return (rusage_dump("_rusage_process()!", rusage_obj));
} /* _rusage_process */


/*
 * _rusage_thread()
 *
 * Notify the QoS overseer about resource usage
 *
 */
static int _rusage_thread(nexus_rusage_obj* rusage_obj, qos_cell * cell) {

  memcpy(&cell->rusage_info.thread, rusage_obj, sizeof(nexus_rusage_thread));

  return (rusage_dump("_rusage_thread()!", rusage_obj));
} /* _rusage_thread */


/*
 * _rusage_memory()
 *
 * Notify the QoS overseer about resource usage
 *
 */
static int _rusage_memory(nexus_rusage_obj* rusage_obj, qos_cell * cell) {

  memcpy(&cell->rusage_info.memory, rusage_obj, sizeof(nexus_rusage_memory));

  return (rusage_dump("_rusage_memory()!", rusage_obj));
} /* _rusage_memory */



static int rusage_dump(char * msg, nexus_rusage_obj * obj) {
  char * type, * op;

  switch(obj->who->type) {
    case RESOURCE_SOCKET:
      type = "SOCKET";
      break;
    case RESOURCE_PROCESS:
      type = "PROCESS";
      break;
    case RESOURCE_THREAD:
      type = "THREAD";
      break;
    case RESOURCE_MEMORY:
      type = "MEMORY";
      break;
  }

  switch(obj->op) {
    case RUSAGE_CREATE:
      op = "CREATE";
      break;
    case RUSAGE_DESTROY:
      op = "DESTROY";
      break;
    case RUSAGE_MODIFY:
      op = "MODIFY";
      break;
  }

  nexus_debug_printf(1, ("%s: resource: %d  type: %s  op: %s\n", 
	       msg, obj->who->id, type, op));

  return ( QOS_SUCCESS );
}



/***********************************************************************
 * REQUEST Interface
 **********************************************************************/

/*
 * qos_request()
 *
 * Submit request for quality of service 
 */

int qos_request(qos_request_obj *request_obj) {

  int rc = QOS_SUCCESS;
  qos_cell * cell;

  qos_enter();

  if (qos_done) {
    qos_exit();
    return(0);
  }

  qos_exit();  /* don't lock up qos modules while attaching */

  if (!qos_attach_assert()) {
    return (QOS_NOT_ATTACHED);
  }

  /* sanity */

  if (request_obj == NULL) {
    return (QOS_FAILURE);
  }

  switch (request_obj->who->type) {
    case RESOURCE_SOCKET:
    case RESOURCE_PROCESS:
    case RESOURCE_THREAD:
    case RESOURCE_MEMORY:
      break;
    default:
      globus_error("qos_request(): invalid type = %d\n", 
		  request_obj->who->type);
      return (QOS_FAILURE);
  }


  qos_enter();

  QosCellFind(cell, request_obj->who);

  if (cell == NULL) {
    QosCellAlloc(qos_request(), cell);
    memcpy(&(cell->key), request_obj->who, sizeof(cell->key));
    ListPrepend(&qosres_list, cell);
  }

  cell->binding_state |= QOS_BOUND_REQUEST;

  if (qos_request_funcs[request_obj->who->type]) {

    rc = (qos_request_funcs[request_obj->who->type])(request_obj, cell);

  }

  /* protects the key from nexus */
  cell->request_info.object.who = &(cell->key);  

  if (QOS_FULLY_BOUND(cell->binding_state)) {
    cell->stack = qosres_pending;
    qosres_pending = cell;

    qos_idle_register(qos_post, (void*)NULL);
  } 

  qos_exit();

  return (rc);

} /* qos_request() */


/*
 * _request_socket()
 *
 * Submit request for quality of service 
 */
static int _request_socket( qos_request_obj *request_obj, qos_cell* cell )
{
  memcpy( &(cell->request_info.socket), request_obj, sizeof(qos_request_socket) );

  return QOS_SUCCESS;
} /* _request_socket() */


/*
 * _request_process()
 *
 * Submit request for quality of service 
 */
static int _request_process( qos_request_obj *request_obj, qos_cell* cell )
{
  memcpy( &(cell->request_info.process), request_obj, sizeof(qos_request_process) );

  return QOS_SUCCESS;
} /* _request_process() */


/*
 * _request_thread()
 *
 * Submit request for quality of service 
 */
static int _request_thread( qos_request_obj *request_obj, qos_cell* cell )
{
  memcpy( &(cell->request_info.thread), request_obj, sizeof(qos_request_thread) );

  return QOS_SUCCESS;
} /* _request_thread() */


/*
 * _request_memory()
 *
 * Submit request for quality of service 
 */
static int _request_memory( qos_request_obj *request_obj, qos_cell* cell )
{
  memcpy( &(cell->request_info.memory), request_obj, sizeof(qos_request_memory) );

  return QOS_SUCCESS;
} /* _request_memory() */



/**********************************************************************
 *  RSR handling routines:  send an RSR
 **********************************************************************/


/* idle calbacks */

/*
 * qos_post()
 *
 * Send resource rusuage and request pair to QosMan
 *
 */
static void qos_post( void * arg, int unused )  {
  nexus_buffer_t buffer;
  int rsr_op = QOS_RESOURCE_OP;
  qos_cell * cell = qosres_pending;

  qos_enter();

  qosres_pending = NULL;

  /*
   *  The callback func pointer is sent to QosMan for possible later use on the
   *  standard QosLib callback EP
   */

  for (; cell; cell = cell->stack) {

    nexus_buffer_init( &buffer,
		       nexus_sizeof_int(2) +
		       nexus_resource_sizeof_name(1) + 
		       nexus_sizeof_rusage( (nexus_rusage_obj *)&(cell->rusage_info), 1) + 
		       qos_sizeof_request( (qos_request_obj*)&(cell->request_info), 1) ,

		       0
      );

    nexus_put_u_long( &buffer, &qoslib_id, 1 );
    nexus_put_int( &buffer, &rsr_op, 1 );

    nexus_resource_name_put( &buffer, &(cell->key) );  /* The resource name contains the type */

    nexus_rusage_put( &buffer, (nexus_rusage_obj *)&(cell->rusage_info) );

    qos_request_put( &buffer, (qos_request_obj *)&(cell->request_info) );

    switch(((nexus_rusage_obj *)&(cell->rusage_info))->op) {
/*
 * On a destroy, free the Cell for real.  
 */
      case RUSAGE_DESTROY:
	QosCellFree(cell);
	break;
    }

    qos_exit();

    nexus_send_rsr( &buffer, &qosman_sp, 
		  QOS_OP_HANDLER_ID, NEXUS_TRUE, NEXUS_FALSE);

    qos_enter();

  }

  qos_exit();

} /* qos_post() */

/***********************************************************************
 * attach_to_qosman()
 *
 * Attachment in Nexus is done by making a request to attach to another
 * process designated by a URL.  If the request is granted, we get a 0
 * return code and a startpoint that points back to the granting process.
 *
 * NOTE: We get a startpoint to the process we just attached to, they
 *       do not get a startpoint back to us.  If we need them to have
 *       a startpoint back to us, we need to send them one using the
 *       startpoint we just received ... the one that points back to them.
 *
 * If the request is denied, we get a non-zero return code back.
 *
 ***********************************************************************/

static int qos_attach(char * attach_url) {
  nexus_startpoint_t qoslib_sp;
  nexus_buffer_t buffer;

  /* try to attach */

  nexus_printf("attempting attach to %s\n", (char*)attach_url);

  if  (nexus_attach((char*) attach_url, &qosman_sp)) {
    globus_error("Could not attach to qosman using url %s: %s.\n", 
		(char*)attach_url, globus_libc_system_error_string(errno));
    return (-1);
  } /* endif */

  /* if things went OK, we now have a startpoint to qosman */

  /*********************************/
  /* send qosman a startpoint      */
  /* to myself and wait for ACK    */
  /*********************************/

  nexus_debug_printf(1, ("attach successful ... sending my startpoint to qosman\n"));

  nexus_startpoint_bind( &qoslib_sp, &qoslib_ep );
  nexus_buffer_init( &buffer, nexus_sizeof_startpoint(&qoslib_sp, 1), 0 );
  nexus_put_startpoint_transfer( &buffer, &qoslib_sp, 1 );

  /* sending startpoint to qosman and invoking attach_handler() there */

  nexus_send_rsr( &buffer,
		  &qosman_sp,
		  QOS_ATTACH_REPLY_HANDLER_ID,
		  NEXUS_TRUE,
		  NEXUS_FALSE );

  return (0);

} /* qos_attach()*/


/*
 * qos_attach_assert()
 *
 * If not connected, get connected.  Use the attacment monitor to
 * prevent multiple attempts.
 */
static nexus_bool_t qos_attach_assert() {
  nexus_bool_t rc = NEXUS_TRUE;
  char * attach_url;

  /*
   * Initialize qosman qoslib endpoint
   */
  qos_enter();

  if (!qoslib_ep_initialized) {
    nexus_endpointattr_init( &qoslib_ep_attr );
    nexus_endpointattr_set_handler_table( &qoslib_ep_attr, qoslib_handlers,
					  sizeof(qoslib_handlers)/sizeof(nexus_handler_t) );
    nexus_endpoint_init( &qoslib_ep, &qoslib_ep_attr );
    qoslib_ep_initialized = NEXUS_TRUE;
  }

  qos_exit();

  qosman_enter();

  if (qosman_monitor.state == QOSMAN_DETACHED) {

    qosman_monitor.state == QOSMAN_ATTACHING;

    qosman_exit();

    if ((attach_url = qos_get_url()) == NULL) {
      rc = NEXUS_FALSE; goto abort;
    }

    if (qos_attach(attach_url) < 0) {
      rc = NEXUS_FALSE; goto abort;
    }

    qosman_enter();

  } 

  /* Monitors can get a false wakeup so we have to test for the desired  */
  /* condition on each wakeup                                            */

  while  ( qosman_monitor.state != QOSMAN_ATTACHED )  {
    nexus_cond_wait( &(qosman_monitor.cond), 
		     &(qosman_monitor.mutex) );
  } /* endwhile */
  qosman_exit();

  /* NexusFree attach_url? */

abort:

  return (rc);

} /* qos_attach_assert() */


/*
 * qos_get_url()
 * 
 * get the attachment url from the url file
 *
 */
static char * qos_get_url(void) {
  static char * url_file = QOSMAN_URL_FILE;

  FILE *fp;
  char buffer[QOSMAN_URL_MAX];

  nexus_debug_printf(1, ("opening attach url file, %s ...", url_file));
  if ((fp = fopen(url_file, "r")) == (FILE *) NULL) {
    return (NULL);
  } else  {
    char * attach_url;

    /* try and read the url file */

    nexus_stdio_lock();
    fscanf(fp, "%s", buffer);
    nexus_stdio_unlock();
    fclose(fp);

    NexusMalloc(gos_get_url(), attach_url, char *, strlen(buffer));
    strcpy(attach_url, buffer);
    return attach_url;
  } /* endif */
} /* qos_get_url() */


/***************************/
/*                         */
/* Nexus Handler Functions */
/*                         */
/***************************/

/*
 * attach_reply_handler()
 */
static void 
qos_attach_reply_handler( nexus_endpoint_t *endpoint, 
			  nexus_buffer_t *buffer,
			  nexus_bool_t is_non_threaded_handler )
{
  /*  This is where qoslib receives its id from QoSMan  */

  nexus_get_u_long( buffer, &qoslib_id, 1 );
  nexus_printf("QosLib recvd client id [%x]\n", qoslib_id );

  /* clean-up */

  nexus_buffer_destroy(buffer);

  qosman_enter();
  qosman_monitor.state = QOSMAN_ATTACHED;
  nexus_cond_signal( &(qosman_monitor.cond) );
  qosman_exit();

} /* end qos_attach_reply_handler() */


/*
 * qoslib_op_handler()
 *
 * This is where qoslib will execute operations on behalf of qosman.
 * The current body here is just for debugging and development.
 */

static void 
qos_op_handler( nexus_endpoint_t *endpoint, 
		nexus_buffer_t *buffer, 
		nexus_bool_t is_non_threaded_handler ) {

  qos_callback *cb_func;
  qos_status_obj arg;
  nexus_resource_name_t dummy_name;

  nexus_get_u_long( buffer, (unsigned long *)(cb_func), 1 );
  nexus_get_int( buffer, (int *)&arg, 1 );

  nexus_printf( "Client recvd arg [%d]\n", arg );

  (*cb_func)( &dummy_name, &arg );  /* Call the App Callback */

  nexus_buffer_destroy(buffer);    /* clean-up */

} /* end qos_op_handler() */


/***************************/
/*                         */
/*  Development Code       */
/*                         */
/***************************/

void qos_list_qosman( void )
{
  nexus_buffer_t buffer;
  int rsr_op = QOS_LIST_OP;

  if (!qos_attach_assert()) {
    return;
  }

  nexus_buffer_init( &buffer, nexus_sizeof_u_long(1)+nexus_sizeof_int(1), 0 );
  nexus_put_u_long( &buffer, &qoslib_id, 1 );
  nexus_put_int( &buffer, &rsr_op, 1 );
  nexus_send_rsr( &buffer, &qosman_sp, QOS_OP_HANDLER_ID, 
		  NEXUS_TRUE, NEXUS_FALSE);
}

#endif /* BUILD_RESOURCE */

