/*
 * Nexus
 * Authors:     James Stepanek
 *              Aerospace
 *
 * nexus_resource.c	- Nexus resources
 */

static char *rcsid = "$Header: /home/globdev/CVS/globus-current/Globus/Communication/nexus/libraries/nexus/nexus_resource.c,v 1.8 1998/08/24 16:40:39 stepanek Exp $";

#include "internal.h"

#if defined(BUILD_RESOURCE)

#include <sys/socket.h>
#include <netinet/in.h>

#include "nexus_resource.h"

#ifdef BUILD_LITE
#define resource_enter()
#define resource_exit()
#else  /* BUILD_LITE */
static nexus_mutex_t	resource_mutex;
#define resource_enter()	nexus_mutex_lock(&resource_mutex);
#define resource_exit()	nexus_mutex_unlock(&resource_mutex);
#endif /* BUILD_LITE */

static int current_id = 0;

static int _sizeof_socket(nexus_rusage_obj*, int);
static int _sizeof_process(nexus_rusage_obj*, int);
static int _sizeof_thread(nexus_rusage_obj*, int);
static int _sizeof_memory(nexus_rusage_obj*, int);

static void _put_socket(nexus_buffer_t*,  nexus_rusage_obj*);
static void _put_process(nexus_buffer_t*,  nexus_rusage_obj*);
static void _put_thread(nexus_buffer_t*,  nexus_rusage_obj*);
static void _put_memory(nexus_buffer_t*,  nexus_rusage_obj*);

static void _get_socket(nexus_buffer_t*,  nexus_rusage_obj*);
static void _get_process(nexus_buffer_t*,  nexus_rusage_obj*);
static void _get_thread(nexus_buffer_t*,  nexus_rusage_obj*);
static void _get_memory(nexus_buffer_t*,  nexus_rusage_obj*);

static struct _rusage_funcs {
  int (*size)	(nexus_rusage_obj*, int);
  void (*put)	(nexus_buffer_t*, nexus_rusage_obj*);
  void (*get)	(nexus_buffer_t*, nexus_rusage_obj*);
} rusage_funcs[NEXUS_RESOURCE_NTYPES] = {
  {_sizeof_socket, _put_socket, _get_socket},		/* SOCKET */
  {_sizeof_process, _put_process, _get_process},	/* PROCESS */
  {_sizeof_thread, _put_thread, _get_thread},		/* THREAD */
  {_sizeof_memory, _put_memory, _get_memory}		/* MEMORY */
};


/*
 * nexus_resource_name()
 */
int nexus_resource_name(nexus_resource_name_t* name , 
			nexus_resource_type type) {

  resource_enter();

  name->id = current_id++;

  resource_exit();

  name->type = type;

  return (0);

}

/*
 * nexus_resource_name_cmp()
 */
int nexus_resource_name_cmp(const nexus_resource_name_t* n1, 
			    const nexus_resource_name_t* n2)   {
  if ((n1 == NULL) || (n2 == NULL)) return (-1);
  if (n1->id > n2->id) return(1);
  if (n1->id < n2->id) return(-1);
  if (n1->id == n2->id) return(0);
}

/*
 * nexus_resource_name_cp()
 */
int nexus_resource_name_cp( nexus_resource_name_t* n1, const nexus_resource_name_t* n2 )
{
  if  ( n1 && n2 )  {
    n1->id = n2->id;
    n1->type = n2->type;
    return 0;
  }
}

/*
 * nexus_resource_sizeof_name()
 */
#undef nexus_resource_sizeof_name
int nexus_resource_sizeof_name(int count) {
  return (nexus_resource_macro_sizeof_name(count));
}

/*
 * nexus_resource_put_name()
 */
void nexus_resource_name_put(nexus_buffer_t *buffer,
			     nexus_resource_name_t *name) {
  nexus_put_int(buffer, &name->id, 1);
  nexus_put_int(buffer, (int*)&name->type, 1);
}

/*
 * nexus_resource_get_name()
 */
void nexus_resource_name_get(nexus_buffer_t *buffer,
			     nexus_resource_name_t *name) {
  nexus_get_int(buffer, &name->id, 1);
  nexus_get_int(buffer, (int*)&name->type, 1);
}

/*
 *  name access routines
 */

/*
 * nexus_context_name()
 */
nexus_resource_name_t *nexus_context_name( void ) {
  nexus_context_t *contextp;

  _nx_context( &contextp );
  return  &(contextp->resource_name);
}


/*
 * nexus_thread_name()
 */
nexus_resource_name_t *nexus_thread_name( void ) {
#ifdef notdef  /* comment out until we decide what to do for threads */
#ifndef BUILD_LITE
  globus_thread_t *tp;

  globus_thread_self( &tp );
  return  &(tp->resource_name);
#else
  return ((nexus_resource_name_t *)NULL);

#endif /* BUILD_LITE */
#endif
}


/*
 * nexus_memory_name()
 */
nexus_resource_name_t *nexus_memory_name( void ) {
  nexus_resource_name_t *new_mem_name;

  new_mem_name = nexus_malloc( sizeof(nexus_resource_name_t) );
  nexus_resource_name( new_mem_name, RESOURCE_MEMORY );
  return new_mem_name;
}


/*
 * nexus_socket_name_sp()
 */
nexus_resource_name_t *nexus_socket_name_sp( nexus_startpoint_t *sp ) {
  nexus_resource_name_t * result;

  if (sp == NULL || sp->mi_proto == NULL || sp->mi_proto->proto == NULL
      || sp->mi_proto->proto->funcs->get_resource_name_sp == NULL) {
    result = NULL;
  } else {
    result = sp->mi_proto->proto->funcs->get_resource_name_sp( sp );
  }
  return (result);
}

/*
 * nexus_socket_name_sp()
 */
nexus_resource_name_t *nexus_socket_name_ep( nexus_proto_type_t proto_type,
					     nexus_endpoint_t *ep ) {
  nexus_resource_name_t * result = NULL;
  nexus_mi_proto_t * mi_proto;

  return _nx_mi_proto_name_for_proto_type(proto_type, ep);

}


/*
 * nexus_sizeof_rusage()
 */
int nexus_sizeof_rusage( nexus_rusage_obj *rusage_obj, int count ) {

  if  ( rusage_funcs[rusage_obj->who->type].size )

    return (rusage_funcs[rusage_obj->who->type].size(rusage_obj, count));

  return (0);
}

/*
 * nexus_rusage_put()
 */
void nexus_rusage_put( nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj ) {

  if  ( rusage_funcs[rusage_obj->who->type].put ) 
    rusage_funcs[rusage_obj->who->type].put( buffer, rusage_obj );
}

/*
 * nexus_rusage_get()
 */
void nexus_rusage_get( nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj ) {

  if  ( rusage_funcs[rusage_obj->who->type].get )
    rusage_funcs[rusage_obj->who->type].get( buffer, rusage_obj );
}


/* SOCKET */

static int _sizeof_socket(nexus_rusage_obj *rusage_obj, int count) {

  return (count * (nexus_sizeof_int(5) + 
		   nexus_sizeof_byte(sizeof(struct sockaddr_in))));
}


static void _put_socket(nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj) {
  nexus_rusage_socket * rug = (nexus_rusage_socket*)rusage_obj;

  nexus_put_int( buffer, (int*)&rug->op, 1 );
  nexus_put_int( buffer, &rug->pid, 1 );
  nexus_put_int( buffer, &rug->fd, 1 );
  nexus_put_int( buffer, &rug->mode, 1 );
  nexus_put_byte( buffer, (nexus_byte_t*)&rug->addr, 
		 sizeof(struct sockaddr_in) );
  nexus_put_int( buffer, &rug->protocol, 1 );
}


static void _get_socket(nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj) {
  nexus_rusage_socket * rug = (nexus_rusage_socket*)rusage_obj;

  nexus_get_int(buffer, (int*)&rug->op, 1);
  nexus_get_int(buffer, &rug->pid, 1);
  nexus_get_int(buffer, &rug->fd, 1);
  nexus_get_int(buffer, &rug->mode, 1);
  nexus_get_byte(buffer, (nexus_byte_t*)&rug->addr, 
		 sizeof(struct sockaddr_in));
  nexus_get_int(buffer, &rug->protocol, 1);
}


/* PROCESS */

static int _sizeof_process(nexus_rusage_obj *rusage_obj, int count){
  return (count * nexus_sizeof_int(2));
}

static void _put_process(nexus_buffer_t *buffer, 
			 nexus_rusage_obj *rusage_obj) {
  nexus_rusage_process * rug = (nexus_rusage_process*)rusage_obj;

  nexus_put_int(buffer, (int*)&rug->op, 1);
  nexus_put_int(buffer, &rug->pid, 1);

}

static void _get_process(nexus_buffer_t *buffer, 
			 nexus_rusage_obj *rusage_obj) {
  nexus_rusage_process * rug = (nexus_rusage_process*)rusage_obj;

  nexus_get_int(buffer, (int*)&rug->op, 1);
  nexus_get_int(buffer, &rug->pid, 1);
}

/* THREAD */

static int _sizeof_thread(nexus_rusage_obj *rusage_obj, int count) {
  return (count * nexus_sizeof_int(3));
}

static void _put_thread(nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj) {
  nexus_rusage_thread * rug = (nexus_rusage_thread*)rusage_obj;

  nexus_put_int(buffer, (int*)&rug->op, 1);
  nexus_put_int(buffer, &rug->pid, 1);
  nexus_put_int(buffer, &rug->tid, 1);
}

static void _get_thread(nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj) {
  nexus_rusage_thread * rug = (nexus_rusage_thread*)rusage_obj;

  nexus_get_int(buffer, (int*)&rug->op, 1);
  nexus_get_int(buffer, &rug->pid, 1);
  nexus_get_int(buffer, &rug->tid, 1);
}


/* MEMORY */

static int _sizeof_memory(nexus_rusage_obj *rusage_obj, int count) {
  return (count * (nexus_sizeof_int(count) + nexus_sizeof_u_long(count)));
}

static void _put_memory(nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj) {
  nexus_rusage_memory * rug = (nexus_rusage_memory*)rusage_obj;

  nexus_put_int(buffer, (int*)&rug->op, 1);
  nexus_put_u_long(buffer, rug->handle, 1);
}

static void _get_memory(nexus_buffer_t *buffer, nexus_rusage_obj *rusage_obj) {
  nexus_rusage_memory * rug = (nexus_rusage_memory*)rusage_obj;

  nexus_get_int(buffer, (int*)&rug->op, 1);
  nexus_get_u_long(buffer, (unsigned long *)&rug->handle, 1);
}


#endif /* BUILD_RESOURCE */
