/******************************************************************************
globus_duct_control_test.c

Description:

    Test program for globus_duct_control API. (and globus_duct_runtime API)

    This program does the following operation in order to test the
    globus_duct_control API:
    
    - initialise a "duct" group (using the size passed in argument to
    the program)
    
    - Get its own control_contact (as an URL or as an LSP, depending on
    program argument)
    
    - Start a set of jobs using duct runtime (globus_duct_runtime_test),
    passing them the control_contact through an env.variable, and monitor
    them checking in.
******************************************************************************/
    
#include <stdio.h>
#include "globus_common.h"
#include "globus_nexus.h"
#include "globus_duct_control.h"
#include "globus_duct_test.h"



#define RUNTIME_PROG "gram_duct_runtime_test"

char * runtime_prog=RUNTIME_PROG;
/*
  define the two call back of globus_duct_control_init although 
  globus_duct_control_init doesn not call them (not implemented yet)
*/

void
duct_control_init_cb(struct globus_duct_control_s * controlp,
		     int                            size, 
		     void                         * userdata)
{
    fprintf(stderr, "duct_control_init_cb activated\n");
} /* duct_control_init_cb() */

void
duct_control_init_user_cb()
{
    fprintf(stderr, "duct_control_init_user_cb activated\n");
} /* duct_control_init_user_cb() */


int start_runtime_process(int nb_process, char * argv[])
{
    int i;
    int rc;
    
    for (i=0; i<nb_process; i++)
    {
    fprintf(stderr,
	    "Fork process %d ",i);
	if (rc=fork())
	{
	    if (rc<0)
	    {
		fprintf(stderr, "Fork failed...");
	    }
	    fprintf(stderr, "Forked PID %d\n",rc);
	}
	else
	{
	    
	    fprintf(stderr,
		    "I am the Child... \n");
	    /* I am a child process*/
	    execv(runtime_prog,argv);
	    printf("Exec failed !\n");
	}
    } /* for group_size : start each applications */

} /* start_runtime_process() */

int
main(
    int					argc,
    char *				argv[])
{
    int                     group_size = 4;
    int                     group_resize = -1;
    int                     nb_I_dont_start;
    globus_bool_t           useURL=GLOBUS_TRUE;

    globus_duct_control_t   controlp;
    char                  * control_contact= GLOBUS_NULL;
    char                  * env_control_contact= GLOBUS_NULL;
    int                     rc;
    int                     i;
    
    /* for parsing arg */
    int errflg = 0;
    int c;
    int Uflg = 0;
    int Lflg = 0;

    

    /* parse arguments */
    while ((c = getopt(argc, argv, "s:r:p:LU")) != EOF)
    {
	switch (c) {
	case 'L':
	    if (Uflg)
		errflg++;
	    else
		Lflg++;
	    break;
	case 'U':
	    if (Lflg)
		errflg++;
	    else
		Uflg++;
	    break;
	case 's':
	    group_size = atoi(optarg);
	    if (group_size < 0) errflg++;
	    break;
	case 'r':
	    group_resize = atoi(optarg);
	    if (group_resize < 0) errflg++;
	    break;
	case 'p':
	    runtime_prog=optarg;
	    break;
	case '?':
	    errflg++;
	}
    }
    if (group_resize!=-1 && group_resize<=group_size) errflg++;
    
    if (errflg) {
	(void)fprintf(stderr,
		      "usage: %s [-s group_size] [-U|-L] [-r resize_size] \n",argv[0]);
	(void)fprintf(stderr,
		      "with :\n");
	(void)fprintf(stderr,
		      "       group_size (positive integer) : size of the duct group (default size: 4)\n");
	(void)fprintf(stderr,
		      "       -U: Use an URL based control_contact (default)\n");
	(void)fprintf(stderr,
		      "       -L: Use an LSP based control_contact\n");
	(void)fprintf(stderr,
		      "       resize_size (positive integer) : size of the duct group  after a resize. Must be bigger than group_size (default: no resize)\n");	exit (2);
    }

    if (group_resize!=-1)
    {
	/* I want to try to resize, and therefor I do not want to start all
	 * the process immediatly
	 */
        nb_I_dont_start=1;
    }
    else
    {
	nb_I_dont_start=0;
    }
	
    
    useURL= (globus_bool_t)Uflg;
    /* parsing done */

    if ( globus_module_activate (GLOBUS_DUCT_CONTROL_MODULE) != GLOBUS_SUCCESS )
    {
	fprintf(stderr,
		"Activation of Module GLOBUS_DUCT_CONTROL_MODULE failed !\n");
	exit(1);
    }
    
    fprintf(stderr,"initialize duct_control...\n");
    /* note: callback and callback_userdata  are not implemented in API yet */
    rc = globus_duct_control_init (&controlp,
				   group_size,
				   duct_control_init_cb,
				   (void *)duct_control_init_user_cb);
	
    fprintf(stderr,"duct_control initialized (size %d)\n",group_size);

    /*
      now "runtime" duct applications can checkin, the internal API handler
      s_checking_approval register them and when they all have registered,
      it call the function s_configure_group to dispatch the topology of
      the group to each nodes.

      I must start them  and  be alive until they all have registered...
    */

    /* before I start the application, lets get our own control_contact */


    if (useURL)
    {
	globus_duct_control_contact_url(&controlp,
					&control_contact);
    }
    else
    {
	globus_duct_control_contact_lsp (&controlp,
					 &control_contact);
    }
    fprintf(stderr,
	    "control_contact: %s\n",
	    control_contact);
    
    /*
      start the "groupt size" application. In this test, it is of course
      gram_duct_runtime_test.
    */
    fprintf(stderr,
	    "\nsetenv %s %s\n",
	    GRAM_DUCT_TEST_CHECKIN_CONTACT_ENV_VAR,
	    control_contact);
   
    env_control_contact = globus_malloc(
	strlen(control_contact)+
	strlen(GRAM_DUCT_TEST_CHECKIN_CONTACT_ENV_VAR)+
	2); /* +2 for the = sign and the \0 */
    if (env_control_contact == GLOBUS_NULL)
    {
	fprintf(stderr, " No more memory !?\n");
	exit(1);
    }
    sprintf(env_control_contact,
	    "%s=%s",
	    GRAM_DUCT_TEST_CHECKIN_CONTACT_ENV_VAR,
	    control_contact);

    if (putenv(env_control_contact))
    {
	fprintf(stderr, "Putenv failed !... No more memory !?\n");
	exit(1);	
    };

    
    fprintf(stderr, "\nStart %d process\n",group_size-nb_I_dont_start);
    if (nb_I_dont_start)
    {
	fprintf(stderr, "(keep %d process for after resizing)\n",nb_I_dont_start);
    }
    
    start_runtime_process(group_size-nb_I_dont_start, argv);

    /* If I want to test resize, do it now:
     */

    if (nb_I_dont_start)
    {
	/* resize the group */
	fprintf(stderr, "\nResizing group to %d\n",group_resize);
	
	rc = globus_duct_control_set_groupsize(&controlp,group_resize);

	/* and start more process : The one extra (group_resize-group_size) and the one I did
	 * not start first just to keep the runtime process to get there structure before
	 * I have resized (nb_I_dont_start)
	 */
	fprintf(stderr, "Start %d more process\n",group_resize-group_size+nb_I_dont_start);
	start_runtime_process(group_resize-group_size+nb_I_dont_start, argv);
    }
    /*
      now wait that all the applications have registered...
    */

    fprintf(stderr,
	    "Parent waiting\n");
    
    while (controlp.size != globus_list_size (controlp.checkins)) 
    {
	globus_poll();
	globus_libc_usleep(500);	
    }
    fprintf(stderr,"duct_control has got a complete group... It can die now.\n");
    
    globus_duct_control_destroy (&controlp);
    fprintf(stderr,"duct_control group destroyed\n");

    exit(0);
}
