/******************************************************************************
globus_duct_runtime.c

Description:
    globus_duct runtime API

CVS Information:

    $Source: /home/globdev/CVS/globus-current/Globus/Communication/duct/libraries/runtime/globus_duct_runtime.c,v $
    $Date: 1999/06/16 22:19:20 $
    $Revision: 1.19 $
    $Author: czajkows $
******************************************************************************/

/******************************************************************************
                             Include header files
******************************************************************************/
#include <assert.h>
#include "globus_nexus.h"
#include "globus_duct_runtime.h"
#include "nxbuff.h"

#include "globus_duct_common.h"

#include "globus_common.h"   /* for globus_list_t and globus_hashtable_t */

#include "utils.h"
#include "nxbuff.h"

/******************************************************************************
                               Type definitions
******************************************************************************/

/******************************************************************************
                          Module specific prototypes
******************************************************************************/
static void 
s_data_msg_handler (globus_nexus_endpoint_t * endpointp,
		    globus_nexus_buffer_t   * bufferp,
		    globus_bool_t       is_non_threaded_handler);

static void 
s_config_group_msg_handler (globus_nexus_endpoint_t * endpointp,
			    globus_nexus_buffer_t   * bufferp,
			    globus_bool_t       is_non_threaded_handler);

static void 
s_abort_msg_handler (globus_nexus_endpoint_t * endpointp,
		     globus_nexus_buffer_t   * bufferp,
		     globus_bool_t       is_non_threaded_handler);

static
void s_proto_negotiate_handler (globus_nexus_endpoint_t * endpointp,
				globus_nexus_buffer_t   * bufferp,
				globus_bool_t      is_non_threaded_handler);

static int
s_strlen (const char *str);

/******************************************************************************
                       Define module specific variables
******************************************************************************/

globus_module_descriptor_t globus_duct_runtime_module =
{
  "globus_duct_runtime",
  globus_duct_runtime_activate,
  globus_duct_runtime_deactivate,
  GLOBUS_NULL /* at exit */,
  GLOBUS_NULL /* get_function_ptr */
};


static
globus_nexus_handler_t s_data_handlert[] =
{
  {
    NEXUS_HANDLER_TYPE_THREADED,
    (globus_nexus_handler_func_t) s_data_msg_handler
  }
};

#define DATA_MSG_ID 0
#define DATA_HANDLERT_SIZE 1

static
globus_nexus_handler_t s_config_handlert[] =
{
  {
    NEXUS_HANDLER_TYPE_THREADED,
    (globus_nexus_handler_func_t) s_config_group_msg_handler
  },
  {
    NEXUS_HANDLER_TYPE_THREADED,
    (globus_nexus_handler_func_t) s_abort_msg_handler
  },
  {
    NEXUS_HANDLER_TYPE_THREADED,
    (globus_nexus_handler_func_t) s_proto_negotiate_handler
  }
};
/* KEEP THIS TABLE CONSISTENT WITH globus_duct_control.c :
 * CONFIG_GROUP_MSG_ID = 0 
 * ABORT_MSG_ID        = 1 
 * PROTO_NEGOTIATE     = 2 */

#define CONFIG_HANDLERT_SIZE 3


/* KEEP THIS DEFINITION CONSISTENT WITH duct-control.c */

#define CHECKIN_MSG_ID  0
#define ABORT_REQ_ID    1

static int 
s_strlen (const char *str)
{
  int len = 0;

  while (str[len] != '\0') len++;

  return len;
}

/******************************************************************************
Function:	globus_duct_runtime_activate()
Description:
Parameters:
Returns:
******************************************************************************/
int 
globus_duct_runtime_activate (void)
{
  if ( globus_module_activate (GLOBUS_COMMON_MODULE) != GLOBUS_SUCCESS ) 
    goto activate_common_module_error;

  if ( globus_module_activate (GLOBUS_THREAD_MODULE) != GLOBUS_SUCCESS )
    goto activate_thread_module_error;

  if ( globus_module_activate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    goto activate_globus_nexus_module_error;

  return GLOBUS_SUCCESS;


activate_globus_nexus_module_error:
  globus_module_deactivate (GLOBUS_NEXUS_MODULE);
  
activate_thread_module_error:
  globus_module_deactivate (GLOBUS_COMMON_MODULE);

activate_common_module_error:
  return GLOBUS_FAILURE;
}

/******************************************************************************
Function:	globus_duct_runtime_deactivate()
Description:
Parameters:
Returns:
******************************************************************************/
int
globus_duct_runtime_deactivate (void)
{
  int rc;

  rc = GLOBUS_SUCCESS;

  if ( globus_module_deactivate (GLOBUS_NEXUS_MODULE) != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  if ( globus_module_deactivate (GLOBUS_THREAD_MODULE) != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  if ( globus_module_deactivate (GLOBUS_COMMON_MODULE) != GLOBUS_SUCCESS )
    rc = GLOBUS_FAILURE;

  return rc;
}

/******************************************************************************
Function:	gram_globus_duct_runtime_init()
Description:
Parameters:
Returns:
******************************************************************************/
int 
globus_duct_runtime_init (globus_duct_runtime_t     * runtimep,
			  const char                * checkin_contact,
			  int                         ignored_checkin_id,
			  globus_duct_data_callback_t data_callback,
			  void                      * data_callback_userdata,
			  globus_duct_config_callback_t config_callback,
			  void                      * config_callback_userdata)
{
  int                err;

  if ( (runtimep == NULL)
       || (config_callback != NULL) ) return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;

  if ( ignored_checkin_id == 0 ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_checkin_id = 0;
  }

  if ( (checkin_contact != NULL) ) {
    err = globus_duct_runtime_make_startpoint (&(runtimep->control),
					       checkin_contact);
    if (err) {
      /* err already set */
      goto runtime_init_sp_dest_error;
    }
    else {
      runtimep->control_initialized = 1;
    }

    runtimep->local_address = -1;
  }
  else {
    runtimep->control_initialized = 0;
    runtimep->local_address = 0;
  }

  runtimep->aborted = 0;
  runtimep->proto_negotiate = 0;
  runtimep->remote_addresses = NULL;
  runtimep->data_callback = data_callback;
  runtimep->data_callback_userdata = data_callback_userdata;
  runtimep->config_callback = config_callback;
  runtimep->config_callback_userdata = config_callback_userdata;

  err = globus_hashtable_init (&(runtimep->remote_data_spst),
			       16 /* zero-information default */,
			       globus_hashtable_int_hash,
			       globus_hashtable_int_keyeq);
  assert (!err);

  err = globus_mutex_init (&(runtimep->mutex), NULL); assert (!err);

  err = globus_cond_init (&(runtimep->cond), NULL); assert (!err);

  err = globus_nexus_endpointattr_init (&(runtimep->data_port.epattr));
  assert (!err);

  err = globus_nexus_endpointattr_init (&(runtimep->config_port.epattr));
  assert (!err);

  err = globus_nexus_endpointattr_set_handler_table (&(runtimep->data_port.epattr),
					      s_data_handlert,
					      DATA_HANDLERT_SIZE);
  assert (!err);

  err = globus_nexus_endpointattr_set_handler_table (&(runtimep->config_port.epattr),
					      s_config_handlert,
					      CONFIG_HANDLERT_SIZE);
  assert (!err);

  err = globus_nexus_endpoint_init (&(runtimep->data_port.ep),
				    &(runtimep->data_port.epattr));
  assert (!err);

  err = globus_nexus_endpoint_init (&(runtimep->config_port.ep),
				    &(runtimep->config_port.epattr));
  assert (!err);

  globus_nexus_endpoint_set_user_pointer (&(runtimep->data_port.ep),
					  (void *) runtimep);
  
  globus_nexus_endpoint_set_user_pointer (&(runtimep->config_port.ep),
					  (void *) runtimep);

  err = globus_nexus_startpoint_bind (&(runtimep->data_port.sp),
				      &(runtimep->data_port.ep));
  assert (!err);

  err = globus_nexus_startpoint_bind (&(runtimep->config_port.sp),
				      &(runtimep->config_port.ep));
  assert (!err);

  /* now send checkin message */
  if ( runtimep->control_initialized ) {
    globus_nexus_buffer_t     buffer;
    globus_nexus_startpoint_t sp_copy;

    err = globus_nexus_buffer_init (&buffer, 0, 0); assert (!err);

    /* the checkin protocol we are using */
    err = nxbuff_put_int (&buffer, GLOBUS_DUCT_CHECKIN_PROTOCOL_MAX_VERSION);
    assert (!err);

    err = globus_nexus_startpoint_copy (&sp_copy, &(runtimep->config_port.sp));
    assert (!err);
    err = nxbuff_put_startpoint_transfer (&buffer, &sp_copy);
    assert (!err);

    /* the range of config protocols we can support */
    err = nxbuff_put_int (&buffer, GLOBUS_DUCT_CONFIG_PROTOCOL_MIN_VERSION);
    assert (!err);
    err = nxbuff_put_int (&buffer, GLOBUS_DUCT_CONFIG_PROTOCOL_MAX_VERSION);
    assert (!err);

    err = globus_nexus_startpoint_copy (&sp_copy, &(runtimep->data_port.sp));
    assert (!err);
    err = nxbuff_put_startpoint_transfer (&buffer, &sp_copy);
    assert (!err);
  
    /* the range of data protocols we can support */
    err = nxbuff_put_int (&buffer, GLOBUS_DUCT_DATA_PROTOCOL_MIN_VERSION);
    assert (!err);
    err = nxbuff_put_int (&buffer, GLOBUS_DUCT_DATA_PROTOCOL_MAX_VERSION);
    assert (!err);

    err = globus_nexus_send_rsr (&buffer, &(runtimep->control),
			  CHECKIN_MSG_ID,
			  NEXUS_TRUE /* destroy buffer */,
			  NEXUS_TRUE /* always safe */);

    globus_nexus_startpoint_flush (&(runtimep->control));

    if (err) {
      /* fatal condition: checkin send failed, so exit! */
      err = GLOBUS_DUCT_ERROR_CHECKIN_FAILED;
      goto runtime_init_checkin_failed;
    }
  }

  err = globus_mutex_lock (&(runtimep->mutex)); assert (!err);

  /* wait for initial configuration command msg */

  while ( (runtimep->local_address == -1)
	  && (! runtimep->aborted) 
	  && (! runtimep->proto_negotiate) ) {
    globus_cond_wait (&(runtimep->cond), &(runtimep->mutex));
  }

  if ( runtimep->aborted ) { 
    err = runtimep->abort_reason;
    goto runtime_init_abort;
  }

  if ( runtimep->proto_negotiate ) {
    /* we can't currently handle any other protocol but the
     * one we just tried. */
    err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);

    err = GLOBUS_DUCT_ERROR_CHECKIN_PROTOCOL_MISMATCH;
    globus_duct_runtime_abort (runtimep, err);
    goto runtime_proto_abort;
  }

  err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);

  return GLOBUS_DUCT_SUCCESS;

 runtime_init_abort:
  globus_mutex_unlock (&(runtimep->mutex));

 runtime_proto_abort:
 runtime_init_checkin_failed:
  globus_duct_runtime_destroy (runtimep);

 runtime_init_sp_dest_error:

  return err;
}


static void 
s_config_group_msg_handler (globus_nexus_endpoint_t * endpointp,
			    globus_nexus_buffer_t   * bufferp,
			    globus_bool_t       ignored_is_non_threaded_handler)
{
  int              err;
  globus_duct_runtime_t * runtimep;
  int config_protocol_version;

  if ( ignored_is_non_threaded_handler == GLOBUS_TRUE ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_is_non_threaded_handler = GLOBUS_TRUE;
  }

  assert (endpointp!=NULL); assert (bufferp!=NULL);

  runtimep = ((globus_duct_runtime_t *) 
	      globus_nexus_endpoint_get_user_pointer (endpointp));
  assert (runtimep!=NULL);

  err = globus_mutex_lock (&(runtimep->mutex)); assert (!err);

  {
    int remote_count;
    int i;

    err = nxbuff_get_int (bufferp, &(config_protocol_version));
    assert (!err);

    /* we only support one protocol right now */
    assert (config_protocol_version 
	    == GLOBUS_DUCT_CONFIG_PROTOCOL_MAX_VERSION);

    err = nxbuff_get_int (bufferp, &(runtimep->data_protocol_version));
    assert (!err);

    err = nxbuff_get_int (bufferp, &(runtimep->local_address)); assert (!err);
    err = nxbuff_get_int (bufferp, &remote_count); assert (!err);

    for (i=0; i<remote_count; i++) {
      int                  remote_addr;
      globus_nexus_startpoint_t * remote_spp;

      remote_spp = globus_malloc (sizeof(globus_nexus_startpoint_t));
      assert (remote_spp!=NULL);
      
      err = nxbuff_get_int (bufferp, &remote_addr); assert (!err);
      err = nxbuff_get_startpoint (bufferp, remote_spp); assert (!err);
      assert (!err);
      
      err = globus_list_insert (&(runtimep->remote_addresses),
				(void *) (long) remote_addr);
      assert (!err);

      err = globus_hashtable_insert (&(runtimep->remote_data_spst),
				 (void *) (long) remote_addr,
				 (void *) remote_spp);
      assert (!err);
    }
  }

  err = globus_cond_broadcast (&(runtimep->cond)); assert (!err);

  
  err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);


  globus_nexus_buffer_destroy (bufferp);
}

static void 
s_abort_msg_handler (globus_nexus_endpoint_t * endpointp,
		     globus_nexus_buffer_t   * bufferp,
		     globus_bool_t       ignored_is_non_threaded_handler)
{
  int              err;
  globus_duct_runtime_t * runtimep;

  if ( ignored_is_non_threaded_handler == GLOBUS_TRUE ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_is_non_threaded_handler = GLOBUS_TRUE;
  }

  assert (endpointp!=NULL); assert (bufferp!=NULL);

  runtimep = ((globus_duct_runtime_t *) 
	      globus_nexus_endpoint_get_user_pointer (endpointp));
  assert (runtimep!=NULL);

  err = globus_mutex_lock (&(runtimep->mutex)); assert (!err);

  err = nxbuff_get_int (bufferp, &(runtimep->abort_reason)); assert (!err);
  runtimep->aborted = 1;

  err = globus_cond_broadcast (&(runtimep->cond)); assert (!err);

  
  err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);


  globus_nexus_buffer_destroy (bufferp);
}

static void 
s_proto_negotiate_handler (globus_nexus_endpoint_t * endpointp,
			   globus_nexus_buffer_t   * bufferp,
			   globus_bool_t       ignored_is_non_threaded_handler)
{
  int              err;
  globus_duct_runtime_t * runtimep;
  int min_config_protocol_version;
  int max_config_protocol_version;

  if ( ignored_is_non_threaded_handler == GLOBUS_TRUE ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_is_non_threaded_handler = GLOBUS_TRUE;
  }

  assert (endpointp!=NULL); assert (bufferp!=NULL);

  runtimep = ((globus_duct_runtime_t *) 
	      globus_nexus_endpoint_get_user_pointer (endpointp));
  assert (runtimep!=NULL);

  err = nxbuff_get_int (bufferp, &min_config_protocol_version);
  assert (!err);
  err = nxbuff_get_int (bufferp, &max_config_protocol_version);
  assert (!err);

  utils_fprintf (stderr, "proto negotiate min %d max %d\n",
		 min_config_protocol_version,
		 max_config_protocol_version);

  err = globus_mutex_lock (&(runtimep->mutex)); assert (!err);

  runtimep->proto_negotiate = 1;

  err = globus_cond_broadcast (&(runtimep->cond)); assert (!err);

  
  err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);


  globus_nexus_buffer_destroy (bufferp);
}

int 
globus_duct_runtime_structure (globus_duct_runtime_t  * runtimep,
			       int             * local_address,
			       int             * remote_count,
			       int            ** remote_addresses)
{
  int i;
  int err;
  globus_list_t * addrs_iter;

  if ( (runtimep == NULL)
       || (local_address == NULL)
       || (remote_count == NULL)
       || (remote_addresses == NULL) ) 
    return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;


  err = globus_mutex_lock (&(runtimep->mutex)); assert (!err);

  if ( runtimep->aborted ) {
    err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);
    return GLOBUS_DUCT_ERROR_CONTROL_ABORTED;
  }

  (*local_address) = runtimep->local_address;
  (*remote_count) = globus_list_size (runtimep->remote_addresses);

  if ( (*remote_count) > 0 ) {
    (*remote_addresses) = globus_malloc (sizeof(int) * (*remote_count));
    assert ((*remote_addresses)!=NULL);
  }
  else {
    (*remote_addresses) = NULL;
  }

  i = 0;
  addrs_iter = runtimep->remote_addresses;
  while (addrs_iter!=NULL) {
    (*remote_addresses)[i] = (int) (long) globus_list_first (addrs_iter);
    addrs_iter = globus_list_rest (addrs_iter);
    i++;
  }


  err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);


  return GLOBUS_DUCT_SUCCESS;
}


int 
globus_duct_runtime_send (globus_duct_runtime_t * runtimep,
			int              dst_addr,
			int              msg_size,
			globus_byte_t    * msg)
{
  int                  i;
  int                  err;
  int                  err2;
  globus_nexus_startpoint_t * spp;
  globus_nexus_buffer_t       buffer;
  int data_protocol;

  if ( (runtimep == NULL)
       || (msg == NULL)
       || (msg_size > GLOBUS_DUCT_MAX_MSG_LENGTH) )
    return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;


  err = globus_mutex_lock (&(runtimep->mutex)); assert (!err);


  if ( runtimep->aborted ) {
    err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);
    return GLOBUS_DUCT_ERROR_CONTROL_ABORTED;
  }

  spp = ((globus_nexus_startpoint_t *) 
	 globus_hashtable_lookup (&(runtimep->remote_data_spst),
			      (void *) (long) dst_addr));
  if (spp==NULL) {
    err = 1;
    goto runtime_send_runtime_sp_error;
  }

  /* we only support one protocol right now */
  data_protocol = runtimep->data_protocol_version;
  assert (data_protocol==GLOBUS_DUCT_DATA_PROTOCOL_MAX_VERSION);

  err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);


  err = globus_nexus_buffer_init (&(buffer), 0, 0); assert (!err);

  err = nxbuff_put_int (&buffer, msg_size); assert (!err);

  for (i=0; i<msg_size; i++) {
    err = nxbuff_put_byte (&buffer, msg[i]); assert (!err);
  }

  err = globus_nexus_send_rsr (&buffer, spp,
			DATA_MSG_ID,
			NEXUS_TRUE /* destroy buffer */,
			NEXUS_TRUE /* always safe */);
  assert (!err);

  globus_nexus_startpoint_flush (spp);

  return GLOBUS_DUCT_SUCCESS;


 runtime_send_runtime_sp_error:

  err2 = globus_mutex_unlock (&(runtimep->mutex)); assert (!err2);

  return err;
}


static void 
s_data_msg_handler (globus_nexus_endpoint_t * endpointp,
		    globus_nexus_buffer_t   * bufferp,
		    globus_bool_t       ignored_is_non_threaded_handler)
{
  int              err;
  globus_duct_runtime_t * runtimep;

  if ( ignored_is_non_threaded_handler == GLOBUS_TRUE ) {
    /* suppress 'unused' warning from ignored parameter */
    ignored_is_non_threaded_handler = GLOBUS_TRUE;
  }

  assert (endpointp!=NULL); assert (bufferp!=NULL);

  runtimep = (globus_duct_runtime_t *) globus_nexus_endpoint_get_user_pointer (endpointp);
  assert (runtimep!=NULL);


  err = globus_mutex_lock (&(runtimep->mutex)); assert (!err);


  if ( runtimep->data_callback != NULL ) {
    int i;
    int msg_size;
    globus_byte_t *msg;
    
    err = nxbuff_get_int (bufferp, &msg_size); assert (!err);
    assert (msg_size <= GLOBUS_DUCT_MAX_MSG_LENGTH);

    msg = globus_malloc (sizeof(globus_byte_t) * msg_size);
    assert (msg!=NULL);

    for (i=0; i<msg_size; i++) {
      err = nxbuff_get_byte (bufferp, &(msg[i])); assert (!err);
    }
    
    (runtimep->data_callback) (runtimep,
			       msg_size,
			       msg,
			       runtimep->data_callback_userdata);
  }
  else assert (0);


  err = globus_mutex_unlock (&(runtimep->mutex)); assert (!err);


  globus_nexus_buffer_destroy (bufferp);
}


void 
globus_duct_runtime_destroy (globus_duct_runtime_t * runtimep)
{
  int err;


  if ( runtimep->control_initialized ) {
    err = globus_nexus_startpoint_destroy (&(runtimep->control));
    assert (!err);
    runtimep->control_initialized = 0;
  }

  while ( runtimep->remote_addresses != NULL ) {
    globus_nexus_startpoint_t * spp;

    spp = ((globus_nexus_startpoint_t *) 
	   globus_hashtable_remove (&(runtimep->remote_data_spst),
			     globus_list_first (runtimep->remote_addresses)));
    assert (spp!=NULL);

    globus_list_remove (&(runtimep->remote_addresses),
		 runtimep->remote_addresses);
    err = globus_nexus_startpoint_destroy (spp); assert (!err);
    globus_free (spp);
  }

  globus_hashtable_destroy (&(runtimep->remote_data_spst));

  err = globus_nexus_startpoint_destroy (&(runtimep->data_port.sp));
  assert (!err);

  err = globus_nexus_startpoint_destroy (&(runtimep->config_port.sp));
  assert (!err);

#if 0
  /* when endpoint destruction is safe */
  err = globus_mutex_destroy (&(runtimep->mutex)); assert (!err);

  err = globus_cond_destroy (&(runtimep->cond)); assert (!err);

  err = globus_nexus_endpoint_destroy (&(runtimep->data_port.ep));
  assert (!err);

  err = globus_nexus_endpoint_destroy (&(runtimep->config_port.ep));
  assert (!err);

  err = globus_nexus_endpointattr_destroy (&(runtimep->data_port.epattr));
  assert (!err);

  err = globus_nexus_endpointattr_destroy (&(runtimep->config_port.epattr));
  assert (!err);

#endif

  return;
}


int 
globus_duct_runtime_make_startpoint (globus_nexus_startpoint_t * spp, 
				     const char         * contact)
{
  int err;

  if ( (spp==NULL)
       || (contact==NULL) ) return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;

  if ( (contact[0] == 'U')
       && (contact[1] == 'R')
       && (contact[2] == 'L') ) {
    /* an attachment URL */
    return globus_nexus_attach ((char *) (contact+3),
				spp);
  }
  else if ( (contact[0] == 'L')
	    && (contact[1] == 'S')
	    && (contact[2] == 'P') ) {
    /* a linearized startpoint */
    globus_byte_t   bbuff[GLOBUS_DUCT_MAX_MSG_LENGTH];
    globus_byte_t * ptr;
    int           format = 0;

    /* contact has form:
     *   >L S P hd hd ... hd<
     * "LSP" prefix identifies this as linearized startpoint
     * hex digit substring should be hex_decoded to obtain:
     *   >d d ... d \0 user-sp<
     *   >d d ... d< is the user-buffer format tag in decimal digits
     *   >user-sp< is the startpoint in user-buffer format
     */

    assert ( (s_strlen (contact+3) % 2) == 0 );

    _nx_hex_decode_byte_array (contact+3, 
			       s_strlen (contact+3) / 2,
			       bbuff);

    ptr = bbuff;
    err = globus_nexus_stdio_lock(); assert (!err);
    err = sscanf ((char *)ptr, "%d", &format); assert (err==1);
    err = globus_nexus_stdio_unlock(); assert (!err);
    while ( (*ptr)!='\0' ) ptr++; ptr++; /* d d ... d \0 user-sp */
    globus_nexus_user_get_startpoint (&(ptr), spp, 1, format);

    return GLOBUS_DUCT_SUCCESS;
  }
  else return GLOBUS_DUCT_ERROR_INVALID_PARAMETER;
}

void
globus_duct_runtime_abort (globus_duct_runtime_t * runtimep,
			   int reason)
{
  int err;

  err = globus_mutex_lock (&(runtimep->mutex));
  assert (!err);

  runtimep->aborted = 1;
  runtimep->abort_reason = reason;

  if ( runtimep->control_initialized ) {
    globus_nexus_buffer_t buffer;

    err = globus_nexus_buffer_init (&buffer, 0, 0); assert (!err);

    err = nxbuff_put_int (&buffer, reason); assert (!err);

    err = globus_nexus_send_rsr (&buffer, 
				 &(runtimep->control),
				 ABORT_REQ_ID,
				 NEXUS_TRUE /* destroy buffer */,
				 NEXUS_TRUE /* always safe */);

    globus_nexus_startpoint_flush (&(runtimep->control));
  }    
}

